"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const apigateway_generated_1 = require("./apigateway.generated");
/**
 * Configure the response received by clients, produced from the API Gateway backend.
 *
 * @resource AWS::ApiGateway::GatewayResponse
 */
class GatewayResponse extends core_1.Resource {
    constructor(scope, id, props) {
        super(scope, id);
        const resource = new apigateway_generated_1.CfnGatewayResponse(this, 'Resource', {
            restApiId: props.restApi.restApiId,
            responseType: props.type.responseType,
            responseParameters: this.buildResponseParameters(props.responseHeaders),
            responseTemplates: props.templates,
            statusCode: props.statusCode,
        });
        this.node.defaultChild = resource;
    }
    buildResponseParameters(responseHeaders) {
        if (!responseHeaders) {
            return undefined;
        }
        const responseParameters = {};
        for (const [header, value] of Object.entries(responseHeaders)) {
            responseParameters[`gatewayresponse.header.${header}`] = value;
        }
        return responseParameters;
    }
}
exports.GatewayResponse = GatewayResponse;
/**
 * Supported types of gateway responses.
 * @see https://docs.aws.amazon.com/apigateway/latest/developerguide/supported-gateway-response-types.html
 */
class ResponseType {
    constructor(type) {
        this.responseType = type;
    }
    /** A custom response type to suppport future cases. */
    static of(type) {
        return new ResponseType(type.toUpperCase());
    }
}
exports.ResponseType = ResponseType;
/**
 * The gateway response for authorization failure.
 */
ResponseType.ACCESS_DENIED = new ResponseType('ACCESS_DENIED');
/**
 * The gateway response for an invalid API configuration.
 */
ResponseType.API_CONFIGURATION_ERROR = new ResponseType('API_CONFIGURATION_ERROR');
/**
 * The gateway response when a custom or Amazon Cognito authorizer failed to authenticate the caller.
 */
ResponseType.AUTHORIZER_FAILURE = new ResponseType('AUTHORIZER_FAILURE');
/**
 * The gateway response for failing to connect to a custom or Amazon Cognito authorizer.
 */
ResponseType.AUTHORIZER_CONFIGURATION_ERROR = new ResponseType('AUTHORIZER_CONFIGURATION_ERROR');
/**
 * The gateway response when the request parameter cannot be validated according to an enabled request validator.
 */
ResponseType.BAD_REQUEST_PARAMETERS = new ResponseType('BAD_REQUEST_PARAMETERS');
/**
 * The gateway response when the request body cannot be validated according to an enabled request validator.
 */
ResponseType.BAD_REQUEST_BODY = new ResponseType('BAD_REQUEST_BODY');
/**
 * The default gateway response for an unspecified response type with the status code of 4XX.
 */
ResponseType.DEFAULT_4XX = new ResponseType('DEFAULT_4XX');
/**
 * The default gateway response for an unspecified response type with a status code of 5XX.
 */
ResponseType.DEFAULT_5XX = new ResponseType('DEFAULT_5XX');
/**
 * The gateway response for an AWS authentication token expired error.
 */
ResponseType.EXPIRED_TOKEN = new ResponseType('EXPIRED_TOKEN');
/**
 * The gateway response for an invalid AWS signature error.
 */
ResponseType.INVALID_SIGNATURE = new ResponseType('INVALID_SIGNATURE');
/**
 * The gateway response for an integration failed error.
 */
ResponseType.INTEGRATION_FAILURE = new ResponseType('INTEGRATION_FAILURE');
/**
 * The gateway response for an integration timed out error.
 */
ResponseType.INTEGRATION_TIMEOUT = new ResponseType('INTEGRATION_TIMEOUT');
/**
 * The gateway response for an invalid API key submitted for a method requiring an API key.
 */
ResponseType.INVALID_API_KEY = new ResponseType('INVALID_API_KEY');
/**
 * The gateway response for a missing authentication token error,
 * including the cases when the client attempts to invoke an unsupported API method or resource.
 */
ResponseType.MISSING_AUTHENTICATION_TOKEN = new ResponseType('MISSING_AUTHENTICATION_TOKEN');
/**
 * The gateway response for the usage plan quota exceeded error.
 */
ResponseType.QUOTA_EXCEEDED = new ResponseType('QUOTA_EXCEEDED');
/**
 * The gateway response for the request too large error.
 */
ResponseType.REQUEST_TOO_LARGE = new ResponseType('REQUEST_TOO_LARGE');
/**
 * The gateway response when API Gateway cannot find the specified resource
 * after an API request passes authentication and authorization.
 */
ResponseType.RESOURCE_NOT_FOUND = new ResponseType('RESOURCE_NOT_FOUND');
/**
 * The gateway response when usage plan, method, stage, or account level throttling limits exceeded.
 */
ResponseType.THROTTLED = new ResponseType('THROTTLED');
/**
 * The gateway response when the custom or Amazon Cognito authorizer failed to authenticate the caller.
 */
ResponseType.UNAUTHORIZED = new ResponseType('UNAUTHORIZED');
/**
 * The gateway response when a payload is of an unsupported media type, if strict passthrough behavior is enabled.
 */
ResponseType.UNSUPPORTED_MEDIA_TYPE = new ResponseType('UNSUPPORTED_MEDIA_TYPE');
/**
 * The gateway response when a request is blocked by AWS WAF.
 */
ResponseType.WAF_FILTERED = new ResponseType('WAF_FILTERED');
//# sourceMappingURL=data:application/json;base64,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