"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const iam = require("../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const amplify_generated_1 = require("./amplify.generated");
const branch_1 = require("./branch");
const domain_1 = require("./domain");
const utils_1 = require("./utils");
/**
 * An Amplify Console application
 */
class App extends core_1.Resource {
    constructor(scope, id, props) {
        var _a, _b, _c;
        super(scope, id);
        this.customRules = props.customRules || [];
        this.environmentVariables = props.environmentVariables || {};
        this.autoBranchEnvironmentVariables = props.autoBranchCreation && props.autoBranchCreation.environmentVariables || {};
        const role = props.role || new iam.Role(this, 'Role', {
            assumedBy: new iam.ServicePrincipal('amplify.amazonaws.com'),
        });
        this.grantPrincipal = role;
        const sourceCodeProviderOptions = (_a = props.sourceCodeProvider) === null || _a === void 0 ? void 0 : _a.bind(this);
        const app = new amplify_generated_1.CfnApp(this, 'Resource', {
            accessToken: (_b = sourceCodeProviderOptions === null || sourceCodeProviderOptions === void 0 ? void 0 : sourceCodeProviderOptions.accessToken) === null || _b === void 0 ? void 0 : _b.toString(),
            autoBranchCreationConfig: props.autoBranchCreation && {
                autoBranchCreationPatterns: props.autoBranchCreation.patterns,
                basicAuthConfig: props.autoBranchCreation.basicAuth && props.autoBranchCreation.basicAuth.bind(this, 'BranchBasicAuth'),
                buildSpec: props.autoBranchCreation.buildSpec && props.autoBranchCreation.buildSpec.toBuildSpec(),
                enableAutoBranchCreation: true,
                enableAutoBuild: props.autoBranchCreation.autoBuild === undefined ? true : props.autoBranchCreation.autoBuild,
                environmentVariables: core_1.Lazy.anyValue({ produce: () => utils_1.renderEnvironmentVariables(this.autoBranchEnvironmentVariables) }, { omitEmptyArray: true }),
                enablePullRequestPreview: props.autoBranchCreation.pullRequestPreview === undefined ? true : props.autoBranchCreation.pullRequestPreview,
                pullRequestEnvironmentName: props.autoBranchCreation.pullRequestEnvironmentName,
                stage: props.autoBranchCreation.stage,
            },
            basicAuthConfig: props.basicAuth && props.basicAuth.bind(this, 'AppBasicAuth'),
            buildSpec: props.buildSpec && props.buildSpec.toBuildSpec(),
            customRules: core_1.Lazy.anyValue({ produce: () => this.customRules }, { omitEmptyArray: true }),
            description: props.description,
            environmentVariables: core_1.Lazy.anyValue({ produce: () => utils_1.renderEnvironmentVariables(this.environmentVariables) }, { omitEmptyArray: true }),
            iamServiceRole: role.roleArn,
            name: props.appName || this.node.id,
            oauthToken: (_c = sourceCodeProviderOptions === null || sourceCodeProviderOptions === void 0 ? void 0 : sourceCodeProviderOptions.oauthToken) === null || _c === void 0 ? void 0 : _c.toString(),
            repository: sourceCodeProviderOptions === null || sourceCodeProviderOptions === void 0 ? void 0 : sourceCodeProviderOptions.repository,
        });
        this.appId = app.attrAppId;
        this.appName = app.attrAppName;
        this.arn = app.attrArn;
        this.defaultDomain = app.attrDefaultDomain;
    }
    /**
     * Import an existing application
     */
    static fromAppId(scope, id, appId) {
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.appId = appId;
            }
        }
        return new Import(scope, id);
    }
    /**
     * Adds a custom rewrite/redirect rule to this application
     */
    addCustomRule(rule) {
        this.customRules.push(rule);
        return this;
    }
    /**
     * Adds an environment variable to this application.
     *
     * All environment variables that you add are encrypted to prevent rogue
     * access so you can use them to store secret information.
     */
    addEnvironment(name, value) {
        this.environmentVariables[name] = value;
        return this;
    }
    /**
     * Adds an environment variable to the auto created branch.
     *
     * All environment variables that you add are encrypted to prevent rogue
     * access so you can use them to store secret information.
     */
    addAutoBranchEnvironment(name, value) {
        this.autoBranchEnvironmentVariables[name] = value;
        return this;
    }
    /**
     * Adds a branch to this application
     */
    addBranch(id, options = {}) {
        return new branch_1.Branch(this, id, {
            ...options,
            app: this,
        });
    }
    /**
     * Adds a domain to this application
     */
    addDomain(id, options = {}) {
        return new domain_1.Domain(this, id, {
            ...options,
            app: this,
        });
    }
}
exports.App = App;
/**
 * The status code for a URL rewrite or redirect rule.
 */
var RedirectStatus;
(function (RedirectStatus) {
    /**
     * Rewrite (200)
     */
    RedirectStatus["REWRITE"] = "200";
    /**
     * Permanent redirect (301)
     */
    RedirectStatus["PERMANENT_REDIRECT"] = "301";
    /**
     * Temporary redirect (302)
     */
    RedirectStatus["TEMPORARY_REDIRECT"] = "302";
    /**
     * Not found (404)
     */
    RedirectStatus["NOT_FOUND"] = "404";
})(RedirectStatus = exports.RedirectStatus || (exports.RedirectStatus = {}));
/**
 * Custom rewrite/redirect rule for an Amplify App.
 *
 * @see https://docs.aws.amazon.com/amplify/latest/userguide/redirects.html
 */
class CustomRule {
    constructor(options) {
        this.source = options.source;
        this.target = options.target;
        this.status = options.status;
        this.condition = options.condition;
    }
}
exports.CustomRule = CustomRule;
/**
 * Sets up a 200 rewrite for all paths to `index.html` except for path
 * containing a file extension.
 */
CustomRule.SINGLE_PAGE_APPLICATION_REDIRECT = new CustomRule({
    source: '</^[^.]+$/>',
    target: '/index.html',
    status: RedirectStatus.REWRITE,
});
//# sourceMappingURL=data:application/json;base64,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