"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const cfn = require("../../aws-cloudformation"); // Automatically re-written from '@aws-cdk/aws-cloudformation'
const cpactions = require("../../aws-codepipeline-actions"); // Automatically re-written from '@aws-cdk/aws-codepipeline-actions'
const cxschema = require("../../cloud-assembly-schema"); // Automatically re-written from '@aws-cdk/cloud-assembly-schema'
const cdk = require("../../core"); // Automatically re-written from '@aws-cdk/core'
/**
 * A class to deploy a stack that is part of a CDK App, using CodePipeline.
 * This composite Action takes care of preparing and executing a CloudFormation ChangeSet.
 *
 * It currently does *not* support stacks that make use of ``Asset``s, and
 * requires the deployed stack is in the same account and region where the
 * CodePipeline is hosted.
 */
class PipelineDeployStackAction {
    constructor(props) {
        var _a, _b;
        this.stack = props.stack;
        const assets = this.stack.node.metadata.filter(md => md.type === cxschema.ArtifactMetadataEntryType.ASSET);
        if (assets.length > 0) {
            // FIXME: Implement the necessary actions to publish assets
            throw new Error(`Cannot deploy the stack ${this.stack.stackName} because it references ${assets.length} asset(s)`);
        }
        const createChangeSetRunOrder = props.createChangeSetRunOrder || 1;
        const executeChangeSetRunOrder = props.executeChangeSetRunOrder || (createChangeSetRunOrder + 1);
        if (createChangeSetRunOrder >= executeChangeSetRunOrder) {
            throw new Error(`createChangeSetRunOrder (${createChangeSetRunOrder}) must be < executeChangeSetRunOrder (${executeChangeSetRunOrder})`);
        }
        const changeSetName = props.changeSetName || 'CDK-CodePipeline-ChangeSet';
        const capabilities = cfnCapabilities(props.adminPermissions, props.capabilities);
        this.prepareChangeSetAction = new cpactions.CloudFormationCreateReplaceChangeSetAction({
            actionName: (_a = props.createChangeSetActionName) !== null && _a !== void 0 ? _a : 'ChangeSet',
            changeSetName,
            runOrder: createChangeSetRunOrder,
            stackName: props.stack.stackName,
            templatePath: props.input.atPath(props.stack.templateFile),
            adminPermissions: props.adminPermissions,
            deploymentRole: props.role,
            capabilities,
        });
        this.executeChangeSetAction = new cpactions.CloudFormationExecuteChangeSetAction({
            actionName: (_b = props.executeChangeSetActionName) !== null && _b !== void 0 ? _b : 'Execute',
            changeSetName,
            runOrder: executeChangeSetRunOrder,
            stackName: this.stack.stackName,
        });
    }
    bind(scope, stage, options) {
        if (this.stack.environment !== cdk.Stack.of(scope).environment) {
            // FIXME: Add the necessary to extend to stacks in a different account
            throw new Error('Cross-environment deployment is not supported');
        }
        stage.addAction(this.prepareChangeSetAction);
        this._deploymentRole = this.prepareChangeSetAction.deploymentRole;
        return this.executeChangeSetAction.bind(scope, stage, options);
    }
    get deploymentRole() {
        if (!this._deploymentRole) {
            throw new Error('Use this action in a pipeline first before accessing \'deploymentRole\'');
        }
        return this._deploymentRole;
    }
    /**
     * Add policy statements to the role deploying the stack.
     *
     * This role is passed to CloudFormation and must have the IAM permissions
     * necessary to deploy the stack or you can grant this role `adminPermissions`
     * by using that option during creation. If you do not grant
     * `adminPermissions` you need to identify the proper statements to add to
     * this role based on the CloudFormation Resources in your stack.
     */
    addToDeploymentRolePolicy(statement) {
        this.deploymentRole.addToPolicy(statement);
    }
    onStateChange(name, target, options) {
        return this.executeChangeSetAction.onStateChange(name, target, options);
    }
    get actionProperties() {
        return this.executeChangeSetAction.actionProperties;
    }
}
exports.PipelineDeployStackAction = PipelineDeployStackAction;
function cfnCapabilities(adminPermissions, capabilities) {
    if (adminPermissions && capabilities === undefined) {
        // admin true default capability to NamedIAM and AutoExpand
        return [cfn.CloudFormationCapabilities.NAMED_IAM, cfn.CloudFormationCapabilities.AUTO_EXPAND];
    }
    else if (capabilities === undefined) {
        // else capabilities are undefined set AnonymousIAM and AutoExpand
        return [cfn.CloudFormationCapabilities.ANONYMOUS_IAM, cfn.CloudFormationCapabilities.AUTO_EXPAND];
    }
    else {
        // else capabilities are defined use them
        return capabilities;
    }
}
//# sourceMappingURL=data:application/json;base64,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