import os, json, shutil, subprocess, time, sys
import requests
from .unitypredictUtils import ReturnValues as Ret
from .unitypredictUtils import DirFileHandlers
from .UnityPredictLocalHost import UnityPredictLocalHost

class UnityPredictCli:
    def __init__(self, uptResponseTimeout: int = 20) -> None:

        self._uptCredFolder = ".unitypredict"
        self._uptCredFile = "credentials"
        self._userRoot = os.path.expanduser("~")
        self._uptCredDir = os.path.join(self._userRoot, self._uptCredFolder)
        self._uptCredPath = os.path.join(self._uptCredDir, self._uptCredFile)

        # self._uptEntryPointAPI = "https://api.dev.unitypredict.net/api/engines/supportedengines"
        self._uptEntryPointAPI = "https://api.prod.unitypredict.com/api/engines/supportedengines"
        self._uptEntryPointFieName = "EntryPoint.py"

        # API list
        self._uptDevAPI = "https://api.dev.unitypredict.net"
        self._uptProdAPI = "https://api.prod.unitypredict.com"
        self._uptEngineAPI = "/api/engines"

        self._uptBasePlatformName = "SL_CPU_BASE_PYTHON_3.12"
        self._uptLocalMainFileName = "main.py"
        self._uptRequirementFileName = "requirements.txt"

        # Config and its keys
        self._uptEngineConfig = {}
        self._uptAppEngineConfigKey = "LocalMockEngineConfig"
        self._uptDeploymentParametersConfigKey = "DeploymentParameters"
        self._uptEngineIdConfigKey = "UnityPredictEngineId"
        self._uptParentRepoIdConfigKey = "ParentRepositoryId"


        self._uptResponseTimeout = uptResponseTimeout


        # Util Driver
        self._dirFileHandlers = DirFileHandlers()
        
        # Get API key
        self._uptApiKeyHeaderPrefix = "APIKEY@"
        self._uptApiKeyDict = {}
        if os.path.exists(self._uptCredPath):
            with open(self._uptCredPath) as credFile:
                self._uptApiKeyDict = json.load(credFile)
            

    # Private API functions
    def _getEnvAPIUrl(self, uptEnv: str = "dev"):

        endpointPrefix = ""
        if uptEnv == "prod":
            endpointPrefix = self._uptProdAPI
        else:
            endpointPrefix = self._uptDevAPI

        return endpointPrefix
    
    def _createEngineOnUpt(self, engineName: str, requestPayload: dict, uptProfile: str = "default", uptEnv: str = "dev"):
        
        endpointPrefix = self._getEnvAPIUrl(uptEnv=uptEnv)
        endpoint = f"{endpointPrefix}{self._uptEngineAPI}"
        apiKey = self._uptApiKeyDict[uptProfile]["UPT_API_KEY"]
        endpointHeader = {"Authorization": f"Bearer {self._uptApiKeyHeaderPrefix}{apiKey}"}

        response = requests.post(endpoint, json=requestPayload, headers=endpointHeader, timeout=self._uptResponseTimeout)

        if response.status_code != 200:
            print (f"Unable to create EngineId for engine {engineName}: {response.text} [{response.status_code}]")
            return None


        try:

            respJson = response.json()
            createdEngineId = respJson["engineId"]

            return createdEngineId

        except Exception as e:
            print (f"Exception occured while creating EngineId: {e}")
            return None
        
    def _updateEngineOnUpt(self, engineName: str, requestPayload: dict, uptProfile: str = "default", uptEnv: str = "dev"):
        
        endpointPrefix = self._getEnvAPIUrl(uptEnv=uptEnv)
        endpoint = f"{endpointPrefix}{self._uptEngineAPI}"
        apiKey = self._uptApiKeyDict[uptProfile]["UPT_API_KEY"]
        endpointHeader = {"Authorization": f"Bearer {self._uptApiKeyHeaderPrefix}{apiKey}"}

        response = requests.post(endpoint, json=requestPayload, headers=endpointHeader, timeout=self._uptResponseTimeout)

        if response.status_code != 200:
            print (f"Unable to update Engine {engineName}: {response.text} [{response.status_code}]")
            return Ret.ERROR

        # print (f"Update engine {engineName} Success!")
        return Ret.SUCCESS

    def _searchEngineOnUpt(self, engineName: str, engineId: str, uptProfile: str = "default", uptEnv: str = "dev"):
        
        endpointPrefix = self._getEnvAPIUrl(uptEnv=uptEnv)
        searchEndpoint = f"{endpointPrefix}{self._uptEngineAPI}/{engineId}"
        apiKey = self._uptApiKeyDict[uptProfile]["UPT_API_KEY"]
        endpointHeader = {"Authorization": f"Bearer {self._uptApiKeyHeaderPrefix}{apiKey}"}

        response = requests.get(searchEndpoint, headers=endpointHeader, timeout=self._uptResponseTimeout)
        if response.status_code != 200:
            return (Ret.ERROR, response)
        
        return (Ret.SUCCESS, response)
    
    def _deployEngineOnUpt(self, engineName: str, engineId: str, uptProfile: str = "default", uptEnv: str = "dev", permitTimeDiff: int = 20):
        
        endpointPrefix = self._getEnvAPIUrl(uptEnv=uptEnv)
        deployEndpoint = f"{endpointPrefix}{self._uptEngineAPI}/{engineId}/deploy"
        apiKey = self._uptApiKeyDict[uptProfile]["UPT_API_KEY"]
        endpointHeader = {"Authorization": f"Bearer {self._uptApiKeyHeaderPrefix}{apiKey}"}

        response = requests.post(deployEndpoint, headers=endpointHeader, timeout=self._uptResponseTimeout)
        if response.status_code != 200:
            return Ret.ERROR
        
        # Check status of deployment on UnityPredict
        deployStatusEndpoint = f"{endpointPrefix}{self._uptEngineAPI}/{engineId}"

        startTime = time.time()
        permitTimeDiffSec = permitTimeDiff * 60 # 20 minutes

        deployReadyFlag: bool = False

        while (True):

            response = requests.get(deployStatusEndpoint, headers=endpointHeader, timeout=self._uptResponseTimeout)
            
            if response.status_code != 200:
                break
            
                
            try:
                respDict : dict = response.json()
                deployStat: str = respDict.get("status", "")
                
                if deployStat.casefold() == "ready":
                    deployReadyFlag = True
                    break

                elif deployStat.casefold() == "preparingimage" or \
                        deployStat.casefold() == "preparingtoupdate" or \
                        deployStat.casefold() == "deployingimage":
                    
                    endTime = time.time()
                    print(f"\rDeploying Engine {engineName} | Status: {deployStat} | Time elapsed : {endTime - startTime} Seconds | ...                      \t\t\t\t\t\t\r", end="")

                    if ((endTime - startTime) >= permitTimeDiffSec):
                        print (f"\nExceeded configured time of {permitTimeDiff} Minutes! Terminating deploy process!")
                        break

                    time.sleep(5)

                    continue

                else:
                    print (f"\nInvalid deploy status received: {deployStat}")
                    break

            except Exception as e:
                pass
            
            time.sleep(1)
        
        if deployReadyFlag == True:
            print ("\n")
            return Ret.SUCCESS
        
        
        print (f"\nEngine {engineName} deployment Error!")
        return Ret.ERROR
        
    
    def _deleteEngineOnUpt(self, engineName: str, engineId: str, uptProfile: str = "default", uptEnv: str = "dev"):
        
        endpointPrefix = self._getEnvAPIUrl(uptEnv=uptEnv)
        searchEndpoint = f"{endpointPrefix}{self._uptEngineAPI}/{engineId}"
        apiKey = self._uptApiKeyDict[uptProfile]["UPT_API_KEY"]
        endpointHeader = {"Authorization": f"Bearer {self._uptApiKeyHeaderPrefix}{apiKey}"}

        response = requests.delete(searchEndpoint, headers=endpointHeader, timeout=self._uptResponseTimeout)
        if response.status_code != 200:
            return Ret.ERROR
        
        return Ret.SUCCESS
    
    def _getFileUploadUrl(self, engineName: str, engineId: str, fileName: str, fileType: str, uptProfile: str = "default", uptEnv: str = "dev"):

        endpointPrefix = self._getEnvAPIUrl(uptEnv=uptEnv)
        fileToUploadUrl = f"{endpointPrefix}{self._uptEngineAPI}/upload/{engineId}/{fileType}?fileName={fileName}"
        apiKey = self._uptApiKeyDict[uptProfile]["UPT_API_KEY"]
        endpointHeader = {"Authorization": f"Bearer {self._uptApiKeyHeaderPrefix}{apiKey}"}

        response = requests.get(fileToUploadUrl, headers=endpointHeader, timeout=self._uptResponseTimeout)
        
        if response.status_code != 200:
            print (f"unable to fetch upload url for the file: {fileName}: {response.status_code}")
            return None
        
        
        try:

            respJson = response.json()
            publicUploadUrl = respJson["publicUploadLink"]

            # print (f"Url fetch for {fileName} of type {fileType} Success!")

            return publicUploadUrl

        except Exception as e:
            print (f"Exception occured while creating EngineId: {e}")
            return None

    def _uploadFileToUploadUrl(self, uploadUrl: str, fileName: str, absFilePath: str, fileContentType: str = "application/octet-stream", uptProfile: str = "default", uptEnv: str = "dev"):

        apiKey = self._uptApiKeyDict[uptProfile]["UPT_API_KEY"]
        endpointHeader = {"Authorization": f"Bearer {self._uptApiKeyHeaderPrefix}{apiKey}"}

        with open(absFilePath, 'rb') as file:
            fileContent = file.read()
            # files = {'file': (fileName, file.read(), fileContentType)}
            response = requests.put(uploadUrl, data=fileContent, timeout=self._uptResponseTimeout)

            if response.status_code != 200:
                print(f"Error in uploading file {fileName}: {response.text} [{response.status_code}]")
                return Ret.ERROR
            
        # print (f"Upload file {fileName} Success!")
        return Ret.SUCCESS


    # Private Unitl Functions

    def _getEngineConfigDetails(self, engineName: str):
        
        currPath = os.getcwd()
        enginePath = os.path.join(currPath, engineName)
        configName = "config.json"
        engineConfigPath = os.path.join(enginePath, configName)

        configContent = self._dirFileHandlers.getFileContent(engineConfigPath)
        if configContent == None:
            return Ret.ERROR
        
        try:
            self._uptEngineConfig = json.loads(configContent)
            return Ret.SUCCESS
        except Exception as e:
            return Ret.ERROR
        
    def _setEngineConfigDetails(self, engineName: str, configContent: str):
        
        currPath = os.getcwd()
        enginePath = os.path.join(currPath, engineName)
        configName = "config.json"
        engineConfigPath = os.path.join(enginePath, configName)

        try:
            self._dirFileHandlers.writeFileContent(engineConfigPath, configContent)
            return Ret.SUCCESS
        except Exception as e:
            return Ret.ERROR
        
    def _uploadLocalFiles(self, engineName: str, engineId: str, uptProfile: str = "default", uptEnv: str = "dev"):
        
        currPath = os.getcwd()
        enginePath = os.path.join(currPath, engineName)

        # Upload Requirements File

        reqFilePath = os.path.join(enginePath, self._uptRequirementFileName)
        reqFileName : str = None

        fileUploadUrl: str = None

        if os.path.exists(reqFilePath):
            
            fileUploadUrl = self._getFileUploadUrl(engineName=engineName, engineId=engineId, 
                                   fileName=self._uptRequirementFileName, fileType="PackageDefinitionFile", uptEnv=uptEnv)
            
            if fileUploadUrl != None:

                ret = self._uploadFileToUploadUrl(uploadUrl=fileUploadUrl, fileName=self._uptRequirementFileName,
                                            absFilePath=reqFilePath)
                
                if ret == Ret.ERROR:
                    reqFileName = None
                else:
                    reqFileName = self._uptRequirementFileName

        # Upload source files
        fileExt: str = ".py"
        mainFile: str = "main.py"

        fileUploadList: list = []

        for items in os.listdir(enginePath):
            
            filePath = os.path.join(enginePath, items)

            if (items == mainFile) or (fileExt not in items) or (not os.path.isfile(filePath)):
                continue

            fileUploadUrl = self._getFileUploadUrl(engineName=engineName, engineId=engineId, 
                                   fileName=items, fileType="SourceFile", uptEnv=uptEnv)
            
            if fileUploadUrl == None:
                continue

            ret = self._uploadFileToUploadUrl(uploadUrl=fileUploadUrl, fileName=items,
                                            absFilePath=filePath)
            
            if ret == Ret.ERROR:
                continue
            
            fileDescription =   {
                                    "fileName": f"{items}",
                                    "fileType": "SourceFile"
                                }
            
            fileUploadList.append(fileDescription)
        
        
        # Upload model files
        modelFolder = self._uptEngineConfig.get(self._uptAppEngineConfigKey, {}).get("ModelsDirPath", "")

        if not os.path.exists(modelFolder):
            return (fileUploadList, reqFileName)
        
        for items in os.listdir(modelFolder):
            
            filePath = os.path.join(modelFolder, items)

            fileUploadUrl = self._getFileUploadUrl(engineName=engineName, engineId=engineId, 
                                   fileName=items, fileType="ModelFiles", uptEnv=uptEnv)
            
            if fileUploadUrl == None:
                continue

            ret = self._uploadFileToUploadUrl(uploadUrl=fileUploadUrl, fileName=items,
                                            absFilePath=filePath)
            
            if ret == Ret.ERROR:
                continue
            
            fileDescription =   {
                                    "fileName": f"{items}",
                                    "fileType": "ModelFiles"
                                }
            
            fileUploadList.append(fileDescription)

        return (fileUploadList, reqFileName)



    def _fetchEntryPointTemplate(self, apiKey : str, uptProfile: str = "default"):

        # print ("Fetching EntryPoint.py template ...")
        headers = {"Authorization": f"Bearer {apiKey}"}
        response = requests.get(self._uptEntryPointAPI, headers=headers, timeout=self._uptResponseTimeout)
        if response.status_code != 200:
            print (f"EntryPoint.py template fetch Failure with error code: {response.status_code}")
            print(response.text)
            return Ret.ENGINE_CREATE_ERROR
        
        try:
            suppPlatforms = response.json()
            for suppPlatform in suppPlatforms:
                if suppPlatform["platformKey"] != self._uptBasePlatformName:
                    continue

                entrypointResp = requests.get(suppPlatform["entryPointTemplateUrl"], timeout=self._uptResponseTimeout)
                if entrypointResp.status_code != 200:
                    print (f"EntryPoint.py template fetch Failure with error code: {entrypointResp.status_code}")
                    print(entrypointResp.text)
                    return Ret.ENGINE_CREATE_ERROR
                

                entrypointContent = entrypointResp.content.decode(entrypointResp.encoding)
                entrypointContent = entrypointContent.replace("\r", "")
                entryPointFile = os.path.join(os.getcwd(), self._uptEntryPointFieName)

                with open(entryPointFile, "w+", encoding="utf-8")as efile:
                    efile.write(entrypointContent)

                break

        except Exception as e:
            print (f"Exception Occured while fetching EntryPoint: {e}")
            return Ret.ENGINE_CREATE_ERROR
        
        # print ("EntryPoint.py template fetch Success!")
        
        return Ret.ENGINE_CREATE_SUCCESS
    
    def _generateMainFile(self):

        mainFileContent = r"""
from unitypredict_engines import UnityPredictLocalHost
from unitypredict_engines import ChainedInferenceRequest, ChainedInferenceResponse, FileReceivedObj, FileTransmissionObj, IPlatform, InferenceRequest, InferenceResponse, OutcomeValue
import EntryPoint

if __name__ == "__main__":

    
    platform = UnityPredictLocalHost()

    testRequest = InferenceRequest()
    # User defined Input Values
    testRequest.InputValues = {} 
    results: InferenceResponse = EntryPoint.run_engine(testRequest, platform)

    # Print Outputs
    if (results.Outcomes != None):
        for outcomKeys, outcomeValues in results.Outcomes.items():
            print ("\n\nOutcome Key: {}".format(outcomKeys))
            for values in outcomeValues:
                infVal: OutcomeValue = values
                print ("Outcome Value: \n{}\n\n".format(infVal.Value))
                print ("Outcome Probability: \n{}\n\n".format(infVal.Probability))
    
    # Print Error Messages (if any)
    print ("Error Messages: {}".format(results.ErrorMessages))

        """

        try:
            MAIN_FILE_PATH = os.path.join(os.getcwd(), self._uptLocalMainFileName)
            with open(MAIN_FILE_PATH, "w+") as mainf:
                mainf.write(mainFileContent)
        except Exception as e:
            print (f"Exception occured while generating main file: {e}")
            return Ret.ENGINE_CREATE_ERROR
        
        # print ("Main file generation Success!")
        return Ret.ENGINE_CREATE_SUCCESS
    
    def _createRequirementsFile(self):
        
        try:
            REQ_FILE_PATH = os.path.join(os.getcwd(), self._uptRequirementFileName)
            with open(REQ_FILE_PATH, "w+") as mainf:
                pass
        except Exception as e:
            print (f"Exception occured while generating requirements file: {e}")
            return Ret.ENGINE_CREATE_ERROR
        
        # print ("Requirements file generation Success!")
        return Ret.ENGINE_CREATE_SUCCESS


    
    def _createEngineId(self, engineName: str, uptProfile: str = "default", uptEnv: str = "dev"):
    
        parentRepositoryId = self._uptEngineConfig.get(self._uptDeploymentParametersConfigKey, {}).get(self._uptParentRepoIdConfigKey, "")
        if parentRepositoryId == "":
            print ("No Parent Repository ID provided! Cannot create EngineId without the Parent Repository!")
            print ("Please configure the ParentRepositoryId in the config.json")
            return (None, Ret.UPT_ENGINE_CREATE_ERROR)

        # Create payload to fetch the created engineId
        requestPayload = {
            "engineName":f"{engineName}",
            "parentRepositoryId":f"{parentRepositoryId}",
            "engineDescription":"",
            "engineId":None
        }

        createdEngineId = self._createEngineOnUpt(engineName=engineName, requestPayload=requestPayload, uptEnv=uptEnv)
        
        if createdEngineId == None:
            return (None, Ret.UPT_ENGINE_CREATE_ERROR)
        
        return (createdEngineId, Ret.UPT_ENGINE_CREATE_SUCCESS)

    def _fetchOrCreateEngineId(self, engineName: str, uptProfile: str = "default", uptEnv: str = "dev"):
        
        try:
            engineId = self._uptEngineConfig.get(self._uptDeploymentParametersConfigKey, {}).get(self._uptEngineIdConfigKey, "")
            if engineId != "":

                searchStat, response = self._searchEngineOnUpt(engineName=engineName, engineId=engineId, uptEnv=uptEnv)

                if searchStat == Ret.ERROR:
                    print (f"Invalid engine id detected! Creating new Engine {engineName} and assigning new EngineId")
                    return self._createEngineId(engineName=engineName, uptEnv=uptEnv)
                    
                engineDetails: dict = response.json()
                return (engineDetails.get("engineId", None), Ret.UPT_ENGINE_UPDATE_SUCCESS)

            else:
                # Create engine id
                print (f"No engine id detected! Creating new Engine {engineName} and assigning new EngineId")
                return self._createEngineId(engineName=engineName, uptEnv=uptEnv)
        except Exception as e:
            print (f"Exception ocured while fetching EngineId: {e}")
            return (None, Ret.UPT_ENGINE_UPDATE_ERROR)
        
    def _updateBaseConfig(self, engineName: str, engineId: str, uptProfile: str = "default", uptEnv: str = "dev"):

        try:
            self._uptEngineConfig[self._uptDeploymentParametersConfigKey][self._uptEngineIdConfigKey] = engineId

            configContent = json.dumps(self._uptEngineConfig, indent=4)

            ret = self._setEngineConfigDetails(engineName=engineName, configContent=configContent)
            if ret == Ret.SUCCESS:
                return Ret.UPT_ENGINE_UPDATE_SUCCESS
            print ("Error occured while updating config file")
            return Ret.UPT_ENGINE_UPDATE_ERROR
        except Exception as e:
            print (f"Exception ocured while updating local config with EngineId: {e}")
            return Ret.UPT_ENGINE_UPDATE_ERROR





    # Binding functions with CLI Flags
    def configureCredentials(self, uptApiKey: str| None, uptProfile: str = "default"):

        if not os.path.exists(self._uptCredDir):

            os.mkdir(self._uptCredDir)
            
        self._uptApiKeyDict[uptProfile] = {
                "UPT_API_KEY": uptApiKey
            }
        
        try:
            with open(self._uptCredPath, "w+") as credFile:
                credFile.write(json.dumps(self._uptApiKeyDict, indent=4))
        except Exception as e:
            print (f"Error in creating file {self._uptCredPath}: {e}")
            return Ret.CRED_CREATE_ERROR
        
        return Ret.CRED_CREATE_SUCCESS
    
    def showProfiles(self):
        
        print ("Credential Profiles: ")
        for keys in self._uptApiKeyDict.keys():
            print(f"{keys}")

    def findEngine(self, engineName: str, uptProfile: str = "default"):
        currPath = os.getcwd()
        enginePath = os.path.join(currPath, engineName)

        if os.path.exists(enginePath):
            return Ret.ENGINE_FOUND
        else:
            return Ret.ENGINE_NOT_FOUND

    
    def createEngine(self, engineName: str, uptProfile: str = "default"):
        
        ret = self.findEngine(engineName=engineName)

        if ret == Ret.ENGINE_FOUND:
            print ("""The engine already exists on the current directory. You can:
                - Change the directory
                - Use [--create] flag to change the name of the engine
                """)
            return ret

        
        currPath = os.getcwd()
        enginePath = os.path.join(currPath, engineName)
        
        # Make Engine Path
        os.mkdir(enginePath)
        
        # Change dir to enginePath
        os.chdir(enginePath)

        
        apiKey = self._uptApiKeyDict[uptProfile]["UPT_API_KEY"]
        # print ("Creating Engine Components ...")
        initEngine = UnityPredictLocalHost(apiKey=apiKey, defaultPlatform=self._uptBasePlatformName)
        
        if not initEngine.isConfigInitialized():
            # print ("Engine Components creation Failed!")
            # Change dir back to parent of enginePath    
            os.chdir(currPath)
            return Ret.ENGINE_CREATE_ERROR

        # print ("Engine Components creation Success!")
        
        # Fetch the entrypoint details
        ret = self._fetchEntryPointTemplate(apiKey=apiKey)
        if ret == Ret.ENGINE_CREATE_ERROR:
            # Change dir back to parent of enginePath    
            os.chdir(currPath)
            return ret
        
        # Generate main file
        ret = self._generateMainFile()
        if ret == Ret.ENGINE_CREATE_ERROR:
            # Change dir back to parent of enginePath    
            os.chdir(currPath)
            return ret
        
        # Generate requirements file
        ret = self._createRequirementsFile()
        if ret == Ret.ENGINE_CREATE_ERROR:
            # Change dir back to parent of enginePath    
            os.chdir(currPath)
            return ret
        
        
        
        # Change dir back to parent of enginePath    
        os.chdir(currPath)    
        return Ret.ENGINE_CREATE_SUCCESS
    
    def removeEngine(self, engineName: str, uptProfile: str = "default"):

        ret = self.findEngine(engineName=engineName)

        if ret == Ret.ENGINE_FOUND:
            enginePath = os.path.join(os.getcwd(), engineName)
            shutil.rmtree(enginePath)
            return Ret.ENGINE_REMOVE_SUCCESS
        else:
            return Ret.ENGINE_REMOVE_ERROR
    
    def runEngine(self, engineName: str, uptProfile: str = "default"):

        ret = self.findEngine(engineName=engineName)

        if ret == Ret.ENGINE_FOUND:
            currentPath = os.getcwd()
            enginePath = os.path.join(currentPath, engineName)
            # Change dir to enginePath
            os.chdir(enginePath)
            mainFile = os.path.join(enginePath, self._uptLocalMainFileName)
            subprocess.run(["python", mainFile])
            os.chdir(currentPath)
        else:
            print (f"Engine {engineName} not found. Could not run engine!")

    def deployEngine(self, engineName: str, uptProfile: str = "default", uptEnv: str = "dev"):

        ret = self.findEngine(engineName=engineName)
        if ret != Ret.ENGINE_FOUND:
            print (f"Requested Engine {engineName} not foeund!")
            return Ret.ENGINE_DEPLOY_ERROR
        
        # Fetch config details
        ret = self._getEngineConfigDetails(engineName=engineName)

        if ret == Ret.ERROR:
            return Ret.ENGINE_DEPLOY_ERROR
        
        engineId, retStatus = self._fetchOrCreateEngineId(engineName=engineName, uptProfile=uptProfile, uptEnv=uptEnv)

        if (((retStatus != Ret.UPT_ENGINE_CREATE_SUCCESS) and (retStatus != Ret.UPT_ENGINE_UPDATE_SUCCESS)) or (engineId == None)):
            print (f"Error in fetching EngineId on the UPT Repository. EngineId: {engineId}")
            return Ret.ENGINE_DEPLOY_ERROR
        
        print (f"Fetched EngineId from UPT: {engineId}")

        if (retStatus == Ret.UPT_ENGINE_CREATE_SUCCESS):
            # New engine created, update the local config with the fetched engineId
            print("Updating local config with the newly created EngineId")
            ret = self._updateBaseConfig(engineName=engineName, engineId=engineId, uptProfile=uptProfile, uptEnv=uptEnv)
            if ret == Ret.UPT_ENGINE_UPDATE_ERROR:
                print ("Error occured in updating the engineID to local config")
                return Ret.ENGINE_DEPLOY_ERROR
            
        ret, response = self._searchEngineOnUpt(engineName=engineName, engineId=engineId, uptEnv=uptEnv)

        if ret == Ret.ERROR:
            print (f"Error occured in detecting engine: {engineName} on UnityPredict Repository")
            return Ret.ENGINE_DEPLOY_ERROR
        
        
        respDict: dict = {}
        try:
            respDict = response.json()
            # print (f"Response after finding engine: {json.dumps(respDict, indent=4)}")
        except Exception as e:
            print (f"Response after finding engine: {response.text}")

        fileList, reqFileName = self._uploadLocalFiles(engineName=engineName, engineId=engineId, uptProfile=uptProfile, uptEnv=uptEnv)
        
        respDict["files"] = fileList
        respDict["packageDefinitionFileName"] = reqFileName
        respDict["engineDescription"] = self._uptEngineConfig.get(self._uptDeploymentParametersConfigKey, {}).get("EngineDescription", "")
        respDict["enginePlatformKey"] = self._uptEngineConfig.get(self._uptDeploymentParametersConfigKey, {}).get("EnginePlatform", "")
        respDict["memoryRequirement"] = self._uptEngineConfig.get(self._uptDeploymentParametersConfigKey, {}).get("Memory", "")
        respDict["storageRequirement"] = self._uptEngineConfig.get(self._uptDeploymentParametersConfigKey, {}).get("Storage", "")
        # print (f"Response after uploading files to engine: {json.dumps(respDict, indent=4)}")

        ret = self._updateEngineOnUpt(engineName=engineName, requestPayload=respDict, uptProfile=uptProfile, uptEnv=uptEnv)

        if ret == Ret.ERROR:
            print (f"Error in updating engine {engineName} on UnityPredict")
            return Ret.ENGINE_DEPLOY_ERROR
        

        print (f"Proceeding to deploy Engine {engineName}...")
        ret = self._deployEngineOnUpt(engineName=engineName, engineId=engineId, uptProfile=uptProfile, uptEnv=uptEnv)

        if ret == Ret.ERROR:
            print (f"Error in deploying engine {engineName} on UnityPredict")
            return Ret.ENGINE_DEPLOY_ERROR

        
        
        # print (f"Update of Engine {engineName} Success!")
        
        return Ret.ENGINE_DEPLOY_SUCCESS
    
    def deleteDeployedEngine(self, engineName: str, uptProfile: str = "default", uptEnv: str = "dev"):
        
        ret = self.findEngine(engineName=engineName)
        if ret != Ret.ENGINE_FOUND:
            print (f"Requested Engine {engineName} not foeund!")
            return Ret.UPT_ENGINE_DELETE_ERROR
        
        # Fetch config details
        ret = self._getEngineConfigDetails(engineName=engineName)

        if ret == Ret.ERROR:
            return Ret.UPT_ENGINE_DELETE_ERROR
        
        engineId = self._uptEngineConfig.get(self._uptDeploymentParametersConfigKey, {}).get(self._uptEngineIdConfigKey, "")
        
        if engineId == "":
            return Ret.UPT_ENGINE_DELETE_ERROR
        
        searchStat, response = self._searchEngineOnUpt(engineName=engineName, engineId=engineId, uptEnv=uptEnv)

        if searchStat == Ret.ERROR:
            print (f"Error occured in detecting engine {engineName} on UnityPredict Repository")
            return Ret.UPT_ENGINE_DELETE_ERROR
        
        ret = self._deleteEngineOnUpt(engineName=engineName, engineId=engineId, uptProfile=uptProfile, uptEnv=uptEnv)

        if ret == Ret.ERROR:
            print (f"Error occured in deleting engine {engineName} on UnityPredict Repository")
            return Ret.UPT_ENGINE_DELETE_ERROR
        
        searchStat, response = self._searchEngineOnUpt(engineName=engineName, engineId=engineId, uptEnv=uptEnv)
        
        if searchStat == Ret.ERROR:
            # print (f"Delete engine {engineName} [{engineId}] on UnityPredict Repository Success!")
            return Ret.UPT_ENGINE_DELETE_SUCCESS
        
        return Ret.UPT_ENGINE_DELETE_ERROR
        





        
