// Copyright (c) Mito
import React, { useState } from 'react';
import '../../../css/elements/MultiToggleBox.css';
import { classNames } from '../../utils/classNames';
import { fuzzyMatch } from '../../utils/strings';
import Row from '../spacing/Row';
import Input from './Input';
import LoadingDots from './LoadingDots';
/*
    So that we don't crash the users browser, we show at most
    10k items to them at once
*/
const MAX_DISPLAYED = 10000;
/*
  A box that contains a variety of options that can be toggled on and off indivigually.
  
  If optional toggleAllOptions are passed, then a Toggle All button is also displayed
  that toggles all the buttons at once.
*/
const MultiToggleBox = (props) => {
    // We can store state in this element, or in the parent element if we want
    const [_searchString, _setSearchString] = useState('');
    const searchString = props.searchState !== undefined ? props.searchState.searchString : _searchString;
    const setSearchString = props.searchState !== undefined ? props.searchState.setSearchString : _setSearchString;
    const height = props.height || 'block';
    const width = props.width || 'block';
    const heightClass = `element-height-${height}`;
    const widthClass = `element-width-${width}`;
    let displayedAllToggled = true;
    const displayedIndexes = [];
    let numDisplayed = 0;
    let maxDisplayed = false;
    // Only display the options that we're searching for, and also collect
    // information about how many children are passed and displayed
    const childrenToDisplay = React.Children.map((props.children), (child) => {
        const title = child.props.title;
        // Don't display if it doesn't match
        if (title === null || title === undefined || fuzzyMatch(title + '', searchString) < .8) {
            return null;
        }
        // Don't display if we've displayed enough already
        if (numDisplayed > MAX_DISPLAYED) {
            maxDisplayed = true;
            return null;
        }
        numDisplayed++;
        displayedAllToggled = displayedAllToggled && child.props.toggled;
        displayedIndexes.push(child.props.index);
        return child;
    });
    const { toggleAllIndexes } = props;
    return (React.createElement("div", { className: classNames('mutli-toggle-box-container', heightClass, widthClass, props.className) },
        props.searchable &&
            React.createElement(Input, { value: searchString, onChange: (e) => {
                    setSearchString(e.target.value);
                }, placeholder: 'Search', width: 'block', className: 'mb-2px' }),
        React.createElement("div", { className: classNames('multi-toggle-box'), 
            // It's hard to get the box to fill the rest of the container,
            // so we do a calculation if the search box is displayed
            style: { height: props.searchable ? 'calc(100% - 30px)' : '100%' } },
            props.loading === true &&
                React.createElement(Row, { justify: 'center' },
                    React.createElement("p", { className: 'text-body-1 text-align-center' },
                        "Loading items",
                        React.createElement(LoadingDots, null))),
            !props.loading && (maxDisplayed || props.isFiltered) &&
                React.createElement(Row, { justify: 'center' },
                    React.createElement("p", { className: 'text-body-1 text-align-center' }, "There are too many items to display. Search to filter down to the items you care about.")),
            !props.loading && numDisplayed === 0 &&
                React.createElement(Row, { justify: 'center' },
                    React.createElement("p", { className: 'text-body-1' }, "No items to display.")),
            toggleAllIndexes !== undefined && numDisplayed > 0 &&
                React.createElement("div", { key: 'Toggle All', className: classNames('multi-toggle-box-row', { 'multi-toggle-box-row-selected': displayedAllToggled }), onClick: () => {
                        toggleAllIndexes(displayedIndexes, !displayedAllToggled);
                    } },
                    React.createElement("input", { key: 'Toggle All', type: "checkbox", name: 'Toggle All', checked: displayedAllToggled }),
                    "Toggle All ",
                    searchString !== '' && " Matching"),
            childrenToDisplay)));
};
export default MultiToggleBox;
//# sourceMappingURL=MultiToggleBox.js.map