# ------------------------------------------------------------------------------------------------ #
# MIT License                                                                                      #
#                                                                                                  #
# Copyright (c) 2020, Microsoft Corporation                                                        #
#                                                                                                  #
# Permission is hereby granted, free of charge, to any person obtaining a copy of this software    #
# and associated documentation files (the "Software"), to deal in the Software without             #
# restriction, including without limitation the rights to use, copy, modify, merge, publish,       #
# distribute, sublicense, and/or sell copies of the Software, and to permit persons to whom the    #
# Software is furnished to do so, subject to the following conditions:                             #
#                                                                                                  #
# The above copyright notice and this permission notice shall be included in all copies or         #
# substantial portions of the Software.                                                            #
#                                                                                                  #
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING    #
# BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND       #
# NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM,     #
# DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,   #
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.          #
# ------------------------------------------------------------------------------------------------ #

import warnings

import jax.numpy as jnp

from .._core.policy import Policy
from ..policy_regularizers import PolicyRegularizer


class PolicyObjective:
    r"""

    Abstract base class for policy objectives. To see a concrete example, have a look at
    :class:`coax.policy_objectives.VanillaPG`.

    Parameters
    ----------
    pi : Policy

        The parametrized policy :math:`\pi_\theta(a|s)`.

    regularizer : PolicyRegularizer, optional

        A policy regularizer, see :mod:`coax.policy_regularizers`.

    """
    REQUIRES_PROPENSITIES = None

    def __init__(self, pi, regularizer):
        if not isinstance(pi, Policy):
            raise TypeError(f"pi must be a Policy, got: {type(pi)}")
        if not isinstance(regularizer, (PolicyRegularizer, type(None))):
            raise TypeError(
                "regularizer must be a PolicyRegularizer, "
                f"got: {type(regularizer)}")
        self.pi = pi
        self.regularizer = regularizer

    @property
    def hyperparams(self):
        r""" Hyperparameters specific to this policy objective. """
        return getattr(self.regularizer, 'hyperparams', {})

    def update(self, transition_batch, Adv):
        r"""

        Update the model parameters (weights) of the underlying function approximator.

        Parameters
        ----------
        transition_batch : TransitionBatch

            A batch of transitions.

        Adv : ndarray

            A batch of advantages :math:`\mathcal{A}(s,a)=q(s,a)-v(s)`.

        Returns
        -------
        metrics : dict of scalar ndarrays

            The structure of the metrics dict is ``{name: score}``.

        """
        grads, state, metrics = self.grads_and_metrics(transition_batch, Adv)
        self.update_from_grads(grads, state)
        return metrics

    def update_from_grads(self, grads, state):
        r"""

        Update the model parameters (weights) of the underlying function approximator given
        pre-computed gradients.

        This method is useful in situations in which computation of the gradients is deligated to a
        separate (remote) process.

        Parameters
        ----------
        grads : pytree with ndarray leaves

            A batch of gradients, generated by the :attr:`grads` method.

        state : pytree

            The internal state of the forward-pass function. See :attr:`Policy.function_state
            <coax.Policy.function_state>` and :func:`haiku.transform_with_state` for more details.

        """
        self.pi.func_approx.update_params(**grads)
        self.pi.func_approx.update_function_state(**state)

    def grads_and_metrics(self, transition_batch, Adv):
        r"""

        Compute the gradients associated with a batch of transitions with
        corresponding advantages.

        Parameters
        ----------
        transition_batch : TransitionBatch

            A batch of transitions.

        Adv : ndarray

            A batch of advantages :math:`\mathcal{A}(s,a)=q(s,a)-v(s)`.

        Returns
        -------
        grads : pytree with ndarray leaves

            A batch of gradients.

        state : pytree

            The internal state of the forward-pass function. See :attr:`Policy.function_state
            <coax.Policy.function_state>` and :func:`haiku.transform_with_state` for more details.

        metrics : dict of scalar ndarrays

            The structure of the metrics dict is ``{name: score}``.

        """
        if self.REQUIRES_PROPENSITIES and jnp.all(transition_batch.logP == 0):
            warnings.warn(
                f"In order for {self.__class__.__name__} to work properly, transition_batch.logP "
                "should be non-zero. Please sample actions with their propensities: "
                "a, logp = pi(s, return_logp=True) and then add logp to your experience tracer, "
                "e.g. experience_replay_buffer.add(s, a, r, done, logp)")
        return self.grad_and_metrics_func(
            self.pi.params, self.pi.function_state, self.pi.rng, transition_batch, Adv,
            **self.hyperparams)

    @property
    def grad_and_metrics_func(self):
        r"""

        JIT-compiled function responsible for computing the gradients, along with the updated
        internal state of the forward-pass function and some performance metrics. This function is
        used by the :attr:`update` method.

        Parameters
        ----------
        params : pytree with ndarray leaves

            The model parameters (weights) used by the underlying policy.

        state : pytree

            The internal state of the forward-pass function. See :attr:`Policy.function_state
            <coax.Policy.function_state>` and :func:`haiku.transform_with_state` for more details.

        rng : PRNGKey

            A key to seed JAX's pseudo-random number generator.

        transition_batch : TransitionBatch

            A batch of transitions. Note that ``transition_batch.logP`` cannot
            be ``None``, as it is required by the PPO-clip objective.

        Adv : ndarray

            A batch of advantages :math:`\mathcal{A}(s,a)=q(s,a)-v(s)`.

        \*\*hyperparams

            Hyperparameters specific to the objective, see :attr:`hyperparams`.

        Returns
        -------
        grads : pytree with ndarray leaves

            A batch of gradients.

        state : pytree

            The internal state of the forward-pass function. See :attr:`Policy.function_state
            <coax.Policy.function_state>` and :func:`haiku.transform_with_state` for more details.

        metrics : dict of scalar ndarrays

            The structure of the metrics dict is ``{name: score}``.


        """
        return self._grad_and_metrics_func
