"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Nodegroup = exports.CapacityType = exports.NodegroupAmiType = void 0;
const aws_iam_1 = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const cluster_1 = require("./cluster");
const eks_generated_1 = require("./eks.generated");
const instance_types_1 = require("./instance-types");
/**
 * The AMI type for your node group.
 *
 * GPU instance types should use the `AL2_x86_64_GPU` AMI type, which uses the
 * Amazon EKS-optimized Linux AMI with GPU support. Non-GPU instances should use the `AL2_x86_64` AMI type, which
 * uses the Amazon EKS-optimized Linux AMI.
 *
 * @stability stable
 */
var NodegroupAmiType;
(function (NodegroupAmiType) {
    NodegroupAmiType["AL2_X86_64"] = "AL2_x86_64";
    NodegroupAmiType["AL2_X86_64_GPU"] = "AL2_x86_64_GPU";
    NodegroupAmiType["AL2_ARM_64"] = "AL2_ARM_64";
})(NodegroupAmiType = exports.NodegroupAmiType || (exports.NodegroupAmiType = {}));
/**
 * Capacity type of the managed node group.
 *
 * @stability stable
 */
var CapacityType;
(function (CapacityType) {
    CapacityType["SPOT"] = "SPOT";
    CapacityType["ON_DEMAND"] = "ON_DEMAND";
})(CapacityType = exports.CapacityType || (exports.CapacityType = {}));
/**
 * The Nodegroup resource class.
 *
 * @stability stable
 */
class Nodegroup extends core_1.Resource {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        var _a, _b, _c, _d, _e, _f, _g;
        super(scope, id, {
            physicalName: props.nodegroupName,
        });
        this.cluster = props.cluster;
        this.desiredSize = (_b = (_a = props.desiredSize) !== null && _a !== void 0 ? _a : props.minSize) !== null && _b !== void 0 ? _b : 2;
        this.maxSize = (_c = props.maxSize) !== null && _c !== void 0 ? _c : this.desiredSize;
        this.minSize = (_d = props.minSize) !== null && _d !== void 0 ? _d : 1;
        if (this.desiredSize > this.maxSize) {
            throw new Error(`Desired capacity ${this.desiredSize} can't be greater than max size ${this.maxSize}`);
        }
        if (this.desiredSize < this.minSize) {
            throw new Error(`Minimum capacity ${this.minSize} can't be greater than desired size ${this.desiredSize}`);
        }
        if (props.instanceType && props.instanceTypes) {
            throw new Error('"instanceType is deprecated, please use "instanceTypes" only.');
        }
        if (props.instanceType) {
            core_1.Annotations.of(this).addWarning('"instanceType" is deprecated and will be removed in the next major version. please use "instanceTypes" instead');
        }
        const instanceTypes = (_e = props.instanceTypes) !== null && _e !== void 0 ? _e : (props.instanceType ? [props.instanceType] : undefined);
        let expectedAmiType = undefined;
        if (instanceTypes && instanceTypes.length > 0) {
            // if the user explicitly configured instance types, we can calculate the expected ami type.
            expectedAmiType = getAmiType(instanceTypes);
            // if the user explicitly configured an ami type, make sure its the same as the expected one.
            if (props.amiType && props.amiType !== expectedAmiType) {
                throw new Error(`The specified AMI does not match the instance types architecture, either specify ${expectedAmiType} or dont specify any`);
            }
        }
        if (!props.nodeRole) {
            const ngRole = new aws_iam_1.Role(this, 'NodeGroupRole', {
                assumedBy: new aws_iam_1.ServicePrincipal('ec2.amazonaws.com'),
            });
            ngRole.addManagedPolicy(aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('AmazonEKSWorkerNodePolicy'));
            ngRole.addManagedPolicy(aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('AmazonEKS_CNI_Policy'));
            ngRole.addManagedPolicy(aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('AmazonEC2ContainerRegistryReadOnly'));
            this.role = ngRole;
        }
        else {
            this.role = props.nodeRole;
        }
        const resource = new eks_generated_1.CfnNodegroup(this, 'Resource', {
            clusterName: this.cluster.clusterName,
            nodegroupName: props.nodegroupName,
            nodeRole: this.role.roleArn,
            subnets: this.cluster.vpc.selectSubnets(props.subnets).subnetIds,
            // if a launch template is configured, we cannot apply a default since it
            // might exist in the launch template as well, causing a deployment failure.
            amiType: props.launchTemplateSpec !== undefined ? props.amiType : ((_f = props.amiType) !== null && _f !== void 0 ? _f : expectedAmiType),
            capacityType: props.capacityType ? props.capacityType.valueOf() : undefined,
            diskSize: props.diskSize,
            forceUpdateEnabled: (_g = props.forceUpdate) !== null && _g !== void 0 ? _g : true,
            // note that we don't check if a launch template is configured here (even though it might configure instance types as well)
            // because this doesn't have a default value, meaning the user had to explicitly configure this.
            instanceTypes: instanceTypes === null || instanceTypes === void 0 ? void 0 : instanceTypes.map(t => t.toString()),
            labels: props.labels,
            releaseVersion: props.releaseVersion,
            remoteAccess: props.remoteAccess ? {
                ec2SshKey: props.remoteAccess.sshKeyName,
                sourceSecurityGroups: props.remoteAccess.sourceSecurityGroups ?
                    props.remoteAccess.sourceSecurityGroups.map(m => m.securityGroupId) : undefined,
            } : undefined,
            scalingConfig: {
                desiredSize: this.desiredSize,
                maxSize: this.maxSize,
                minSize: this.minSize,
            },
            tags: props.tags,
        });
        if (props.launchTemplateSpec) {
            if (props.diskSize) {
                // see - https://docs.aws.amazon.com/eks/latest/userguide/launch-templates.html
                // and https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-eks-nodegroup.html#cfn-eks-nodegroup-disksize
                throw new Error('diskSize must be specified within the launch template');
            }
            /**
             * Instance types can be specified either in `instanceType` or launch template but not both. AS we can not check the content of
             * the provided launch template and the `instanceType` property is preferrable. We allow users to define `instanceType` property here.
             * see - https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-eks-nodegroup.html#cfn-eks-nodegroup-instancetypes
             */
            // TODO: update this when the L1 resource spec is updated.
            resource.addPropertyOverride('LaunchTemplate', {
                Id: props.launchTemplateSpec.id,
                Version: props.launchTemplateSpec.version,
            });
        }
        // managed nodegroups update the `aws-auth` on creation, but we still need to track
        // its state for consistency.
        if (this.cluster instanceof cluster_1.Cluster) {
            // see https://docs.aws.amazon.com/en_us/eks/latest/userguide/add-user-role.html
            this.cluster.awsAuth.addRoleMapping(this.role, {
                username: 'system:node:{{EC2PrivateDNSName}}',
                groups: [
                    'system:bootstrappers',
                    'system:nodes',
                ],
            });
        }
        this.nodegroupArn = this.getResourceArnAttribute(resource.attrArn, {
            service: 'eks',
            resource: 'nodegroup',
            resourceName: this.physicalName,
        });
        this.nodegroupName = this.getResourceNameAttribute(resource.ref);
    }
    /**
     * Import the Nodegroup from attributes.
     *
     * @stability stable
     */
    static fromNodegroupName(scope, id, nodegroupName) {
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.nodegroupName = nodegroupName;
            }
        }
        return new Import(scope, id);
    }
}
exports.Nodegroup = Nodegroup;
function getAmiTypeForInstanceType(instanceType) {
    return instance_types_1.INSTANCE_TYPES.graviton2.includes(instanceType.toString().substring(0, 3)) ? NodegroupAmiType.AL2_ARM_64 :
        instance_types_1.INSTANCE_TYPES.graviton.includes(instanceType.toString().substring(0, 2)) ? NodegroupAmiType.AL2_ARM_64 :
            instance_types_1.INSTANCE_TYPES.gpu.includes(instanceType.toString().substring(0, 2)) ? NodegroupAmiType.AL2_X86_64_GPU :
                instance_types_1.INSTANCE_TYPES.inferentia.includes(instanceType.toString().substring(0, 4)) ? NodegroupAmiType.AL2_X86_64_GPU :
                    NodegroupAmiType.AL2_X86_64;
}
// this function examines the CPU architecture of every instance type and determines
// what ami type is compatible for all of them. it either throws or produces a single value because
// instance types of different CPU architectures are not supported.
function getAmiType(instanceTypes) {
    const amiTypes = new Set(instanceTypes.map(i => getAmiTypeForInstanceType(i)));
    if (amiTypes.size == 0) { // protective code, the current implementation will never result in this.
        throw new Error(`Cannot determine any ami type comptaible with instance types: ${instanceTypes.map(i => i.toString).join(',')}`);
    }
    if (amiTypes.size > 1) {
        throw new Error('instanceTypes of different CPU architectures is not allowed');
    }
    return amiTypes.values().next().value;
}
//# sourceMappingURL=data:application/json;base64,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