"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.KubectlProvider = void 0;
const path = require("path");
const iam = require("@aws-cdk/aws-iam");
const lambda = require("@aws-cdk/aws-lambda");
const core_1 = require("@aws-cdk/core");
const cr = require("@aws-cdk/custom-resources");
const lambda_layer_awscli_1 = require("@aws-cdk/lambda-layer-awscli");
const lambda_layer_kubectl_1 = require("@aws-cdk/lambda-layer-kubectl");
const cluster_1 = require("./cluster");
class KubectlProvider extends core_1.NestedStack {
    constructor(scope, id, props) {
        super(scope, id);
        const cluster = props.cluster;
        if (!cluster.kubectlRole) {
            throw new Error('"kubectlRole" is not defined, cannot issue kubectl commands against this cluster');
        }
        if (cluster.kubectlPrivateSubnets && !cluster.kubectlSecurityGroup) {
            throw new Error('"kubectlSecurityGroup" is required if "kubectlSubnets" is specified');
        }
        const memorySize = cluster.kubectlMemory ? cluster.kubectlMemory.toMebibytes() : 1024;
        const handler = new lambda.Function(this, 'Handler', {
            code: lambda.Code.fromAsset(path.join(__dirname, 'kubectl-handler')),
            runtime: lambda.Runtime.PYTHON_3_7,
            handler: 'index.handler',
            timeout: core_1.Duration.minutes(15),
            description: 'onEvent handler for EKS kubectl resource provider',
            memorySize,
            environment: cluster.kubectlEnvironment,
            // defined only when using private access
            vpc: cluster.kubectlPrivateSubnets ? cluster.vpc : undefined,
            securityGroups: cluster.kubectlSecurityGroup ? [cluster.kubectlSecurityGroup] : undefined,
            vpcSubnets: cluster.kubectlPrivateSubnets ? { subnets: cluster.kubectlPrivateSubnets } : undefined,
        });
        // allow user to customize the layer
        if (!props.cluster.kubectlLayer) {
            handler.addLayers(new lambda_layer_awscli_1.AwsCliLayer(this, 'AwsCliLayer'));
            handler.addLayers(new lambda_layer_kubectl_1.KubectlLayer(this, 'KubectlLayer'));
        }
        else {
            handler.addLayers(props.cluster.kubectlLayer);
        }
        this.handlerRole = handler.role;
        this.handlerRole.addToPrincipalPolicy(new iam.PolicyStatement({
            actions: ['eks:DescribeCluster'],
            resources: [cluster.clusterArn],
        }));
        // allow this handler to assume the kubectl role
        cluster.kubectlRole.grant(this.handlerRole, 'sts:AssumeRole');
        const provider = new cr.Provider(this, 'Provider', {
            onEventHandler: handler,
            vpc: cluster.kubectlPrivateSubnets ? cluster.vpc : undefined,
            vpcSubnets: cluster.kubectlPrivateSubnets ? { subnets: cluster.kubectlPrivateSubnets } : undefined,
        });
        this.serviceToken = provider.serviceToken;
        this.roleArn = cluster.kubectlRole.roleArn;
    }
    static getOrCreate(scope, cluster) {
        // if this is an "owned" cluster, it has a provider associated with it
        if (cluster instanceof cluster_1.Cluster) {
            return cluster._attachKubectlResourceScope(scope);
        }
        // if this is an imported cluster, we need to provision a custom resource provider in this stack
        // we will define one per stack for each cluster based on the cluster uniqueid
        const uid = `${core_1.Names.nodeUniqueId(cluster.node)}-KubectlProvider`;
        const stack = core_1.Stack.of(scope);
        let provider = stack.node.tryFindChild(uid);
        if (!provider) {
            provider = new KubectlProvider(stack, uid, { cluster });
        }
        return provider;
    }
}
exports.KubectlProvider = KubectlProvider;
//# sourceMappingURL=data:application/json;base64,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