"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const assert_1 = require("@aws-cdk/assert");
const aws_ec2_1 = require("@aws-cdk/aws-ec2");
const core_1 = require("@aws-cdk/core");
const lib_1 = require("../lib");
describe('tests', () => {
    test('test specifying nonstandard port works', () => {
        const stack = new core_1.Stack(undefined, undefined, { env: { account: '1234', region: 'test' } });
        stack.node.setContext('availability-zones:1234:test', ['test-1a', 'test-1b']);
        const vpc = new aws_ec2_1.Vpc(stack, 'VCP');
        const lb = new lib_1.LoadBalancer(stack, 'LB', { vpc });
        lb.addListener({
            externalProtocol: lib_1.LoadBalancingProtocol.HTTP,
            externalPort: 8080,
            internalProtocol: lib_1.LoadBalancingProtocol.HTTP,
            internalPort: 8080,
        });
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ElasticLoadBalancing::LoadBalancer', {
            Listeners: [{
                    InstancePort: '8080',
                    InstanceProtocol: 'http',
                    LoadBalancerPort: '8080',
                    Protocol: 'http',
                }],
        }));
    });
    test('add a health check', () => {
        // GIVEN
        const stack = new core_1.Stack();
        const vpc = new aws_ec2_1.Vpc(stack, 'VCP');
        // WHEN
        new lib_1.LoadBalancer(stack, 'LB', {
            vpc,
            healthCheck: {
                interval: core_1.Duration.minutes(1),
                path: '/ping',
                protocol: lib_1.LoadBalancingProtocol.HTTPS,
                port: 443,
            },
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ElasticLoadBalancing::LoadBalancer', {
            HealthCheck: {
                HealthyThreshold: '2',
                Interval: '60',
                Target: 'HTTPS:443/ping',
                Timeout: '5',
                UnhealthyThreshold: '5',
            },
        }));
    });
    test('add a listener and load balancing target', () => {
        // GIVEN
        const stack = new core_1.Stack();
        const vpc = new aws_ec2_1.Vpc(stack, 'VCP');
        const elb = new lib_1.LoadBalancer(stack, 'LB', {
            vpc,
            healthCheck: {
                interval: core_1.Duration.minutes(1),
                path: '/ping',
                protocol: lib_1.LoadBalancingProtocol.HTTPS,
                port: 443,
            },
        });
        // WHEN
        elb.addListener({ externalPort: 80, internalPort: 8080 });
        elb.addTarget(new FakeTarget());
        // THEN: at the very least it added a security group rule for the backend
        assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::SecurityGroup', {
            SecurityGroupEgress: [
                {
                    Description: 'Port 8080 LB to fleet',
                    CidrIp: '666.666.666.666/666',
                    FromPort: 8080,
                    IpProtocol: 'tcp',
                    ToPort: 8080,
                },
            ],
        }));
    });
    test('enable cross zone load balancing', () => {
        // GIVEN
        const stack = new core_1.Stack();
        const vpc = new aws_ec2_1.Vpc(stack, 'VCP');
        // WHEN
        new lib_1.LoadBalancer(stack, 'LB', {
            vpc,
            crossZone: true,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ElasticLoadBalancing::LoadBalancer', {
            CrossZone: true,
        }));
    });
    test('disable cross zone load balancing', () => {
        // GIVEN
        const stack = new core_1.Stack();
        const vpc = new aws_ec2_1.Vpc(stack, 'VCP');
        // WHEN
        new lib_1.LoadBalancer(stack, 'LB', {
            vpc,
            crossZone: false,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ElasticLoadBalancing::LoadBalancer', {
            CrossZone: false,
        }));
    });
    test('cross zone load balancing enabled by default', () => {
        // GIVEN
        const stack = new core_1.Stack();
        const vpc = new aws_ec2_1.Vpc(stack, 'VCP');
        // WHEN
        new lib_1.LoadBalancer(stack, 'LB', {
            vpc,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ElasticLoadBalancing::LoadBalancer', {
            CrossZone: true,
        }));
    });
    test('use specified subnet', () => {
        // GIVEN
        const stack = new core_1.Stack();
        const vpc = new aws_ec2_1.Vpc(stack, 'VCP', {
            subnetConfiguration: [
                {
                    name: 'public',
                    subnetType: aws_ec2_1.SubnetType.PUBLIC,
                    cidrMask: 21,
                },
                {
                    name: 'private1',
                    subnetType: aws_ec2_1.SubnetType.PRIVATE,
                    cidrMask: 21,
                },
                {
                    name: 'private2',
                    subnetType: aws_ec2_1.SubnetType.PRIVATE,
                    cidrMask: 21,
                },
            ],
        });
        // WHEN
        new lib_1.LoadBalancer(stack, 'LB', {
            vpc,
            subnetSelection: {
                subnetName: 'private1',
            },
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ElasticLoadBalancing::LoadBalancer', {
            Subnets: vpc.selectSubnets({
                subnetName: 'private1',
            }).subnetIds.map((subnetId) => stack.resolve(subnetId)),
        }));
    });
});
class FakeTarget {
    constructor() {
        this.connections = new aws_ec2_1.Connections({
            peer: aws_ec2_1.Peer.ipv4('666.666.666.666/666'),
        });
    }
    attachToClassicLB(_loadBalancer) {
        // Nothing to do. Normally we set a property on ourselves so
        // our instances know to bind to the LB on startup.
    }
}
//# sourceMappingURL=data:application/json;base64,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