"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.DeployAssert = void 0;
const aws_cdk_lib_1 = require("aws-cdk-lib");
const constructs_1 = require("constructs");
const assertions_1 = require("../assertions");
const hash_1 = require("../private/hash");
const sdk_1 = require("../sdk");
const DEPLOY_ASSERT_SYMBOL = Symbol.for('@aws-cdk/integ-tests.DeployAssert');
/**
 * Construct that allows for registering a list of assertions
 * that should be performed on a construct
 */
class DeployAssert extends constructs_1.Construct {
    constructor(scope, props) {
        super(scope, 'Default');
        this.scope = props?.stack ?? new aws_cdk_lib_1.Stack(scope, 'DeployAssert');
        Object.defineProperty(this, DEPLOY_ASSERT_SYMBOL, { value: true });
    }
    /**
     * Returns whether the construct is a DeployAssert construct
     */
    static isDeployAssert(x) {
        return x !== null && typeof (x) === 'object' && DEPLOY_ASSERT_SYMBOL in x;
    }
    /**
     * Finds a DeployAssert construct in the given scope
     */
    static of(construct) {
        const scopes = constructs_1.Node.of(constructs_1.Node.of(construct).root).findAll();
        const deployAssert = scopes.find(s => DeployAssert.isDeployAssert(s));
        if (!deployAssert) {
            throw new Error('No DeployAssert construct found in scopes');
        }
        return deployAssert;
    }
    awsApiCall(service, api, parameters, outputPaths) {
        return new sdk_1.AwsApiCall(this.scope, `AwsApiCall${service}${api}`, {
            api,
            service,
            parameters,
            outputPaths,
        });
    }
    invokeFunction(props) {
        const hash = hash_1.md5hash(this.scope.resolve(props));
        return new sdk_1.LambdaInvokeFunction(this.scope, `LambdaInvoke${hash}`, props);
    }
    expect(id, expected, actual) {
        new assertions_1.EqualsAssertion(this.scope, `EqualsAssertion${id}`, {
            expected,
            actual,
        });
    }
}
exports.DeployAssert = DeployAssert;
//# sourceMappingURL=data:application/json;base64,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