# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['rich_theme_manager']

package_data = \
{'': ['*']}

install_requires = \
['rich>=11.0']

entry_points = \
{'console_scripts': ['rich_theme_manager = rich_theme_manager:main']}

setup_kwargs = {
    'name': 'rich-theme-manager',
    'version': '0.7.0',
    'description': 'Manage rich themes for CLI applications',
    'long_description': '# Rich Theme Manager\n\n## Description\n\nImplements a basic "theme manager" class for managing rich [Themes](https://rich.readthedocs.io/en/stable/style.html#style-themes) in your [rich](https://github.com/Textualize/rich) CLI application.\n\nThe rich package provides an easy way to define custom styles and themes for use with rich.  This package provides a simple way to manage the themes: e.g. to list, add, remove themes, for the user to preview themes, and to manage themes on disk.\n\n## What problem does this solve?\n\nConsider this scenario from the rich [documentation](https://rich.readthedocs.io/en/stable/style.html#style-themes):\n\nIf you re-use styles it can be a maintenance headache if you ever want to modify an attribute or color – you would have to change every line where the style is used. Rich provides a Theme class which you can use to define custom styles that you can refer to by name. That way you only need to update your styles in one place.\n\nStyle themes can make your code more semantic, for instance a style called "warning" better expresses intent that "italic magenta underline".\n\nTo use a style theme, construct a Theme instance and pass it to the Console constructor. Here’s an example:\n\n```python\nfrom rich.console import Console\nfrom rich.theme import Theme\ncustom_theme = Theme({\n    "info": "dim cyan",\n    "warning": "magenta",\n    "danger": "bold red"\n})\nconsole = Console(theme=custom_theme)\nconsole.print("This is information", style="info")\nconsole.print("[warning]The pod bay doors are locked[/warning]")\nconsole.print("Something terrible happened!", style="danger")\n```\n\nI highly recommend the use of Themes in your rich application instead of hard-coding colors and styles. However, there\'s still a problem of managing themes.  For example, letting the user of your application change the default theme styles you\'ve chosen.  What if they are color blind and can\'t distinguish the colors you\'ve selected? What if they\'re using a light terminal background and you\'ve selected colors best suited for a dark terminal?  \n\nThe rich `Theme` class provides everything you to manage these situations but doing so requires a fair amount of boiler plate code for each application. This package attempts to provide an easy solution for these scenarios with a minimal amount of code.  Instead of having to implement management of theme files and allowing the user to list or preview themes yourself, you can use the `ThemeManager` class to manage themes for you.\n\n## Synopsis\n\nUsing `rich_theme_manager` is easy and takes just a few lines of code.  Import `Theme` from `rich_theme_manager` instead of from `rich.theme` (`rich_theme_manager.Theme` subclasses `rich.theme.Theme`) then use `ThemeManager` to manage themes.  `ThemeManager` can be created with or without a `theme_dir` argument. If you don\'t provide `theme_dir`, `ThemeManager` will not manage themes on disk.  This may still be useful for allowing the user to list and preview themes.  If you do provide `theme_dir`, any default themes passed to `ThemeManager` will be written to `theme_dir` and any theme config files found in `theme_dir` will be loaded.\n\n```python\nfrom rich.console import Console\nfrom rich.style import Style\nimport pathlib\n\nfrom rich_theme_manager import Theme, ThemeManager\n\n\nTHEMES = [\n    Theme(\n        name="dark",\n        description="Dark mode theme",\n        tags=["dark"],\n        styles={\n            "info": "dim cyan",\n            "warning": "bold magenta",\n            "danger": "bold red",\n        },\n    ),\n    Theme(\n        name="light",\n        description="Light mode theme",\n        styles={\n            "info": Style(color="#22863a", bold=True),\n            "warning": Style(color="#032f62", bold=True),\n            "danger": Style(color="#b31d28", bold=True, underline=True, italic=True),\n        },\n    ),\n    Theme(\n        name="mono",\n        description="Monochromatic theme",\n        tags=["mono", "colorblind"],\n        styles={\n            "info": "italic",\n            "warning": "bold",\n            "danger": "reverse bold",\n        },\n    ),\n]\n\nif __name__ == "__main__":\n    # you can specify a config directory to save/load themes to/from\n    theme_dir = pathlib.Path("~/.rich_theme_manager/themes").expanduser()\n    theme_dir.expanduser().mkdir(parents=True, exist_ok=True)\n\n    theme_manager = ThemeManager(theme_dir=theme_dir, themes=THEMES)\n    theme_manager.list_themes()\n    print("\\n")\n\n    dark = theme_manager.get("dark")\n    theme_manager.preview_theme(dark)\n    console = Console(theme=dark)\n    print("\\n")\n\n    console.print("This is information", style="info")\n    console.print("[warning]The pod bay doors are locked[/warning]")\n    console.print("Something terrible happened!", style="danger")\n```\n\n![Example output](https://github.com/RhetTbull/rich_theme_manager/raw/main/images/example1.png)\n\n## Example app\n\nA simple example app that demonstrates the ThemeManager class comes with rich_theme_manager in `__main__.py`:\n\n`python -m rich_theme_manager`:\n\n```text\nusage: rich_theme_manager [-h] [--example [EXAMPLE]] [--list] [--preview THEME] [--config THEME]\n\nExample CLI usage of rich_theme_manager\n\noptional arguments:\n  -h, --help           show this help message and exit\n  --example [EXAMPLE]  Show example output for theme.\n  --list               List themes.\n  --preview THEME      Preview theme.\n  --config THEME       Print configuration for theme THEME.\n```\n\n`python -m rich_theme_manager --list`:\n\n![Example --list output](https://github.com/RhetTbull/rich_theme_manager/raw/main/images/list.png)\n\n`python -m rich_theme_manager --preview dark`:\n\n![Example --preview output](https://github.com/RhetTbull/rich_theme_manager/raw/main/images/preview_dark.png)\n\n## Documentation\n\n### Theme class\n\n`rich_theme_manager.Theme` is a subclass of `rich.theme.Theme` and provides additional functionality for managing themes.\n\n```python\n    Theme(\n          name: str,\n          description: Optional[str] = None,\n          styles: Optional[Mapping[str, StyleType]] = None,\n          inherit: bool = True,\n          tags: Optional[List[str]] = None,\n          path: Optional[str] = None,\n    )\n```\n\nArguments\n\n* `name`: The name of the theme; required\n* `description`: An optional description of the theme.\n* `styles`: An optional mapping of style names to styles.\n* `inherit`: Whether the theme inherits from the default theme.\n* `tags`: An optional list of tags for the theme; useful for allowing user to filter themes.\n* `path`: The path to the theme file; in normal use this is not needed as `ThemeManager` will automatically create the theme file.\n\nProperties\n\n* `Theme().name`: The name of the theme\n* `Theme().description`: Description of the theme\n* `Theme().tags`: List of tags for the theme\n* `Theme().inherit`: bool indicating whether the theme inherits from the default theme\n* `Theme().style_names`: List of names for styles in the theme\n* `Theme().path`: The path to the theme file; (getter/setter)\n* `Theme().config`: Contents of a configuration file for the theme (same format as `rich.theme.Theme().config` but with an additional `[metadata]` section)\n\nMethods\n\n* `Theme().save()`: Save the theme to disk (to `Theme().path`)\n* `Theme().load()`: Load the theme from disk (from `Theme().path`)\n* `Theme().to_file(path: str)`: Save the theme to disk (to `path`)\n\nClass Methods:\n\n* `Theme.from_file(config_file: IO[str], source: Optional[str] = None, inherit: bool = True)` -> `Theme`: Load a theme from a text mode configuration file (in [configparser](https://docs.python.org/3/library/configparser.html) INI format).\n* `Theme.read(path: str, inherit: bool = True) -> Theme`: Load a theme from disk (from `path`)\n\nThe `.theme` INI file format looks like this:\n\n```ini\n[metadata]\nname = dark\ndescription = Dark mode theme\ntags = dark\ninherit = True\n\n[styles]\ndanger = bold red\ninfo = dim cyan\nwarning = bold magenta\n```\n\nHere\'s an real world example of a theme INI file from one of my [apps](https://github.com/RhetTbull/osxphotos):\n\n```INI\n[metadata]\nname = dark\ndescription = Dark mode theme\ntags = dark\ninherit = True\n\n[styles]\nbar.back = rgb(68,71,90)\nbar.complete = rgb(249,38,114)\nbar.finished = rgb(80,250,123)\nbar.pulse = rgb(98,114,164)\ncolor = rgb(248,248,242)\ncount = rgb(139,233,253)\nerror = bold rgb(255,85,85)\nfilename = bold rgb(189,147,249)\nfilepath = bold rgb(80,250,123)\nhighlight = bold #000000 on #d73a49\nnum = bold rgb(139,233,253)\nprogress.elapsed = rgb(139,233,253)\nprogress.percentage = rgb(255,121,198)\nprogress.remaining = rgb(139,233,253)\ntime = bold rgb(139,233,253)\nuuid = rgb(255,184,108)\nwarning = bold rgb(241,250,140)\n```\n\n`Theme` implements the [rich Console protocol](https://rich.readthedocs.io/en/stable/protocol.html) which means that you use `rich.print()` and `rich.console.Console().print()` to print a theme to the console.  Doing so results in a preview of the theme which visually shows the colors and styles used in the theme.\n\n![Theme preview](https://github.com/RhetTbull/rich_theme_manager/raw/main/images/theme_preview_print.png)\n\nThe `Theme` preview will use default sample text for each style. You can change the sample text by setting the `rich_theme_manager.theme.SAMPLE_TEXT` global variable.\n\n![Theme preview with sample text](https://github.com/RhetTbull/rich_theme_manager/raw/main/images/theme_preview_sample_text.png)\n\n`Theme` implements the `__eq__` method so two `Theme` instances can be easily compared for equality. `Theme` instances are considered equal if all properties with exception of `path` are equal.\n\n### ThemeManager class\n\n```python\n    ThemeManager(\n        theme_dir: Optional[str] = None,\n        themes: Optional[List[Theme]] = None,\n    )\n```\n\nArguments:\n\n* `theme_dir`: Optional directory to save/load themes to/from.\n* `themes`: Optional list of Theme objects\n\nIf provided, `theme_dir` must exist.  If `theme_dir` is set (for example, using [click.get_app_dir](https://click.palletsprojects.com/en/8.0.x/api/?highlight=get_app_dir#click.get_app_dir)), upon initialization `ThemeManager` will save any default themes provided via `themes` to `theme_dir` and load any themes from `theme_dir`.  Theme files are standard INI files as created by [configparser](https://docs.python.org/3/library/configparser.html) and are named `<name>.theme` where `<name>` is the name of the Theme (see `Theme.name`).  If a theme file already exists, it will be loaded and `ThemeManager` will not overwrite it.\n\nProperties:\n\n* `ThemeManager().themes`: List of themes\n\nMethods:\n\n* `ThemeManager().add(theme: Theme, overwrite=False) -> None`: Add a theme to the list of managed themes.  If `overwrite` is True, the theme file will be overwritten if it already exists.\n* `ThemeManager().remove(theme: Theme) -> None`: Remove a theme from the list of managed themes and delete the theme file if it exists.\n* `ThemeManager().get(theme_name: str) -> Theme`: Get a theme by name. Raises \'ValueError` if no theme with the given name is found.\n* `ThemeManager().load_themes(theme_dir=None) -> None`: Load themes from `theme_dir` (or `ThemeManager().theme_dir` if not provided).  Any `.theme` files found in `theme_dir` will be loaded and added to the list of managed themes.\n* `ThemeManager().write_themes(overwrite=False) -> None`: Write themes to file (as specified in each `Theme().path` which will be set automatically by `ThemeManager`).  If `overwrite` is True, the theme file will be overwritten if it already exists.\n\n* `ThemeManager().list_themes(show_path: bool = True, theme_names: Optional[List[str]] = None, console: Optional[Console] = None) -> None`: Print a list of themes to the console.  If `show_path` is True, the path to the theme file will be printed.  If `theme_names` is provided, only themes with names in the list will be printed. An optional `rich.console.Console()` instance may be provided to print to a specific console.\n\n![ThemeManager list example](https://github.com/RhetTbull/rich_theme_manager/raw/main/images/theme_manager_list.png)\n\nClass Methods:\n\n* `ThemeManager.preview_theme(theme: Theme, sample_text: Optional[str] = None, show_path: bool = True, console: Optional[Console] = None) -> None`: Print a preview of the theme to the console showing the style of each style in the theme.  If `sample_text` is provided, it will be used as the sample text to preview otherwise a default string will be used.  If `show_path` is True, the path to the theme file will be printed.  An optional `rich.console.Console()` instance may be provided to print to a specific console.\n\n![ThemeManager preview example](https://github.com/RhetTbull/rich_theme_manager/raw/main/images/theme_manager_preview.png)\n\n## Test Coverage\n\n100% coverage of all code with exception of the example CLI app.\n\n```text\nName                               Stmts   Miss  Cover\n------------------------------------------------------\nrich_theme_manager/__init__.py         5      0   100%\nrich_theme_manager/manager.py         71      0   100%\nrich_theme_manager/theme.py          134      0   100%\ntests/__init__.py                      0      0   100%\ntests/conftest.py                      7      0   100%\ntests/test_rich_theme_manager.py     256      0   100%\n------------------------------------------------------\nTOTAL                                473      0   100%\n```\n\n## License\n\nMIT License\n\n## Contributing\n\nContributions of all kinds are welcome!  Please submit pull requests, issues, and/or suggestions to the [github repo](https://github.com/RhetTbull/rich_theme_manager).\n\n## Credits\n\nThank you to [Will McGugan](https://github.com/willmcgugan) for creating [rich](https://github.com/Textualize/rich) and helping to make our command line interfaces more beautiful!\n\n## Projects Using Rich Theme Manager\n\n* [osxphotos](https://github.com/RhetTbull/osxphotos): Python app to export pictures and associated metadata from Apple Photos on macOS. Also includes a package to provide programmatic access to the Photos library, pictures, and metadata.\n',
    'author': 'Rhet Turnbull',
    'author_email': 'rturnbull+git@gmail.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/RhetTbull/rich_theme_manager',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'entry_points': entry_points,
    'python_requires': '>=3.8,<4.0.0',
}


setup(**setup_kwargs)
