"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.SubscriptionProtocol = exports.Subscription = void 0;
const aws_iam_1 = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const sns_generated_1 = require("./sns.generated");
/**
 * A new subscription.
 *
 * Prefer to use the `ITopic.addSubscription()` methods to create instances of
 * this class.
 */
class Subscription extends core_1.Resource {
    constructor(scope, id, props) {
        super(scope, id);
        if (props.rawMessageDelivery && ['http', 'https', 'sqs'].indexOf(props.protocol) < 0) {
            throw new Error('Raw message delivery can only be enabled for HTTP/S and SQS subscriptions.');
        }
        if (props.filterPolicy) {
            if (Object.keys(props.filterPolicy).length > 5) {
                throw new Error('A filter policy can have a maximum of 5 attribute names.');
            }
            this.filterPolicy = Object.entries(props.filterPolicy)
                .reduce((acc, [k, v]) => ({ ...acc, [k]: v.conditions }), {});
            let total = 1;
            Object.values(this.filterPolicy).forEach(filter => { total *= filter.length; });
            if (total > 100) {
                throw new Error(`The total combination of values (${total}) must not exceed 100.`);
            }
        }
        this.deadLetterQueue = this.buildDeadLetterQueue(props);
        new sns_generated_1.CfnSubscription(this, 'Resource', {
            endpoint: props.endpoint,
            protocol: props.protocol,
            topicArn: props.topic.topicArn,
            rawMessageDelivery: props.rawMessageDelivery,
            filterPolicy: this.filterPolicy,
            region: props.region,
            redrivePolicy: this.buildDeadLetterConfig(this.deadLetterQueue),
        });
    }
    buildDeadLetterQueue(props) {
        if (!props.deadLetterQueue) {
            return undefined;
        }
        const deadLetterQueue = props.deadLetterQueue;
        deadLetterQueue.addToResourcePolicy(new aws_iam_1.PolicyStatement({
            resources: [deadLetterQueue.queueArn],
            actions: ['sqs:SendMessage'],
            principals: [new aws_iam_1.ServicePrincipal('sns.amazonaws.com')],
            conditions: {
                ArnEquals: { 'aws:SourceArn': props.topic.topicArn },
            },
        }));
        return deadLetterQueue;
    }
    buildDeadLetterConfig(deadLetterQueue) {
        if (deadLetterQueue) {
            return {
                deadLetterTargetArn: deadLetterQueue.queueArn,
            };
        }
        else {
            return undefined;
        }
    }
}
exports.Subscription = Subscription;
/**
 * The type of subscription, controlling the type of the endpoint parameter.
 */
var SubscriptionProtocol;
(function (SubscriptionProtocol) {
    /**
     * JSON-encoded message is POSTED to an HTTP url.
     */
    SubscriptionProtocol["HTTP"] = "http";
    /**
     * JSON-encoded message is POSTed to an HTTPS url.
     */
    SubscriptionProtocol["HTTPS"] = "https";
    /**
     * Notifications are sent via email.
     */
    SubscriptionProtocol["EMAIL"] = "email";
    /**
     * Notifications are JSON-encoded and sent via mail.
     */
    SubscriptionProtocol["EMAIL_JSON"] = "email-json";
    /**
     * Notification is delivered by SMS
     */
    SubscriptionProtocol["SMS"] = "sms";
    /**
     * Notifications are enqueued into an SQS queue.
     */
    SubscriptionProtocol["SQS"] = "sqs";
    /**
     * JSON-encoded notifications are sent to a mobile app endpoint.
     */
    SubscriptionProtocol["APPLICATION"] = "application";
    /**
     * Notifications trigger a Lambda function.
     */
    SubscriptionProtocol["LAMBDA"] = "lambda";
})(SubscriptionProtocol = exports.SubscriptionProtocol || (exports.SubscriptionProtocol = {}));
//# sourceMappingURL=data:application/json;base64,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