"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.CreateImageSimulation = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const data_type_1 = require("../../domain/data-type");
const response_code_1 = require("../../domain/response-code");
const aws_api_step_1 = require("../../parent-steps/automation/aws-api-step");
const wait_for_and_assert_1 = require("../../parent-steps/automation/sub-steps/wait-for-and-assert");
const prune_and_transform_record_1 = require("../../utils/prune-and-transform-record");
const automation_step_simulation_1 = require("../automation-step-simulation");
const automation_simulation_base_1 = require("./automation-simulation-base");
/**
 * AutomationStep implemenation for aws:createImage
 * https://docs.aws.amazon.com/systems-manager/latest/userguide/automation-action-create.html
 */
class CreateImageSimulation extends automation_simulation_base_1.AutomationSimulationBase {
    constructor(step, props) {
        super(step);
        this.action = 'aws:createImage';
        this.createImageStep = step;
        this.props = props;
    }
    executeStep(inputs) {
        console.log(`CreateImage: Creating image of ${this.createImageStep.instanceId}`);
        const imageId = this.createImage(inputs);
        console.log(`CreateImage: Waiting for ${imageId} to be available`);
        const state = this.waitForImageAvailable(imageId);
        console.log(`CreateImage: Image ${imageId} is available`);
        return {
            ImageId: imageId,
            ImageState: state,
        };
    }
    createImage(inputs) {
        const apiParamMap = {
            InstanceId: this.createImageStep.instanceId,
            Name: this.createImageStep.imageName,
            Description: this.createImageStep.imageDescription,
            NoReboot: this.createImageStep.noReboot,
            BlockDeviceMappings: this.createImageStep.blockDeviceMappings,
        };
        const apiParams = prune_and_transform_record_1.pruneAndTransformRecord(apiParamMap, x => x.resolve(inputs));
        const result = new automation_step_simulation_1.AutomationStepSimulation(new aws_api_step_1.AwsApiStep(new aws_cdk_lib_1.Stack(), 'createImage', {
            service: 'EC2',
            pascalCaseApi: 'CreateImage',
            apiParams,
            outputs: [{
                    outputType: data_type_1.DataTypeEnum.STRING,
                    name: 'ImageId',
                    selector: '$.ImageId',
                }],
        }), this.props).invoke({});
        if (result.responseCode !== response_code_1.ResponseCode.SUCCESS) {
            throw new Error(`Create image failed for ${apiParams.InstanceId}: ${result.stackTrace}`);
        }
        return result.outputs?.['createImage.ImageId'];
    }
    waitForImageAvailable(imageId) {
        wait_for_and_assert_1.waitForAndAssertImageAvailable({
            imageId: imageId,
        }, this.props);
        return 'available';
    }
}
exports.CreateImageSimulation = CreateImageSimulation;
_a = JSII_RTTI_SYMBOL_1;
CreateImageSimulation[_a] = { fqn: "@cdklabs/cdk-ssm-documents.CreateImageSimulation", version: "0.0.22" };
//# sourceMappingURL=data:application/json;base64,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