"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.StringStep = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const choice_1 = require("../../domain/choice");
const data_type_1 = require("../../domain/data-type");
const operation_1 = require("../../domain/operation");
const string_variable_1 = require("../../interface/variables/string-variable");
const approve_step_1 = require("../../parent-steps/automation/approve-step");
const assert_aws_resource_step_1 = require("../../parent-steps/automation/assert-aws-resource-step");
const aws_api_step_1 = require("../../parent-steps/automation/aws-api-step");
const branch_step_1 = require("../../parent-steps/automation/branch-step");
const execute_script_step_1 = require("../../parent-steps/automation/execute-script-step");
const pause_step_1 = require("../../parent-steps/automation/pause-step");
const sleep_step_1 = require("../../parent-steps/automation/sleep-step");
const wait_for_resource_step_1 = require("../../parent-steps/automation/wait-for-resource-step");
const composite_step_1 = require("./composite-step");
// eslint-disable-next-line
const yaml = require('js-yaml');
/**
 * StringStep allows for including a step from an existing automation document verbatim.
 * This is useful in that it allows developers to integrate with existing document steps.
 * This step can be used just as you would use any other Step including simulation and deployment.
 */
class StringStep extends composite_step_1.CompositeAutomationStep {
    constructor(scope, id, props) {
        super(scope, id);
        const sharedProps = {};
        sharedProps.name = props.name;
        sharedProps.action = props.action;
        if (props.maxAttempts) {
            sharedProps.maxAttempts = props.maxAttempts;
        }
        if (props.timeoutSeconds) {
            sharedProps.timeoutSeconds = props.timeoutSeconds;
        }
        if (props.isEnd != undefined) {
            sharedProps.isEnd = props.isEnd;
        }
        if (props.isCritical != undefined) {
            sharedProps.isCritical = props.isCritical;
        }
        if (props.outputs != undefined) {
            sharedProps.outputs = this.toOutputs(props.outputs);
        }
        const { Service, Api, PropertySelector, DesiredValues, ...restParams } = props.inputs;
        switch (props.action) {
            case 'aws:executeAwsApi':
                this.automationStep = new aws_api_step_1.AwsApiStep(this, props.name, {
                    service: Service,
                    pascalCaseApi: Api,
                    apiParams: restParams,
                    outputs: sharedProps.outputs,
                    ...sharedProps,
                });
                break;
            case 'aws:waitForAwsResourceProperty':
                this.automationStep = new wait_for_resource_step_1.WaitForResourceStep(this, props.name, {
                    service: Service,
                    pascalCaseApi: Api,
                    apiParams: restParams,
                    selector: PropertySelector,
                    desiredValues: DesiredValues,
                    ...sharedProps,
                });
                break;
            case 'aws:assertAwsResourceProperty':
                this.automationStep = new assert_aws_resource_step_1.AssertAwsResourceStep(this, props.name, {
                    service: Service,
                    pascalCaseApi: Api,
                    apiParams: restParams,
                    selector: PropertySelector,
                    desiredValues: DesiredValues,
                    ...sharedProps,
                });
                break;
            case 'aws:pause':
                this.automationStep = new pause_step_1.PauseStep(this, props.name, {
                    ...sharedProps,
                });
                break;
            case 'aws:sleep':
                this.automationStep = new sleep_step_1.SleepStep(this, props.name, {
                    sleepSeconds: this.parseDurationToSeconds(restParams.Duration),
                    ...sharedProps,
                });
                break;
            case 'aws:executeScript':
                const inputs = {};
                Object.entries(restParams.InputPayload).forEach(([key, value]) => inputs[key] = this.toVariable(value));
                this.automationStep = new execute_script_step_1.ExecuteScriptStep(this, props.name, {
                    language: execute_script_step_1.ScriptLanguage.fromRuntime(restParams.Runtime, restParams.Handler),
                    inputPayload: inputs,
                    code: execute_script_step_1.ScriptCode.inline(restParams.Script),
                    ...sharedProps,
                });
                break;
            case 'aws:branch':
                this.automationStep = new branch_step_1.BranchStep(this, props.name, {
                    choices: this.toChoices(restParams.Choices),
                    defaultStepName: restParams.Default,
                    ...sharedProps,
                });
                break;
            case 'aws:approve':
                this.automationStep = new approve_step_1.ApproveStep(this, props.name, {
                    approvers: restParams.Approvers,
                    minRequiredApprovals: restParams.MinRequiredApprovals,
                    message: restParams.Message,
                    notificationArn: restParams.NotificationArn,
                    ...sharedProps,
                });
                break;
            default:
                throw new Error('Action not implemented as StringStep ' + props.action);
        }
    }
    /**
       * Builds a step object from a yaml declaration.
       * You may cast the step afterword to the associated Step for this action
       * if you wish to gain access to action specific functionality,
       * @returns Step associated with the provided activity.
       */
    static fromYaml(scope, inputYaml) {
        return StringStep.fromObject(scope, yaml.load(inputYaml));
    }
    /**
       * Builds a step object from a json declaration.
       * You may cast the step afterword to the associated Step for this action
       * if you wish to gain access to action specific functionality,
       * @returns Step associated with the provided activity.
       */
    static fromJson(scope, json) {
        return StringStep.fromObject(scope, JSON.parse(json));
    }
    /**
       * Builds a step object from an object.
       * You may cast the step afterword to the associated Step for this action
       * if you wish to gain access to action specific functionality,
       * @returns Step associated with the provided activity.
       */
    static fromObject(scope, props) {
        return new StringStep(scope, props.name, props);
    }
    addToDocument(doc) {
        doc.addStep(this.automationStep);
    }
    toOutputs(declaredOutputs) {
        return declaredOutputs.map(declaredOutput => this.toOutput(declaredOutput));
    }
    toOutput(declaredOutput) {
        return {
            name: declaredOutput.Name,
            selector: declaredOutput.Selector,
            outputType: data_type_1.DataType.fromDataType(declaredOutput.Type).dataTypeEnum,
        };
    }
    toChoices(declaredChoices) {
        return declaredChoices.map(declaredChoice => this.toChoice(declaredChoice));
    }
    toChoice(declaredChoice) {
        const operationEntry = Object.entries(declaredChoice)
            .filter(entry => operation_1.OperationEvaluator.STRING_TO_OPERATION[entry[0]] != undefined)[0];
        return new choice_1.Choice({
            jumpToStepName: declaredChoice.NextStep,
            variable: this.toVariable(declaredChoice.Variable),
            operation: operation_1.OperationEvaluator.fromOperationName(operationEntry[0]),
            constant: operationEntry[1],
        });
    }
    toVariable(variable) {
        if (variable.includes('{{')) {
            return new string_variable_1.StringVariable(variable.replace('{{', '').replace('}}', '').trim());
        }
        return new string_variable_1.HardCodedString(variable.trim());
    }
    parseDurationToSeconds(iso8601Duration) {
        // https://stackoverflow.com/a/29153059
        const iso8601DurationRegex = /(-)?P(?:([.,\d]+)Y)?(?:([.,\d]+)M)?(?:([.,\d]+)W)?(?:([.,\d]+)D)?T(?:([.,\d]+)H)?(?:([.,\d]+)M)?(?:([.,\d]+)S)?/;
        const matches = iso8601Duration.match(iso8601DurationRegex);
        if (matches == undefined) {
            throw new Error('Could not parse Duration' + iso8601Duration);
        }
        const periods = {
            years: matches[2] === undefined ? 0 : Number.parseInt(matches[2]),
            months: matches[3] === undefined ? 0 : Number.parseInt(matches[3]),
            weeks: matches[4] === undefined ? 0 : Number.parseInt(matches[4]),
            days: matches[5] === undefined ? 0 : Number.parseInt(matches[5]),
            hours: matches[6] === undefined ? 0 : Number.parseInt(matches[6]),
            minutes: matches[7] === undefined ? 0 : Number.parseInt(matches[7]),
            seconds: matches[8] === undefined ? 0 : Number.parseInt(matches[8]),
        };
        return periods.seconds +
            periods.minutes * 60 +
            periods.hours * 60 * 60 +
            periods.days * 60 * 60 * 24 +
            periods.weeks * 60 * 60 * 24 * 7 +
            periods.months * 60 * 60 * 24 * 30.5 +
            periods.years * 60 * 60 * 24 * 365;
    }
    ;
}
exports.StringStep = StringStep;
_a = JSII_RTTI_SYMBOL_1;
StringStep[_a] = { fqn: "@cdklabs/cdk-ssm-documents.StringStep", version: "0.0.22" };
//# sourceMappingURL=data:application/json;base64,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