import { AutomationStep } from '../parent-steps/automation-step';
/**
 * Steps can specify an action to take onFailure.
 * See docs here: https://docs.aws.amazon.com/systems-manager/latest/userguide/automation-actions.html#failProp
 * The supported actions are abort (default), continue, or invoking a specific step.
 * This behavior can be adopted by using the static methods available on OnFailure.
 */
export declare abstract class OnFailure {
    /**
     * Invoke a specific step. Provide the step object to execute for the onFailure action.
     * If you don't have a handle to the step object, use the invokeStepByName function.
     */
    static invokeStep(step: AutomationStep): OnCancel;
    /**
     * Invoke a specific step by the step name for the OnFailure action.
     */
    static invokeStepByName(stepName: string): OnCancel;
    /**
     * To abort execution if a failure occurs during execution of the current step.
     * (This is the default behavior.)
     */
    static abort(): OnCancel;
    /**
     * To continue execution of the subsequent step if a failure occurs during execution of the current step.
     */
    static continue(): OnFailure;
    abstract stepToInvoke(currentStep: AutomationStep): string;
    abstract toSsmValue(): string;
}
/**
 * Steps can specify an action to take onCancel.
 * See docs here: https://docs.aws.amazon.com/systems-manager/latest/userguide/automation-actions.html#canProp
 * The supported actions are abort (default) or invoking a specific step.
 * This behavior can be adopted by using the static methods available on OnCancel.
 */
export declare abstract class OnCancel extends OnFailure {
    /**
     * Invoke a specific step. Provide the step object to execute for the onCancel action.
     * If you don't have a handle to the step object, use the invokeStepByName function.
     * Not all action types may be invoked during cancellation. See documentation referenced on this class.
     */
    static invokeStep(step: AutomationStep): OnCancel;
    /**
     * Invoke a specific step by the step name for the OnCancel action.
     * Not all action types may be invoked during cancellation. See documentation referenced on this class.
     */
    static invokeStepByName(stepName: string): OnCancel;
    /**
     * To abort execution if a cancellation occurs during execution of the current step.
     * (This is the default behavior.)
     */
    static abort(): OnCancel;
}
export declare class StepFallback extends OnCancel {
    static readonly ILLEGAL_FALLBACK_ACTIONS: string[];
    readonly step: AutomationStep;
    constructor(step: AutomationStep, validateStepAction: boolean);
    stepToInvoke(_currentStep: AutomationStep): string;
    toSsmValue(): string;
}
export declare class StepNameFallback extends OnCancel {
    readonly step: string;
    readonly validateStepAction: boolean;
    constructor(step: string, validateStepAction: boolean);
    stepToInvoke(currentStep: AutomationStep): string;
    toSsmValue(): string;
}
export declare class Abort extends OnCancel {
    stepToInvoke(_currentStep: AutomationStep): string;
    toSsmValue(): string;
}
export declare class Continue extends OnFailure {
    stepToInvoke(currentStep: AutomationStep): string;
    toSsmValue(): string;
}
