# -*- coding: utf-8 -*-
from setuptools import setup

package_dir = \
{'': '.'}

packages = \
['jsonbourne', 'jsonbourne.dev']

package_data = \
{'': ['*']}

modules = \
['JSON', 'david_webb']
install_requires = \
['xtyping>=0.5.0']

extras_require = \
{'all': ['pydantic>=1.8.2', 'python-rapidjson>=0.9.1', 'orjson>=3.0.0,<4.0.0'],
 'full': ['pydantic>=1.8.2', 'python-rapidjson>=0.9.1', 'orjson>=3.0.0,<4.0.0'],
 'orjson': ['orjson>=3.0.0,<4.0.0'],
 'pydantic': ['pydantic>=1.8.2'],
 'rapidjson': ['python-rapidjson>=0.9.1']}

setup_kwargs = {
    'name': 'jsonbourne',
    'version': '0.21.4',
    'description': 'EZPZ JSON',
    'long_description': '<a href="https://github.com/dynamic-graphics-inc/dgpy-libs">\n<img align="right" src="https://github.com/dynamic-graphics-inc/dgpy-libs/blob/main/docs/images/dgpy_banner.svg?raw=true" alt="drawing" height="120" width="300"/>\n</a>\n\n# jsonbourne\n\n[![Wheel](https://img.shields.io/pypi/wheel/jsonbourne.svg)](https://img.shields.io/pypi/wheel/jsonbourne.svg)\n[![Version](https://img.shields.io/pypi/v/jsonbourne.svg)](https://img.shields.io/pypi/v/jsonbourne.svg)\n[![py_versions](https://img.shields.io/pypi/pyversions/jsonbourne.svg)](https://img.shields.io/pypi/pyversions/jsonbourne.svg)\n[![Code style: black](https://img.shields.io/badge/code%20style-black-000000.svg)](https://github.com/psf/black)\n\n**Install:** `pip install jsonbourne`\n\n - Python json lib/pkg that makes json feel like the JSON module in javascript/typescript:\n     - `from jsonbourne import JSON; JSON.parse(JSON.stringify({"key": "value"}))`\n     - Automatically uses best json-lib-backend avalible (`orjson`/`python-rapidjson`) ~ can be configured\n - Hybrid dict/class object (`jsonbourne.JsonObj`):\n     - Dot-notation getting/setting (featuring protected attributes!)\n     - All your favorite python dictionary methods (`items`, `keys`, `update`, `values`) and more!\n     - Works with `pydantic` and `attrs`\n - FastAPI:\n     - JSONBOURNEResponse ~ auto use the best\n - No hard dependencies ~ works with python-stdlib-json as well as `orjson` and `python-rapidjson`\n - `jsonbourne.JsonObj` uses list/dict comprehensions (some are recursive) everywhere because \'why not?\' and it is a bit faster\n\n## Usage:\n\n### JSON ~ `from jsonbourne import JSON`\n\n**Importing:**\n\n\n```python\n# Importing JSON:\nfrom jsonbourne import JSON\n# or\nimport JSON\n\n# Importing jsonbourne:\nimport jsonbourne\nimport david_webb  # jsonbourne\'s `True` identity\n```\n\n**JSON basics:**\n\n\n```python\nimport JSON  # Module included with jsonbourne\nstring_stringify = JSON.stringify({"a":1, "b":2, "c":3})  # \'{"a": 1, "b": 2, "c": 3}\'\nstring_dumps = JSON.dumps({"a":1, "b":2, "c":3})  # \'{"a": 1, "b": 2, "c": 3}\'\nstring_dumps\n```\n\n\n\n\n    \'{"a":1,"b":2,"c":3}\'\n\n\n\n#### JSON option kwargs ~ `pretty` & `sort_keys`\n\n**pretty:**\n\n\n```python\nstring_dumps = JSON.stringify({"b":2, "a":1, "c":3}, pretty=True)  # \'{"a": 1, "b": 2, "c": 3}\'\nprint(string_dumps)\n```\n\n    {\n      "b": 2,\n      "a": 1,\n      "c": 3\n    }\n\n\n**sort_keys:**\n\n\n```python\nstring_dumps = JSON.stringify({"b":2, "a":1, "c":3}, pretty=True, sort_keys=True)  # \'{"a": 1, "b": 2, "c": 3}\'\nprint(string_dumps)\n```\n\n    {\n      "a": 1,\n      "b": 2,\n      "c": 3\n    }\n\n\n### JsonObj & JSON\n\n- Python dictionary/object with dot access\n- Protections against setting class/obj attributes\n- Is as javascript-y as possible (keys have to be strings -- ints/floats will be converted to strings)\n- Create a `jsonbourne.JsonObj` with `jsonbourne.JSON`\n- Recursive jsonification\n- Allows for kwarging (`**json_obj`)\n- Works with `pydantic` and `attrs`\n\n#### Make an empty JsonObj\n\nThe following 3 examples all produce the same thing\n\n```python\nfrom jsonbourne import JSON\nj = JSON()  # j => JsonObj(**{})\n# OR\nimport JSON\nj = JSON()  # j => JsonObj(**{})\n# OR\nfrom jsonbourne import JsonObj\nj = JsonObj()  # j => JsonObj(**{})\n```\n\n#### From a dictionary o data\n\n\n```python\nimport datetime\n\ndata = {\n    "key": "value",\n    "list": [1, 2, 3, 4, 5],\n    "dt": datetime.datetime(1970, 1, 1, 0, 0, 0, 1),\n    "sub": {\'b\': 3, \'key\': \'val\', \'a\': 1,},\n    "timedelta": datetime.timedelta(days=2),\n}\n\nJSON(data)\n```\n\n\n\n\n<pre>JsonObj(**{\n    \'dt\': datetime.datetime(1970, 1, 1, 0, 0, 0, 1),\n    \'key\': \'value\',\n    \'list\': [1, 2, 3, 4, 5],\n    \'sub\': {\'a\': 1, \'b\': 3, \'key\': \'val\'},\n    \'timedelta\': datetime.timedelta(days=2)\n})</pre>\n\n\n\n### Dot access\n\n\n```python\nJSON(data).sub.b\n```\n\n\n\n\n    3\n\n\n\n\n```python\nstringified_data = JSON(data).stringify(pretty=True)\nprint(stringified_data)\n```\n\n    {\n      "key": "value",\n      "list": [\n        1,\n        2,\n        3,\n        4,\n        5\n      ],\n      "dt": "1970-01-01T00:00:00.000001",\n      "sub": {\n        "b": 3,\n        "key": "val",\n        "a": 1\n      },\n      "timedelta": 172800.0\n    }\n\n\n\n```python\nparsed_data = JSON(stringified_data)\nparsed_data\n```\n\n\n\n\n<pre>JsonObj(**{\n    \'dt\': \'1970-01-01T00:00:00.000001\',\n    \'key\': \'value\',\n    \'list\': [1, 2, 3, 4, 5],\n    \'sub\': {\'a\': 1, \'b\': 3, \'key\': \'val\'},\n    \'timedelta\': 172800.0\n})</pre>\n\n\n\n\n```python\nlist(parsed_data.keys())\n```\n\n\n\n\n    [\'key\', \'list\', \'dt\', \'sub\', \'timedelta\']\n\n\n\n\n```python\nlist(parsed_data.items())\n```\n\n\n\n\n    [(\'key\', \'value\'),\n     (\'list\', [1, 2, 3, 4, 5]),\n     (\'dt\', \'1970-01-01T00:00:00.000001\'),\n     (\'sub\', JsonObj(**{\'b\': 3, \'key\': \'val\', \'a\': 1})),\n     (\'timedelta\', 172800.0)]\n\n\n\n\n```python\nlist(parsed_data.dot_keys())\n```\n\n\n\n\n    [(\'key\',),\n     (\'list\',),\n     (\'dt\',),\n     (\'sub\', \'b\'),\n     (\'sub\', \'key\'),\n     (\'sub\', \'a\'),\n     (\'timedelta\',)]\n\n\n\n\n```python\nlist(parsed_data.dot_items())\n```\n\n\n\n\n    [((\'key\',), \'value\'),\n     ((\'list\',), [1, 2, 3, 4, 5]),\n     ((\'dt\',), \'1970-01-01T00:00:00.000001\'),\n     ((\'sub\', \'b\'), 3),\n     ((\'sub\', \'key\'), \'val\'),\n     ((\'sub\', \'a\'), 1),\n     ((\'timedelta\',), 172800.0)]\n\n\n\n\n```python\nparsed_data[(\'sub\', \'key\')]\n```\n\n\n\n\n    \'val\'\n\n\n\n\n```python\nparsed_data.dot_lookup(\'sub.key\')\n```\n\n\n\n\n    \'val\'\n\n\n\n\n```python\n{**parsed_data}\n```\n\n\n\n\n    {\'key\': \'value\',\n     \'list\': [1, 2, 3, 4, 5],\n     \'dt\': \'1970-01-01T00:00:00.000001\',\n     \'sub\': JsonObj(**{\'b\': 3, \'key\': \'val\', \'a\': 1}),\n     \'timedelta\': 172800.0}\n\n\n\n\n```python\n# fully eject\nparsed_data.eject()\n```\n\n\n\n\n    {\'key\': \'value\',\n     \'list\': [1, 2, 3, 4, 5],\n     \'dt\': \'1970-01-01T00:00:00.000001\',\n     \'sub\': {\'b\': 3, \'key\': \'val\', \'a\': 1},\n     \'timedelta\': 172800.0}\n\n\n\n#### Protected keys\n\n`jsonbourne.JsonObj` protects against setting attributes like `\'items\'` through dot-notation.\n\n\n```python\nfrom jsonbourne import JSON\nj = JSON()\nj.key = \'value\'\ntry:  # CANNOT set \'items\' using dot-access\n\tj.items = [1, 2, 3, 4]\nexcept ValueError:\n\tpass\n# CAN set \'items\' through key/item access\nj[\'items\'] = [1, 2, 3, 4]\nprint(j.__dict__)\nprint(j)\nj_items = j.items\nprint(\'items\', j_items)\n# Getting \'items\' through dot-access returns the `items()` method\nassert j.items != [1, 2, 3, 4]\n# Getting \'items\' with key-access returns the stored value\nassert j[\'items\'] == [1, 2, 3, 4]\n```\n\n    {\'_data\': {\'key\': \'value\', \'items\': [1, 2, 3, 4]}}\n    JsonObj(**{\n        \'items\': [1, 2, 3, 4], \'key\': \'value\'\n    })\n    items <bound method JsonObj.items of JsonObj(**{\'key\': \'value\', \'items\': [1, 2, 3, 4]})>\n\n\n### pydantic & jsonbourne\n\n - `from jsonbourne.pydantic import JsonBaseModel`\n - Allows for aliases when getting/setting attribute(s)\n - Supports `__post_init__` (like dataclasses)\n\n#### Basic usage:\n\n\n```python\nfrom jsonbourne import JsonObj\nfrom jsonbourne.pydantic import JsonBaseModel\n\nclass JsonSubObj(JsonBaseModel):\n    herm: int\n\n    def to_dict(self):\n        return self.dict()\n\n    def to_json(self, *args, **kwargs):\n        return self.json()\n\n    @classmethod\n    def from_json(cls, json_string: str):\n        return JsonSubObj(json.loads(json_string))\n\nclass JsonObjModel(JsonBaseModel):\n    a: int\n    b: int\n    c: str\n    d: JsonObj\n    e: JsonSubObj\n\n    #\n    @property\n    def a_property(self) -> str:\n        return "prop_value"\n\n    def to_json(self, *args, **kwargs):\n        return self.json()\n\n    @classmethod\n    def from_json(cls, json_string: str):\n        return cls(**json.loads(json_string))\n\nobj = JsonObjModel(\n    **{"a": 1, "b": 2, "c": "herm", "d": {"nested": "nestedval"}, "e": {"herm": 2}}\n)\nobj\n```\n\n\n\n\n<pre>JsonObjModel(**{\n     \'a\': 1, \'b\': 2, \'c\': \'herm\', \'d\': JsonObj(**{\'nested\': \'nestedval\'}), \'e\': {\'herm\': 2}\n})</pre>\n\n\n\n\n```python\n# respects properties (which I don\'t think pydantic does(currently))\nobj.a_property\n```\n\n\n\n\n    \'prop_value\'\n\n\n\n___\n\n## JSON backend/lib\n\n**`jsonbourne` finds the best json-lib (python-rapidjson/orjson) it can!** On import `jsonbourne` gets to\nwork spying on your python env. `jsonbourne`, the most highly qualified json-CIA-agent, will import the best\npython-json library it can find; if `jsonbourne`\'s cover is blown (meaning: the\nonly json library found is the python stdlib json), `jsonbourne` will fallback\nto\nthe python stdlib json.\n\n`jsonbourne` will look for the following json-packages in the following order:\n\n  1) `rapidjson`\n  2) `orjson`\n\n### Custom lib preferences\n\n\n```python\nfrom jsonbourne import import_json\njson = import_json(("rapidjson", "orjson"))  # prefer rapidjson over orjson\nstring = json.dumps({"a":1, "b":2, "c":3})\nprint(json)\nprint(string)\n```\n\n    <class \'jsonbourne.json.RAPIDJSON\'>\n    {"a":1,"b":2,"c":3}\n\n\n### Installing better JSON lib:\n\n#### `orjson`\n\n- `pip install orjson` [pip]\n\n#### `rapidjson`/`python-rapidjson`\n\n- `pip install python-rapidjson` [pip]\n- `conda install -c anaconda python-rapidjson` [conda anaconda/defaults]\n- `conda install -c conda-forge python-rapidjson` [conda-forge]\n\n',
    'author': 'jesse',
    'author_email': 'jesse@dgi.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/dynamic-graphics-inc/dgpy-libs/tree/main/libs/jsonbourne',
    'package_dir': package_dir,
    'packages': packages,
    'package_data': package_data,
    'py_modules': modules,
    'install_requires': install_requires,
    'extras_require': extras_require,
    'python_requires': '>=3.7,<4.0',
}


setup(**setup_kwargs)
