'''
Created on 2022-04-30

@author: wf
'''
#from lodstorage.trulytabular import WikidataProperty
#from lodstorage.lod import LOD
#from spreadsheet.wikidata import Wikidata
from spreadsheet.googlesheet import GoogleSheet
import pprint
import collections

class WikibaseQuery(object):
    '''
    a Query for Wikibase
    '''

    def __init__(self,entity:str,debug:bool=False):
        '''
        Constructor
        
        Args:
            entity(str): the entity this query represents
            debug(bool): if True switch on debugging
        '''
        self.debug=debug
        self.entity=entity
        self.propertiesByName={}
        self.propertiesById={}
        self.propertiesByVarname={}
        
    def addPropertyFromDescriptionRow(self,row):
        '''
        add a property from the given row
        
        Args:
            row(dict): the row to add
        '''
        propName=row['PropertyName']
        propId=row['PropertyId']
        # properties might contain blank - replace for SPARQL variable names
        propVarname=propName.replace(" ","_")
        row['PropVarname']=propVarname
        # set the two lookups
        self.propertiesByName[propName]=row
        self.propertiesById[propId]=row
        self.propertiesByVarname[propVarname]=row
        
    def getColumnTypeAndVarname(self,propName):
        '''
        get a signature tuple consisting of columnName, propertType and SPARQL variable Name for the given property Name
        
        Args:
            propName(str): the name of the property
            
        Returns:
            column,propType,varName tupel
        '''
        column=self.propertiesByName[propName]["Column"]
        propType=self.propertiesByName[propName]["Type"]
        varName=self.propertiesByName[propName]["PropVarname"]
        return column,propType,varName
        
        
    def inFilter(self,values,propName:str="short_name",lang:str="en"):
        '''
        create a SPARQL IN filter clause
        
        Args:
            values(list): the list of values to filter for
            propName(str): the property name to filter with
            lang(str): the language to apply
        '''
        filterClause=f"\n  FILTER(?{propName} IN("
        delim=""
        for value in values:
            filterClause+=f"{delim}\n    '{value}'@{lang}"
            delim=","
        filterClause+="\n  ))."
        return filterClause
    
    def getValuesClause(self,values,propName:str="short_name",lang:str=None,ignoreEmpty:bool=True):
        '''
        create a SPARQL Values clause
        
        Args:
            values(list): the list of values to create values for
            propName(str): the property name to assign the values for
            ignoreEmpty(bool): ignore empty values if True
        Returns:
            str: the SPARQL values clause
        '''
        valuesClause=f"\n  VALUES(?{propName}) {{"
        if lang is not None:
            lang=f'@{lang}'
        else:
            lang=''
        for value in values:
            if value or not ignoreEmpty:
                valuesClause+=f"\n  ( '{value}'{lang} )"
        valuesClause+="\n  }."
        return valuesClause
        
    def asSparql(self,filterClause:str=None,orderClause:str=None,pk:str=None,lang:str="en"):
        '''
        get the sparqlQuery for this query optionally applying a filterClause
        
        Args:
            filterClause(str): a filter to be applied (if any)
            orderClause(str): an orderClause to be applied (if any)
            pk(str): primaryKey (if any)
            lang(str): the language to be used for labels
        '''
        sparql=f"""# 
# get {self.entity} records 
#  
PREFIX pq: <http://www.wikidata.org/prop/qualifier/>
PREFIX p: <http://www.wikidata.org/prop/>
PREFIX schema: <http://schema.org/>
PREFIX wd: <http://www.wikidata.org/entity/>
PREFIX wdt: <http://www.wikidata.org/prop/direct/>
PREFIX wikibase: <http://wikiba.se/ontology#>
PREFIX rdfs: <http://www.w3.org/2000/01/rdf-schema#>
SELECT ?item ?itemLabel ?itemDescription
"""
        for propVarname,row in self.propertiesByVarname.items():
            propValue=row["Value"]
            propType=row["Type"]
            # items will automatically fetch labels
            propLabel=f" ?{propVarname}Label" if not propType else ""
            if not propValue:
                sparql+=f"\n  ?{propVarname}{propLabel}"
        sparql+="""\nWHERE {
  ?item rdfs:label ?itemLabel.
  FILTER(LANG(?itemLabel) = "%s")
  OPTIONAL { 
    ?item schema:description ?itemDescription.
    FILTER(LANG(?itemDescription) = "%s")
  }
""" % (lang,lang)
        for propVarname,row in self.propertiesByVarname.items():
            propName=row["PropertyName"]
            propValue=row["Value"]
            propId=row["PropertyId"]
            propType=row["Type"]
            if propValue:
                sparql+=f"\n  ?item wdt:{propId} wd:{propValue}."
            else:
                # primary keys are not optional
                optional=pk is None or not propName==pk
                if optional:
                    sparql+=f"\n  OPTIONAL {{"
                sparql+=f"\n    ?item wdt:{propId} ?{propVarname}."
                if not propType:
                    sparql+=f"\n    ?{propVarname} rdfs:label ?{propVarname}Label."
                    sparql+=f"""\n    FILTER(LANG(?{propVarname}Label) = "{lang}")"""
                if optional:
                    sparql+=f"\n  }}"
        if filterClause is not None:
                sparql+=f"\n{filterClause}"        
        sparql+="\n}"
        if orderClause is not None:
            sparql+=f"\n{orderClause}"
        return sparql
            

    @classmethod
    def ofGoogleSheet(cls,url:str,sheetName:str="Wikidata",debug:bool=False)->dict:
        '''
        create a dict of wikibaseQueries from the given google sheets row descriptions
        
        Args:
            url(str): the url of the sheet
            sheetName(str): the name of the sheet with the description
            debug(bool): if True switch on debugging
        '''
        gs=GoogleSheet(url)
        gs.open([sheetName])
        entityMapRows=gs.asListOfDicts(sheetName)
        return WikibaseQuery.ofMapRows(entityMapRows,debug=debug)
        
    @classmethod
    def ofMapRows(cls,entityMapRows:list,debug:bool=False):
        '''
        create a dict of wikibaseQueries from the given entityMap list of dicts
        
        Args:
            entityMapRows(list): a list of dict with row descriptions
            debug(bool): if True switch on debugging
        '''
        queries={}
        entityMapDict={}
        for row in entityMapRows:
            if "Entity" in row:
                entity=row["Entity"]
                if not entity in entityMapDict:
                    entityMapDict[entity]={}
                entityRows=entityMapDict[entity]
                if "PropertyName" in row:
                    propertyName=row["PropertyName"]
                    entityRows[propertyName]=row    
        if debug:
            pprint.pprint(entityMapDict)
        for entity in entityMapDict:
            wbQuery=WikibaseQuery.ofEntityMap(entity,entityMapDict[entity])
            queries[entity]=wbQuery
        return queries
    
    @classmethod
    def ofEntityMap(cls,entity:str,entityMap:dict):
        '''
        create a WikibaseQuery for the given entity and entityMap
        
        Args:
            entity(str): the entity name
            entityMap(dict): the entity property descriptions
        Returns:
            WikibaseQuery
        '''
        wbQuery=WikibaseQuery(entity)
        for row in entityMap.values():
            wbQuery.addPropertyFromDescriptionRow(row)
        return wbQuery
        
        
    