import requests
import numpy as np
from opentps.core.data.plan._rtPlan import RTPlan
from opentps.core.data.plan._scanAlgoPlan import ScanAlgoPlan
from opentps.core.io.serializedObjectIO import saveRTPlan
from opentps.core.processing.planDeliverySimulation.deliverySimulationConfig import DeliverySimulationConfig

class BDT:
    def __init__(self, plan: RTPlan):
        self.plan = plan
        config = DeliverySimulationConfig()
        self.gantry = config.gantry
        self.url = config.gateway


    def getPBSTimings(self, sort_spots="true"):
        """
        Add timings for each spot in the plan:
        INPUT:
            jsonFilesList: list of json files (one for each beam) generated by createJSONFiles
        OUTPUT:
            RTPlan with timings
        """
        plan = self.plan.copy()
        gantry_angles = [] if plan._beams==[] else [beam.gantryAngle for beam in plan._beams]
        for index,beam in enumerate(plan._beams):
            data = ScanAlgoPlan(plan, self.gantry, index, sort_spots=sort_spots)
            gantry_angle = float(data.gantryangle) if self.gantry=="PPlus" else float(data.gantryAngle)
            scanAlgo = requests.post(self.url,json=data.__dict__).json()
            if 'cause' in scanAlgo:
               print("!!!!!!!!! ScanAlgo ERROR in beam !!!!!!! ", index)
               print('\n')
               print(scanAlgo['cause'])
               print('\n')
            else:
                if self.gantry == "PPlus":
                    index_beam = np.where(np.array(gantry_angles)==gantry_angle)[0][0]
                    plan = self.parsePPlusResponse(plan, scanAlgo, index_beam)
                elif self.gantry == "POne":
                    index_beam = np.where(np.array(gantry_angles)==gantry_angle)[0][0]
                    plan = self.parsePOneResponse(plan, scanAlgo, index_beam)
                else:
                    raise Exception(f'Unknown gantry type {self.gantry}')

        return plan


    def parsePPlusResponse(self, plan, scanAlgo, index_beam):
        """
        Parse response from ScanAlgo for PPlus gantry + reorder spots according to spot timings
        INPUTS:
            plan: RTPlan
            scanAlgo: output from scanAlgo
            index_beam: index number of the beam in plan._beams
        OUTPUT:
            RTPlan with timings
        """
        assert len(plan._beams[index_beam]._layers) == len(scanAlgo['layer'])
        for l in range(len(plan._beams[index_beam]._layers)):
            # identify current spot in layer
            original_layer = plan._beams[index_beam]._layers[l]
            SA_layer = scanAlgo['layer'][l]['spot']
            if len(SA_layer) != len(original_layer._x):
                N = len(SA_layer)
                conversion_coeff = self.get_charge_to_MU_conversion(original_layer, scanAlgo['layer'][l])
                original_layer._x = np.array([])
                original_layer._y = np.array([])
                original_layer._mu = np.array([])
                original_layer._timings = np.array([])
                SA_x = [SA_layer[i]['clinicalx'] for i in range(N)]
                SA_y = [SA_layer[i]['clinicaly'] for i in range(N)]
                SA_w = [SA_layer[i]['charge'] * conversion_coeff for i in range(N)]
                SA_t = [SA_layer[i]['start'] for i in range(N)]
                original_layer.appendSpot(SA_x, SA_y, SA_w, SA_t)
            else:
                original_layer._timings = np.zeros(len(SA_layer))
                for s in range(len(original_layer._x)):
                    index_spot_scanAlgo = self.find_spot_index_json(scanAlgo['layer'][l],
                        original_layer._x[s],
                        original_layer._y[s])
                    start_time = float(SA_layer[index_spot_scanAlgo]['start']) / 1000
                    original_layer._timings[s] = start_time
                # Reorder spots according to spot timings
                order = np.argsort(original_layer._timings)
                original_layer.reorderSpots(order)
        return plan



    def parsePOneResponse(self, plan, scanAlgo, index_beam):
        """
        Parse response from ScanAlgo for POne gantry + reorder spots according to spot timings
        Since there is an intrinsec rescanning for the POne, the plan is modified accordingly, i.e.
            - Spots are added in the plan corresponding to the rescanning
            - SpotMU are weighted according to the dose delivered in each burst
        Timings are computed with postprocessing (i.e. layer time, burst time accumulation)
        INPUTS:
            plan: RTPlan
            scanAlgo: output from scanAlgo
            index_beam: index number of the beam in plan._beams
        OUTPUT:
            RTPlan with timings
        """
        assert len(plan._beams[index_beam]._layers) == len(scanAlgo['layers'])
        burst_switching_time = 150
        cumul_layer_time = 0
        for l in range(len(plan._beams[index_beam]._layers)):
            assert 'bursts' in scanAlgo['layers'][l]
            original_layer = plan._beams[index_beam]._layers[l]
            SA_burst = scanAlgo['layers'][l]['bursts']
            cumul_layer_time += float(scanAlgo['layers'][l]["switchingTime"])

            conversion_coeff = self.get_charge_to_MU_conversion(original_layer, scanAlgo['layers'][l])
            original_layer._x = np.array([])
            original_layer._y = np.array([])
            original_layer._mu = np.array([])
            original_layer._timings = np.array([])

            cumul_burst_time = 0.
            for burst in SA_burst:
                N = len(burst['spots'])
                SA_x = np.array([burst['spots'][i]['clinicalX'] for i in range(N)])
                SA_y = np.array([burst['spots'][i]['clinicalY'] for i in range(N)])
                SA_w = np.array([burst['spots'][i]['targetCharge'] * conversion_coeff for i in range(N)])
                SA_t = np.array([burst['spots'][i]['startTime'] for i in range(N)])
                SA_t = SA_t + cumul_burst_time
                original_layer.appendSpot(SA_x, SA_y, SA_w, SA_t)
                cumul_burst_time += burst_switching_time + burst['spots'][-1]['startTime'] + burst['spots'][-1]['duration']

        return plan


    def find_spot_index_json(self, json_arr, pos_x, pos_y, return_first=True):
        """
        Find index of spot corresposding to position (pos_x,pos_y) in JSON file
        """
        if self.gantry=="PPlus":
            indices = []
            pos_x_name = 'clinicalx'
            pos_y_name = 'clinicaly'
            for i in range(len(json_arr['spot'])):
                if np.isclose(float(json_arr['spot'][i][pos_x_name]), float(pos_x)) and np.isclose(float(json_arr['spot'][i][pos_y_name]), float(pos_y)):
                    indices.append(i)
        else: # self.gantry=="POne"
            indices = {}
            pos_x_name = 'clinicalX'
            pos_y_name = 'clinicalY'
            for b, burst in enumerate(json_arr['bursts']):
                indices[b] = []
                for s, spot in enumerate(burst['spots']):
                    if np.isclose(float(spot[pos_x_name]), float(pos_x)) and np.isclose(float(spot[pos_y_name]), float(pos_y)):
                        indices[b].append(s)

        if len(indices)==0:
            return None
        elif len(indices)==1 or return_first:
            return indices[0]
        else:
            return indices


    def get_charge_to_MU_conversion(self, original_layer, SA_layer):
        if self.gantry == 'PPlus':
            index_spot_scanAlgo = self.find_spot_index_json(SA_layer,
                    original_layer._x[0],
                    original_layer._y[0], return_first=False)
            factor = 1
            if isinstance(index_spot_scanAlgo, list):
                factor = len(index_spot_scanAlgo)
                index_spot_scanAlgo = index_spot_scanAlgo[0]
            conversion_coeff = original_layer._mu[0] / (SA_layer['spot'][index_spot_scanAlgo]['charge'] * factor)
        elif self.gantry == 'POne':
            index_spot_scanAlgo = self.find_spot_index_json(SA_layer,
                    original_layer._x[0],
                    original_layer._y[0], return_first=False)
            # total_charge = sum([item for sublist in list(index_spot_scanAlgo.values()) for item in sublist])
            total_charge = sum([SA_layer['bursts'][b]['spots'][s]['targetCharge'] for b,spot_ind in index_spot_scanAlgo.items() for s in spot_ind])
            conversion_coeff = original_layer._mu[0] / total_charge
        else:
            raise NotImplementedError(f'{self.Gantry} not implemented')

        return conversion_coeff

    def get_timings_and_save_plan(self, output_path):
        plan_with_timings = self.getPBSTimings(sort_spots="true")
        saveRTPlan(plan_with_timings, output_path)