# 美股的规则样例
import pendulum
from typing import Union


class _Rule():
    # [交易规则] -----------------------------------------------------------
    # T = 0 -> T0交易
    # T = 1 -> T1交易 ...
    T = 0

    # [时区] --------------------------------------------------------------
    TIMEZONE: str

    # [账户余额] -----------------------------------------------------------
    ACCOUNT_MONEY: Union[int, float]

    # [购买金额] -----------------------------------------------------------
    # BUY_MONEY         单次开仓的默认金额
    BUY_MONEY: Union[int, float, None]
    # BUY_MONEY_RATE    单次开仓的默认金额占比，开仓金额为：BUY_MONEY_RATE * ACCOUNT_MONEY
    BUY_MONEY_RATE: Union[int, float, None]
    # BUY_MONEY和BUY_MONEY_RATE同时出现，以BUY_MONEY_RATE为主

    # [止盈止损] -----------------------------------------------------------
    # 多单止盈，挂单价格
    LONG_TP_RATE: Union[int, float, None]
    # 多单止损，出发止损价格，以市价单平仓
    LONG_SL_RATE: Union[int, float, None]
    # 空单止盈，挂单价格
    SHORT_TP_RATE: Union[int, float, None]
    # 空单止损，出发止损价格，以市价单平仓
    SHORT_SL_RATE: Union[int, float, None]

    # [杠杆] ---------------------------------------------------------------
    # 多单杠杆
    LONG_LEVER: Union[int, float]
    # 空单杠杆
    SHORT_LEVER: Union[int, float]

    # [仓位] ---------------------------------------------------------------
    # 多单仓位
    LONG_POSITION: int
    # 空单仓位
    SHORT_POSITION: int
    # 总仓位
    POSITION: int

    # [交易时间] ------------------------------------------------------------
    # 运行购买的时间
    BUY_PERIODS: list
    # 运行卖出的时间
    SELL_PERIODS: list
    # 运行抛出的时间
    THROW_PERIODS: list

    # [最长订单时间（分钟）] ---------------------------------------------------
    MAX_HOLD_ORDER_MINUTE: Union[int, float, None]

    # [手续费率] -------------------------------------------------------------
    # 买入的手续费
    BUY_COMMISSION_RATE: Union[int, float]
    # 卖出的手续费
    SELL_COMMISSION_RATE: Union[int, float]

    # [开盘与收盘时刻] ---------------------------------------------------------
    # 开盘的时刻
    OPEN_TIME: str
    # 收盘的时刻
    CLOSE_TIME: str

    def __init__(self):
        for i in range(len(self.BUY_PERIODS)):
            if not self.BUY_PERIODS[i]:
                continue
            for j in range(2):
                self.BUY_PERIODS[i][j] = pendulum.from_format(self.BUY_PERIODS[i][j], 'HH:mm:ss').time()

        for i in range(len(self.SELL_PERIODS)):
            if not self.SELL_PERIODS[i]:
                continue
            for j in range(2):
                self.SELL_PERIODS[i][j] = pendulum.from_format(self.SELL_PERIODS[i][j], 'HH:mm:ss').time()

        for i in range(len(self.THROW_PERIODS)):
            if not self.THROW_PERIODS[i]:
                continue
            for j in range(2):
                self.THROW_PERIODS[i][j] = pendulum.from_format(self.THROW_PERIODS[i][j], 'HH:mm:ss').time()


class RuleAmericaStock(_Rule):
    # [交易规则] -----------------------------------------------------------
    # T = 0 -> T0交易
    # T = 1 -> T1交易 ...
    T = 0
    # [时区] --------------------------------------------------------------
    TIMEZONE = 'America/New_York'  # 美股时间
    # [账户余额] -----------------------------------------------------------
    ACCOUNT_MONEY = 1000
    # [购买金额] -----------------------------------------------------------
    BUY_MONEY_RATE = None  # 单次开仓的默认金额占比
    BUY_MONEY = 100  # 单次开仓的默认金额
    # [止盈止损] -----------------------------------------------------------
    LONG_TP_RATE = None  # 多单止盈，挂单价格
    LONG_SL_RATE = None  # 多单止损，出发止损价格，以市价单平仓
    SHORT_TP_RATE = None  # 空单止盈，挂单价格
    SHORT_SL_RATE = None  # 空单止损，出发止损价格，以市价单平仓
    # [杠杆] ---------------------------------------------------------------
    LONG_LEVER = 1  # 多单杠杆
    SHORT_LEVER = 1  # 空单杠杆
    # [仓位] ---------------------------------------------------------------
    LONG_POSITION = 100  # 多单仓位
    SHORT_POSITION = 100  # 空单仓位
    POSITION = 100  # 总仓位
    # [最长订单时间（分钟）] ---------------------------------------------------
    BUY_PERIODS = [
        ['09:30:00', '16:00:00']  # 可以买入时间段为09:30:00 ~ 16:00:00
    ]
    # 运行卖出的时间
    SELL_PERIODS = [
        ['09:30:00', '16:00:00']  # 可以卖出时间段为09:30:00 ~ 16:00:00
    ]
    # 运行抛出的时间
    THROW_PERIODS = [[]]
    # [最长订单时间（分钟）] ---------------------------------------------------
    MAX_HOLD_ORDER_MINUTE = None
    # [手续费率] -------------------------------------------------------------
    BUY_COMMISSION_RATE = 0  # 买入的手续费
    SELL_COMMISSION_RATE = 0  # 卖出的手续费
    # [开盘与收盘时刻] ---------------------------------------------------------
    OPEN_TIME = '09:30:00'  # 开盘的时刻
    CLOSE_TIME = '16:00:00'  # 收盘的时刻


class RuleChinaStock():
    # [交易规则] -----------------------------------------------------------
    # T = 0 -> T0交易
    # T = 1 -> T1交易 ...
    T = 0
    # [时区] --------------------------------------------------------------
    TIMEZONE = 'Asia/Shanghai'  # 中国时间
    # [账户余额] -----------------------------------------------------------
    ACCOUNT_MONEY = 1000
    # [购买金额] -----------------------------------------------------------
    BUY_MONEY_RATE = None  # 单次开仓的默认金额占比
    BUY_MONEY = 100  # 单次开仓的默认金额
    # [止盈止损] -----------------------------------------------------------
    LONG_TP_RATE = None  # 多单止盈，挂单价格
    LONG_SL_RATE = None  # 多单止损，出发止损价格，以市价单平仓
    SHORT_TP_RATE = None  # 空单止盈，挂单价格
    SHORT_SL_RATE = None  # 空单止损，出发止损价格，以市价单平仓
    # [杠杆] ---------------------------------------------------------------
    LONG_LEVER = 1  # 多单杠杆
    SHORT_LEVER = 1  # 空单杠杆
    # [仓位] ---------------------------------------------------------------
    LONG_POSITION = 100  # 多单仓位
    SHORT_POSITION = 100  # 空单仓位
    POSITION = 100  # 总仓位
    # [最长订单时间（分钟）] ---------------------------------------------------
    BUY_PERIODS = [
        ['09:30:00', '11:30:00'],  # 可以买入时间段为09:30:00 ~ 11:30:00
        ['13:00:00', '15:00:00'],  # 可以买入时间段为13:00:00 ~ 15:00:00
    ]
    # 运行卖出的时间
    SELL_PERIODS = [
        ['09:30:00', '11:30:00'],  # 可以卖出时间段为09:30:00 ~ 11:30:00
        ['13:00:00', '15:00:00'],  # 可以卖出时间段为13:00:00 ~ 15:00:00
    ]
    # 运行抛出的时间
    THROW_PERIODS = [[]]
    # [最长订单时间（分钟）] ---------------------------------------------------
    MAX_HOLD_ORDER_MINUTE = None
    # [手续费率] -------------------------------------------------------------
    BUY_COMMISSION_RATE = 0.0001  # 买入的手续费
    SELL_COMMISSION_RATE = 0.001  # 卖出的手续费
    # [开盘与收盘时刻] ---------------------------------------------------------
    OPEN_TIME = '09:30:00'  # 开盘的时刻
    CLOSE_TIME = '15:00:00'  # 收盘的时刻


class RuleOkx():
    # [交易规则] -----------------------------------------------------------
    # T = 0 -> T0交易
    # T = 1 -> T1交易 ...
    T = 0
    # [时区] --------------------------------------------------------------
    TIMEZONE = 'Asia/Shanghai'  # 中国时间
    # [账户余额] -----------------------------------------------------------
    ACCOUNT_MONEY = 1000
    # [购买金额] -----------------------------------------------------------
    BUY_MONEY_RATE = None  # 单次开仓的默认金额占比
    BUY_MONEY = 100  # 单次开仓的默认金额
    # [止盈止损] -----------------------------------------------------------
    LONG_TP_RATE = None  # 多单止盈，挂单价格
    LONG_SL_RATE = None  # 多单止损，出发止损价格，以市价单平仓
    SHORT_TP_RATE = None  # 空单止盈，挂单价格
    SHORT_SL_RATE = None  # 空单止损，出发止损价格，以市价单平仓
    # [杠杆] ---------------------------------------------------------------
    LONG_LEVER = 1  # 多单杠杆
    SHORT_LEVER = 1  # 空单杠杆
    # [仓位] ---------------------------------------------------------------
    LONG_POSITION = 100  # 多单仓位
    SHORT_POSITION = 100  # 空单仓位
    POSITION = 100  # 总仓位
    # [最长订单时间（分钟）] ---------------------------------------------------
    BUY_PERIODS = [
        ['00:00:00', '23:59:59']  # 可以买入时间段为00:00:00 ~ 23:59:59
    ]
    # 运行卖出的时间
    SELL_PERIODS = [
        ['00:00:00', '23:59:59']  # 可以卖出时间段为00:00:00 ~ 23:59:59
    ]
    # 运行抛出的时间
    THROW_PERIODS = [[]]
    # [最长订单时间（分钟）] ---------------------------------------------------
    MAX_HOLD_ORDER_MINUTE = None
    # [手续费率] -------------------------------------------------------------
    BUY_COMMISSION_RATE = 0.00035  # 买入的手续费
    SELL_COMMISSION_RATE = 0.00035  # 卖出的手续费
    # [开盘与收盘时刻] ---------------------------------------------------------
    OPEN_TIME = '00:00:00'  # 开盘的时刻
    CLOSE_TIME = '00:00:00'  # 收盘的时刻


class RuleBinance():
    # [交易规则] -----------------------------------------------------------
    # T = 0 -> T0交易
    # T = 1 -> T1交易 ...
    T = 0
    # [时区] --------------------------------------------------------------
    TIMEZONE = 'America/New_York'  # 美股时间
    # [账户余额] -----------------------------------------------------------
    ACCOUNT_MONEY = 1000
    # [购买金额] -----------------------------------------------------------
    BUY_MONEY_RATE = None  # 单次开仓的默认金额占比
    BUY_MONEY = 100  # 单次开仓的默认金额
    # [止盈止损] -----------------------------------------------------------
    LONG_TP_RATE = None  # 多单止盈，挂单价格
    LONG_SL_RATE = None  # 多单止损，出发止损价格，以市价单平仓
    SHORT_TP_RATE = None  # 空单止盈，挂单价格
    SHORT_SL_RATE = None  # 空单止损，出发止损价格，以市价单平仓
    # [杠杆] ---------------------------------------------------------------
    LONG_LEVER = 1  # 多单杠杆
    SHORT_LEVER = 1  # 空单杠杆
    # [仓位] ---------------------------------------------------------------
    LONG_POSITION = 100  # 多单仓位
    SHORT_POSITION = 100  # 空单仓位
    POSITION = 100  # 总仓位
    # [最长订单时间（分钟）] ---------------------------------------------------
    BUY_PERIODS = [
        ['00:00:00', '23:59:59']  # 可以买入时间段为00:00:00 ~ 23:59:59
    ]
    # 运行卖出的时间
    SELL_PERIODS = [
        ['00:00:00', '23:59:59']  # 可以卖出时间段为00:00:00 ~ 23:59:59
    ]
    # 运行抛出的时间
    THROW_PERIODS = [[]]
    # [最长订单时间（分钟）] ---------------------------------------------------
    MAX_HOLD_ORDER_MINUTE = None
    # [手续费率] -------------------------------------------------------------
    BUY_COMMISSION_RATE = 0.00035  # 买入的手续费
    SELL_COMMISSION_RATE = 0.00035  # 卖出的手续费
    # [开盘与收盘时刻] ---------------------------------------------------------
    OPEN_TIME = '00:00:00'  # 开盘的时刻
    CLOSE_TIME = '00:00:00'  # 收盘的时刻

class RuleHuobi():
    # [交易规则] -----------------------------------------------------------
    # T = 0 -> T0交易
    # T = 1 -> T1交易 ...
    T = 0
    # [时区] --------------------------------------------------------------
    TIMEZONE = 'Asia/Shanghai'  # 中国时间
    # [账户余额] -----------------------------------------------------------
    ACCOUNT_MONEY = 1000
    # [购买金额] -----------------------------------------------------------
    BUY_MONEY_RATE = None  # 单次开仓的默认金额占比
    BUY_MONEY = 100  # 单次开仓的默认金额
    # [止盈止损] -----------------------------------------------------------
    LONG_TP_RATE = None  # 多单止盈，挂单价格
    LONG_SL_RATE = None  # 多单止损，出发止损价格，以市价单平仓
    SHORT_TP_RATE = None  # 空单止盈，挂单价格
    SHORT_SL_RATE = None  # 空单止损，出发止损价格，以市价单平仓
    # [杠杆] ---------------------------------------------------------------
    LONG_LEVER = 1  # 多单杠杆
    SHORT_LEVER = 1  # 空单杠杆
    # [仓位] ---------------------------------------------------------------
    LONG_POSITION = 100  # 多单仓位
    SHORT_POSITION = 100  # 空单仓位
    POSITION = 100  # 总仓位
    # [最长订单时间（分钟）] ---------------------------------------------------
    BUY_PERIODS = [
        ['00:00:00', '23:59:59']  # 可以买入时间段为00:00:00 ~ 23:59:59
    ]
    # 运行卖出的时间
    SELL_PERIODS = [
        ['00:00:00', '23:59:59']  # 可以卖出时间段为00:00:00 ~ 23:59:59
    ]
    # 运行抛出的时间
    THROW_PERIODS = [[]]
    # [最长订单时间（分钟）] ---------------------------------------------------
    MAX_HOLD_ORDER_MINUTE = None
    # [手续费率] -------------------------------------------------------------
    BUY_COMMISSION_RATE = 0.00035  # 买入的手续费
    SELL_COMMISSION_RATE = 0.00035  # 卖出的手续费
    # [开盘与收盘时刻] ---------------------------------------------------------
    OPEN_TIME = '00:00:00'  # 开盘的时刻
    CLOSE_TIME = '00:00:00'  # 收盘的时刻

if __name__ == '__main__':
    pass
