# AUTOGENERATED! DO NOT EDIT! File to edit: nbs/05_losses.ipynb (unless otherwise specified).

__all__ = ['LOSSES', 'FastaiLoss', 'WeightedLoss', 'JointLoss', 'Poly1CrossEntropyLoss', 'get_loss']

# Cell
import torch
import torch.nn as nn
import torch.nn.functional as F
from torch.nn.modules.loss import _Loss
import fastai
from fastai.torch_core import TensorBase
import segmentation_models_pytorch as smp
from .utils import import_package

# Cell
LOSSES = ['CrossEntropyLoss', 'DiceLoss', 'SoftCrossEntropyLoss', 'CrossEntropyDiceLoss',  'JaccardLoss', 'FocalLoss', 'LovaszLoss', 'TverskyLoss', 'Poly1CrossEntropyLoss']

# Cell
class FastaiLoss(_Loss):
    'Wrapper class around loss function for handling different tensor types.'
    def __init__(self, loss, axis=1):
        super().__init__()
        self.loss = loss
        self.axis=axis

    #def _contiguous(self, x): return TensorBase(x.contiguous())
    def _contiguous(self,x):
        return TensorBase(x.contiguous()) if isinstance(x,torch.Tensor) else x

    def forward(self, *input):
        #input = map(self._contiguous, input)
        input = [self._contiguous(x) for x in input]
        return self.loss(*input) #

# Cell
# from https://github.com/BloodAxe/pytorch-toolbelt/blob/develop/pytorch_toolbelt/losses/joint_loss.py
class WeightedLoss(_Loss):
    '''
    Wrapper class around loss function that applies weighted with fixed factor.
    This class helps to balance multiple losses if they have different scales
    '''
    def __init__(self, loss, weight=1.0):
        super().__init__()
        self.loss = loss
        self.weight = weight

    def forward(self, *input):
        return self.loss(*input) * self.weight

class JointLoss(_Loss):
    'Wrap two loss functions into one. This class computes a weighted sum of two losses.'

    def __init__(self, first: nn.Module, second: nn.Module, first_weight=1.0, second_weight=1.0):
        super().__init__()
        self.first = WeightedLoss(first, first_weight)
        self.second = WeightedLoss(second, second_weight)

    def forward(self, *input):
        return self.first(*input) + self.second(*input)

# Cell
class Poly1CrossEntropyLoss(nn.Module):
    def __init__(self,
                 num_classes: int,
                 epsilon: float = 1.0,
                 reduction: str = "mean"):
        """
        Create instance of Poly1CrossEntropyLoss
        :param num_classes:
        :param epsilon:
        :param reduction: one of none|sum|mean, apply reduction to final loss tensor
        """
        super(Poly1CrossEntropyLoss, self).__init__()
        self.num_classes = num_classes
        self.epsilon = epsilon
        self.reduction = reduction
        return

    def forward(self, logits, labels):
        """
        Forward pass
        :param logits: tensor of shape [BNHW]
        :param labels: tensor of shape [BHW]
        :return: poly cross-entropy loss
        """
        labels_onehot = F.one_hot(labels, num_classes=self.num_classes).to(device=logits.device,
                                                                           dtype=logits.dtype)
        labels_onehot = torch.moveaxis(labels_onehot, -1, 1)
        pt = torch.sum(labels_onehot * F.softmax(logits, dim=1), dim=1)
        CE = F.cross_entropy(input=logits, target=labels, reduction='none')
        poly1 = CE + self.epsilon * (1 - pt)
        if self.reduction == "mean":
            poly1 = poly1.mean()
        elif self.reduction == "sum":
            poly1 = poly1.sum()
        return poly1

# Cell
def get_loss(loss_name, mode='multiclass', classes=[1], smooth_factor=0., alpha=0.5, beta=0.5, gamma=2.0, reduction='mean', **kwargs):
    'Load losses from based on loss_name'

    assert loss_name in LOSSES, f'Select one of {LOSSES}'

    if loss_name=="CrossEntropyLoss":
        loss = fastai.losses.CrossEntropyLossFlat(axis=1)

    if loss_name=="SoftCrossEntropyLoss":
        loss = smp.losses.SoftCrossEntropyLoss(smooth_factor=smooth_factor, **kwargs)

    elif loss_name=="DiceLoss":
        loss = smp.losses.DiceLoss(mode=mode, classes=classes, **kwargs)

    elif loss_name=="JaccardLoss":
        loss = smp.losses.JaccardLoss(mode=mode, classes=classes, **kwargs)

    elif loss_name=="FocalLoss":
        loss = smp.losses.FocalLoss(mode=mode, alpha=alpha, gamma=gamma, reduction=reduction, **kwargs)

    elif loss_name=="LovaszLoss":
        loss = smp.losses.LovaszLoss(mode=mode, **kwargs)

    elif loss_name=="TverskyLoss":
        kornia = import_package('kornia')
        loss = kornia.losses.TverskyLoss(alpha=alpha, beta=beta, **kwargs)

    elif loss_name=="CrossEntropyDiceLoss":
        dc = smp.losses.DiceLoss(mode=mode, classes=classes, **kwargs)
        ce = fastai.losses.CrossEntropyLossFlat(axis=1)
        loss = JointLoss(ce, dc, 1, 1)

    elif loss_name=="Poly1CrossEntropyLoss":
        loss = Poly1CrossEntropyLoss(num_classes=max(classes)+1)

    return loss