"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.PredefinedMetric = exports.TargetTrackingScalingPolicy = void 0;
const cdk = require("@aws-cdk/core");
const applicationautoscaling_generated_1 = require("./applicationautoscaling.generated");
class TargetTrackingScalingPolicy extends cdk.Construct {
    constructor(scope, id, props) {
        if ((props.customMetric === undefined) === (props.predefinedMetric === undefined)) {
            throw new Error('Exactly one of \'customMetric\' or \'predefinedMetric\' must be specified.');
        }
        if (props.customMetric && !props.customMetric.toMetricConfig().metricStat) {
            throw new Error('Only direct metrics are supported for Target Tracking. Use Step Scaling or supply a Metric object.');
        }
        super(scope, id);
        const resource = new applicationautoscaling_generated_1.CfnScalingPolicy(this, 'Resource', {
            policyName: props.policyName || this.node.uniqueId,
            policyType: 'TargetTrackingScaling',
            scalingTargetId: props.scalingTarget.scalableTargetId,
            targetTrackingScalingPolicyConfiguration: {
                customizedMetricSpecification: renderCustomMetric(props.customMetric),
                disableScaleIn: props.disableScaleIn,
                predefinedMetricSpecification: props.predefinedMetric !== undefined ? {
                    predefinedMetricType: props.predefinedMetric,
                    resourceLabel: props.resourceLabel,
                } : undefined,
                scaleInCooldown: props.scaleInCooldown && props.scaleInCooldown.toSeconds(),
                scaleOutCooldown: props.scaleOutCooldown && props.scaleOutCooldown.toSeconds(),
                targetValue: props.targetValue,
            },
        });
        this.scalingPolicyArn = resource.ref;
    }
}
exports.TargetTrackingScalingPolicy = TargetTrackingScalingPolicy;
function renderCustomMetric(metric) {
    if (!metric) {
        return undefined;
    }
    const c = metric.toMetricConfig().metricStat;
    if (c.statistic.startsWith('p')) {
        throw new Error(`Cannot use statistic '${c.statistic}' for Target Tracking: only 'Average', 'Minimum', 'Maximum', 'SampleCount', and 'Sum' are supported.`);
    }
    return {
        dimensions: c.dimensions,
        metricName: c.metricName,
        namespace: c.namespace,
        statistic: c.statistic,
        unit: c.unitFilter,
    };
}
/**
 * One of the predefined autoscaling metrics
 */
var PredefinedMetric;
(function (PredefinedMetric) {
    /**
     * DYNAMODB_READ_CAPACITY_UTILIZATIO
     * @see https://docs.aws.amazon.com/autoscaling/application/APIReference/API_PredefinedMetricSpecification.html
     */
    PredefinedMetric["DYNAMODB_READ_CAPACITY_UTILIZATION"] = "DynamoDBReadCapacityUtilization";
    /**
     * DYANMODB_WRITE_CAPACITY_UTILIZATION
     * @see https://docs.aws.amazon.com/autoscaling/application/APIReference/API_PredefinedMetricSpecification.html
     */
    PredefinedMetric["DYANMODB_WRITE_CAPACITY_UTILIZATION"] = "DynamoDBWriteCapacityUtilization";
    /**
     * ALB_REQUEST_COUNT_PER_TARGET
     * @see https://docs.aws.amazon.com/autoscaling/application/APIReference/API_PredefinedMetricSpecification.html
     */
    PredefinedMetric["ALB_REQUEST_COUNT_PER_TARGET"] = "ALBRequestCountPerTarget";
    /**
     * RDS_READER_AVERAGE_CPU_UTILIZATION
     * @see https://docs.aws.amazon.com/autoscaling/application/APIReference/API_PredefinedMetricSpecification.html
     */
    PredefinedMetric["RDS_READER_AVERAGE_CPU_UTILIZATION"] = "RDSReaderAverageCPUUtilization";
    /**
     * RDS_READER_AVERAGE_DATABASE_CONNECTIONS
     * @see https://docs.aws.amazon.com/autoscaling/application/APIReference/API_PredefinedMetricSpecification.html
     */
    PredefinedMetric["RDS_READER_AVERAGE_DATABASE_CONNECTIONS"] = "RDSReaderAverageDatabaseConnections";
    /**
     * EC2_SPOT_FLEET_REQUEST_AVERAGE_CPU_UTILIZATION
     * @see https://docs.aws.amazon.com/autoscaling/application/APIReference/API_PredefinedMetricSpecification.html
     */
    PredefinedMetric["EC2_SPOT_FLEET_REQUEST_AVERAGE_CPU_UTILIZATION"] = "EC2SpotFleetRequestAverageCPUUtilization";
    /**
     * EC2_SPOT_FLEET_REQUEST_AVERAGE_NETWORK_IN
     * @see https://docs.aws.amazon.com/autoscaling/application/APIReference/API_PredefinedMetricSpecification.html
     */
    PredefinedMetric["EC2_SPOT_FLEET_REQUEST_AVERAGE_NETWORK_IN"] = "EC2SpotFleetRequestAverageNetworkIn";
    /**
     * EC2_SPOT_FLEET_REQUEST_AVERAGE_NETWORK_OUT
     * @see https://docs.aws.amazon.com/autoscaling/application/APIReference/API_PredefinedMetricSpecification.html
     */
    PredefinedMetric["EC2_SPOT_FLEET_REQUEST_AVERAGE_NETWORK_OUT"] = "EC2SpotFleetRequestAverageNetworkOut";
    /**
     * SAGEMAKER_VARIANT_INVOCATIONS_PER_INSTANCE
     * @see https://docs.aws.amazon.com/autoscaling/application/APIReference/API_PredefinedMetricSpecification.html
     */
    PredefinedMetric["SAGEMAKER_VARIANT_INVOCATIONS_PER_INSTANCE"] = "SageMakerVariantInvocationsPerInstance";
    /**
     * ECS_SERVICE_AVERAGE_CPU_UTILIZATION
     * @see https://docs.aws.amazon.com/autoscaling/application/APIReference/API_PredefinedMetricSpecification.html
     */
    PredefinedMetric["ECS_SERVICE_AVERAGE_CPU_UTILIZATION"] = "ECSServiceAverageCPUUtilization";
    /**
     * ECS_SERVICE_AVERAGE_MEMORY_UTILIZATION
     * @see https://docs.aws.amazon.com/autoscaling/application/APIReference/API_PredefinedMetricSpecification.html
     */
    PredefinedMetric["ECS_SERVICE_AVERAGE_MEMORY_UTILIZATION"] = "ECSServiceAverageMemoryUtilization";
    /**
     * LAMBDA_PROVISIONED_CONCURRENCY_UTILIZATION
     * @see https://docs.aws.amazon.com/lambda/latest/dg/monitoring-metrics.html#monitoring-metrics-concurrency
     */
    PredefinedMetric["LAMBDA_PROVISIONED_CONCURRENCY_UTILIZATION"] = "LambdaProvisionedConcurrencyUtilization";
})(PredefinedMetric = exports.PredefinedMetric || (exports.PredefinedMetric = {}));
//# sourceMappingURL=data:application/json;base64,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