"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.BaseScalableAttribute = void 0;
const cdk = require("@aws-cdk/core");
const scalable_target_1 = require("./scalable-target");
/**
 * Represent an attribute for which autoscaling can be configured
 *
 * This class is basically a light wrapper around ScalableTarget, but with
 * all methods protected instead of public so they can be selectively
 * exposed and/or more specific versions of them can be exposed by derived
 * classes for individual services support autoscaling.
 *
 * Typical use cases:
 *
 * - Hide away the PredefinedMetric enum for target tracking policies.
 * - Don't expose all scaling methods (for example Dynamo tables don't support
 *   Step Scaling, so the Dynamo subclass won't expose this method).
 */
class BaseScalableAttribute extends cdk.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.props = props;
        this.target = new scalable_target_1.ScalableTarget(this, 'Target', {
            serviceNamespace: this.props.serviceNamespace,
            scalableDimension: this.props.dimension,
            resourceId: this.props.resourceId,
            role: this.props.role,
            minCapacity: props.minCapacity !== undefined ? props.minCapacity : 1,
            maxCapacity: props.maxCapacity,
        });
    }
    /**
     * Scale out or in based on time
     */
    doScaleOnSchedule(id, props) {
        this.target.scaleOnSchedule(id, props);
    }
    /**
     * Scale out or in based on a metric value
     */
    doScaleOnMetric(id, props) {
        this.target.scaleOnMetric(id, props);
    }
    /**
     * Scale out or in in order to keep a metric around a target value
     */
    doScaleToTrackMetric(id, props) {
        this.target.scaleToTrackMetric(id, props);
    }
}
exports.BaseScalableAttribute = BaseScalableAttribute;
//# sourceMappingURL=data:application/json;base64,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