import abc
from uuid import UUID
from shared.domain.model import AggregateRoot, Repository as BaseRepository
from sqlalchemy import create_engine
from sqlalchemy.orm import sessionmaker
from sqlalchemy.dialects.mysql import BINARY
from sqlalchemy.types import TypeDecorator


class Repository(BaseRepository):

    def __init__(self, aggregate: AggregateRoot, dsn: str):
        super().__init__(aggregate)
        some_engine = create_engine(dsn)
        Session = sessionmaker(bind=some_engine)

        self.__session = Session()

    def add(self, aggregate: AggregateRoot):
        self.__session.add(aggregate)
        self.__session.commit()

    def find(self, **kwargs) -> AggregateRoot:
        return self.__session.query(self.__aggregate).filter_by(**kwargs).first()


class Orm(metaclass=abc.ABCMeta):
    @classmethod
    def __subclasshook__(cls, subclass):
        if subclass is not Orm:
            return NotImplemented

        if hasattr(subclass, "start_mappers") and callable(subclass.start_mappers):
            return True

        return NotImplemented

    @abc.abstractmethod
    def start_mappers(self) -> None:
        raise NotImplemented


class BinaryUUID(TypeDecorator):
    '''Optimize UUID keys. Store as 16 bit binary, retrieve as uuid.
    inspired by:
        http://mysqlserverteam.com/storing-uuid-values-in-mysql-tables/
    '''

    impl = BINARY(16)

    def process_literal_param(self, value, dialect):
        pass

    @property
    def python_type(self):
        pass

    def process_bind_param(self, value, dialect):
        try:
            return value.hex
        except AttributeError:
            try:
                return UUID(value).hex
            except TypeError:
                # for some reason we ended up with the bytestring
                # ¯\_(ツ)_/¯
                # I'm not sure why you would do that,
                # but here you go anyway.
                return value

    def process_result_value(self, value, dialect):
        return UUID(bytes=value)
