# AUTOGENERATED! DO NOT EDIT! File to edit: ../notebooks/survey_scenario/leximpact_survey_scenario.ipynb.

# %% auto 0
__all__ = [
    "tc",
    "log",
    "config",
    "annee_erfs",
    "annee_pote",
    "annee_de_calcul",
    "variables",
    "leximpact_tbs",
    "propagate_new_variable",
    "monte_carlo_computer",
    "quimen",
    "quifam",
    "quifoy",
    "idmen",
    "idfam",
    "idfoy",
    "menage_id",
    "famille_id",
    "foyer_fiscal_id",
    "noindiv",
    "leximpact_tbs_extension",
    "LeximpactErfsSurveyScenario",
]

# %% ../notebooks/survey_scenario/leximpact_survey_scenario.ipynb 3
import logging
import unittest
from multiprocessing import Process, Queue
from typing import Any, Optional

import numpy as np
import pandas as pd
from leximpact_socio_fisca_simu_etat.config import Configuration
from openfisca_core import reforms

# from openfisca_core.reforms import Reform
from openfisca_core.taxbenefitsystems import TaxBenefitSystem
from openfisca_france_data import france_data_tax_benefit_system
from openfisca_france_data.erfs_fpr.scenario import ErfsFprSurveyScenario
from openfisca_france_data.model.base import ETERNITY, Individu, Variable
from openfisca_survey_manager.input_dataframe_generator import set_table_in_survey
from openfisca_survey_manager.survey_collections import SurveyCollection

# Ci-dessous, ce sont les imports de fonctions rédigées en 2022
# (ou des fonctions 2021 qui sont ré-adaptées pour Survey-Scenario dans de nouveaux notebooks)
from .calibration_sur_marges import LexCalibration
from .helpers_survey_scenario import get_copules
from leximpact_prepare_data.inflation_calibration_values import (
    get_margins,
    inflation_coefs,
    inflation_foyers_fiscaux,
)
from .monte_carlo_computer import apply_Monte_Carlo

# %% ../notebooks/survey_scenario/leximpact_survey_scenario.ipynb 4
tc = unittest.TestCase()
pd.set_option("display.max_columns", None)
log = logging.getLogger(__name__)

config = Configuration(project_folder="leximpact-prepare-data")
log.debug(config)

# Import des années
annee_erfs = config.get("YEAR_ERFS")
annee_pote = config.get("YEAR_POTE")
annee_de_calcul = config.get("YEAR_COMPUTATION")


# %% ../notebooks/survey_scenario/leximpact_survey_scenario.ipynb 7
def propagate_new_variable(survey_scenario, year_insertion, new_variable, new_array):
    """
        Ajoute une nouvelle variable dans la base selon son année de calcul,
        et la propage dans toutes les années

    :year_insertion:    L'année de calcul de la variable
    :new_variable:      Le nom de la variable
    :new_array:         Les données correspondant à cette variable

    """

    survey_scenario.simulation.set_input(new_variable, year_insertion, new_array)

    # Inflation progressive
    for year in range(year_insertion, survey_scenario.final_year + 1):
        print(year)
        survey_scenario.inflator(year, year + 1, new_variable)

    # Deflation progressive
    for year in range(year_insertion, survey_scenario.annee_erfs, -1):
        print(year)
        survey_scenario.deflator(year, year - 1, new_variable)


# %% ../notebooks/survey_scenario/leximpact_survey_scenario.ipynb 8
def monte_carlo_computer(scenario, year, reference_var, new_var):
    # Charger les copules
    copules = get_copules(year, new_var)

    # On travaille avec une unique entité pour reference_var et new_var (pour le moment: les foyers)
    ref_entity = scenario.tax_benefit_system.get_variable(reference_var).entity.key
    var_entity = scenario.tax_benefit_system.get_variable(new_var).entity.key

    if var_entity == ref_entity == "foyer_fiscal":
        data_frame_by_entity = scenario.create_data_frame_by_entity(
            variables=[reference_var, new_var, "weight_foyers"], period=year
        )
        df = data_frame_by_entity["foyer_fiscal"]
    else:
        raise Exception(
            "Attention, (pour l'instant) on ne peut calculer les copules que d'une variable en foyers vers une variable en foyers"
        )

    # On travaille directement avec une base en foyers
    df, errors = apply_Monte_Carlo(
        df,
        copules,
        bucket_level_col_name=reference_var,
        out_col=new_var,
        nb_tirage=5,  # 20
        seed=None,
        use_force_mean_with_factor=False,
        debug=False,
    )

    # On vérifie qu'on a toujours le même total de poids dans la base et dans la simulation (ie. pas de distortion)
    tc.assertEqual(
        scenario.calculate_variable("weight_foyers", period=year).sum(),
        df["weight_foyers"].sum(),
    )

    #     # Insertion dans le scenario
    #     scenario.simulation.delete_arrays(new_var, year)
    #     scenario.simulation.set_input(new_var, year, df[new_var])

    #     total = scenario.calculate_variable(new_var, period=year).sum()
    #     print(f"Injection de {total} € pour {new_var}")
    #     # On vérifie qu'on a bien le total de variable
    #     tc.assertAlmostEqual(total / df[new_var].sum(), 1, places=2)
    # Return the array of results
    return df[new_var]


# %% ../notebooks/survey_scenario/leximpact_survey_scenario.ipynb 11
class quimen(Variable):
    is_period_size_independent = True
    value_type = int
    entity = Individu
    label = "Rôle dans le ménage"
    definition_period = ETERNITY


class quifam(Variable):
    is_period_size_independent = True
    value_type = int
    entity = Individu
    label = "Rôle dans la famille"
    definition_period = ETERNITY


class quifoy(Variable):
    is_period_size_independent = True
    value_type = int
    entity = Individu
    label = "Rôle dans le foyer fiscal"
    definition_period = ETERNITY


class idmen(Variable):
    is_period_size_independent = True
    value_type = int
    entity = Individu
    label = "Identifiant ménage dans openfisca-france-data"
    definition_period = ETERNITY


class idfam(Variable):
    is_period_size_independent = True
    value_type = int
    entity = Individu
    label = "Identifiant famille dans openfisca-france-data"
    definition_period = ETERNITY


class idfoy(Variable):
    is_period_size_independent = True
    value_type = int
    entity = Individu
    label = "Identifiant foyer fiscal dans openfisca-france-data"
    definition_period = ETERNITY


# variables ajoutées pour openfisca-survey-manager.AbstractSurveyScenario._set_used_as_input_variables_by_entity
# suite à l'appel d'init_from_data (vu via tests) :


class menage_id(Variable):
    is_period_size_independent = True
    value_type = int
    entity = Individu
    label = "Identifiant ménage"  # dans openfisca-survey-manager ?
    definition_period = ETERNITY


class famille_id(Variable):
    is_period_size_independent = True
    value_type = int
    entity = Individu
    label = "Identifiant famille"  # dans openfisca-survey-manager ?
    definition_period = ETERNITY


class foyer_fiscal_id(Variable):
    is_period_size_independent = True
    value_type = int
    entity = Individu
    label = "Identifiant foyer fiscal"  # dans openfisca-survey-manager ?
    definition_period = ETERNITY


class noindiv(Variable):
    is_period_size_independent = True
    value_type = str  # champ texte de 10 caractères
    entity = Individu
    label = "Identifiant des individus dans l'enquête ERFS-FPR de l'INSEE"
    definition_period = ETERNITY


variables = [
    quimen,
    quifam,
    quifoy,
    idmen,
    idfam,
    idfoy,
    menage_id,
    famille_id,
    foyer_fiscal_id,
    noindiv,
]


class leximpact_tbs_extension(reforms.Reform):
    def apply(self):
        for variable in variables:
            if variable == Variable:
                continue
            try:
                self.add_variable(variable)
            except AttributeError:
                self.update_variable(variable)


leximpact_tbs = leximpact_tbs_extension(france_data_tax_benefit_system)


# %% ../notebooks/survey_scenario/leximpact_survey_scenario.ipynb 14
class LeximpactErfsSurveyScenario(ErfsFprSurveyScenario):
    """Survey scenario spécialisé pour l'ERFS-FPR utilisée par Leximpact."""

    def __init__(
        self,
        annee_erfs: int = annee_erfs,
        final_year: int = annee_de_calcul,
        rebuild_input_data: bool = False,
        init_from_data: bool = True,
        # TODO baseline_tax_benefit_system non utilisé par leximpact ;
        # l'utiliser pour la loi en vigueur et migrer tax_benefit_system vers le TBS  du PLF ?
        baseline_tax_benefit_system: Optional[TaxBenefitSystem] = None,
        tax_benefit_system_plf: Optional[TaxBenefitSystem] = None,
        data: Any = None,
        # TODO utiliser reform pour la réforme usager et préciser si c'est un fichier Python de réforme ou un TBS ?
        tax_benefit_system_reform: Optional[TaxBenefitSystem] = None,
        collection: str = "openfisca_erfs_fpr",
        survey_name: str = "input",
    ):
        """Crée un `LeximpactErfsSurveyScenario`.

        :param annee_erfs:                  L'année des données de l'enquête.
        :param rebuild_input_data:          Si l'on doit formatter les données (raw) ou pas.
        :param init_from_data:              Si on veut suspendre l'initialisation automatique par les données
        :param tax_benefit_system:          Le `TaxBenefitSystem` déjà réformé.
        :param baseline_tax_benefit_system: Le `TaxBenefitSystem` au droit courant.
        :param data:                        Les données de l'enquête.
        :param reform:                      Reform OpenFisca.
        :param collection:                  Collection à lire.
        :param survey_name:                 Nom de l'enquête.
        """

        super().__init__(annee_erfs)
        self.collection = collection
        self.annee_erfs = int(annee_erfs)
        self.final_year = int(final_year)

        # La liste des variables openfisca minimales dont la pipeline a besoin pour des résultats corrects :
        # variables de base + variables attendues par leximpact.
        # Ces variables ne doivent pas être neutralisées.
        self._set_used_as_input_variables()
        # non_neutralizable_variables hérite d'une liste de variables d'ErfsFprSurveyScenario
        self.non_neutralizable_variables += self.used_as_input_variables

        # ## Initialisation des Baseline/ Non baseline TaxBenefitSystems
        if baseline_tax_benefit_system is None:
            baseline_tax_benefit_system = leximpact_tbs

        if tax_benefit_system_plf is None:
            tax_benefit_system_plf = leximpact_tbs

        self.set_tax_benefit_systems(
            tax_benefit_system=tax_benefit_system_plf,
            baseline_tax_benefit_system=baseline_tax_benefit_system,
        )

        # ## Création de la base de données sur les périodes voulues
        # S'il n'y a pas de données, on sait où les trouver.
        if data is None:
            # List of years available
            years_available = []
            survey_collection = SurveyCollection.load(collection=collection)
            survey = survey_collection.get_survey(survey_name)
            for table_name, _ in survey.tables.items():
                if table_name[-4:].isnumeric():
                    years_available.append(int(table_name[-4:]))
            years_available = list(set(years_available))

            # List of years to create
            years = [year for year in range(self.annee_erfs, self.final_year + 1)]

            print(f"{years_available=} vs {years=}")

            data = {"input_data_table_by_entity_by_period": {}, "survey": survey_name}
            current_year = None

            for year in years:
                if data["input_data_table_by_entity_by_period"].get(year) is None:
                    data["input_data_table_by_entity_by_period"][year] = {}
                if year in years_available:
                    data_year = year
                else:
                    data_year = self.annee_erfs
                    print(f"WARNING: no data for {year}, will took {data_year}")
                for table_name, _ in survey.tables.items():
                    current_year = table_name[-4:]
                    if current_year.isnumeric():
                        current_year = int(current_year)
                        entity = table_name[:-5]
                        if current_year == data_year:
                            # print(f"Using {table_name} for {entity} for {year}")
                            data["input_data_table_by_entity_by_period"][year][
                                entity
                            ] = table_name
                    else:
                        print(
                            f"WARNING {table_name} will be ignored because it has no year !!!"
                        )

        print("Données du scénario : \n", data)
        self.input_data = data
        self.rebuild_input_data = rebuild_input_data

        if init_from_data:
            self.input_from_data()

    # Initialisation custom
    def input_from_data(self, var_list=None):
        # On défini nos variables d'interet sont définies dans self.used_as_input_variables

        # Initialisation du scenario avec les données
        self.init_from_data(
            data=self.input_data, rebuild_input_data=self.rebuild_input_data
        )

        # Initialisation des tables
        # survey_collection = SurveyCollection.load(collection="openfisca_erfs_fpr")
        # survey = survey_collection.get_survey("input")
        # self.survey = survey
        # Individus
        # individu = survey.get_values(table=f"individu_{annee_erfs}", ignorecase=True)
        # Menages
        # menage = survey.get_values(table=f"menage_{annee_erfs}", ignorecase=True)
        # Ajout des poids de leur menage aux individus
        # individu, menage = get_wprm(individu, menage)

    def propagate_var(self, var, year_start, year_end, replace=False):
        for year in range(int(year_start) + 1, int(year_end) + 1):
            array_var = self.calculate_variable(var, period=year_start)
            try:
                self.simulation.set_input(var, year, array_var)
            except ValueError:
                if replace:
                    self.simulation.delete_arrays(var, year)
                    self.simulation.set_input(var, year, array_var)
                    print(
                        "Attention, la variable existante a été remplacée par les nouvelles valeurs"
                    )
                else:
                    print(
                        "Attention, la variable existe déjà dans la base: on n'a pas remplacé la valeur"
                    )
                    continue

    # Propagation des valeurs d'une année de notre simulation sur les années suivantes
    def _propagate_whole_database(self, year_start, year_end, replace):
        for var in self.used_as_input_variables:
            if var in self.simulation.tax_benefit_system.variables:
                self.propagate_var(var, year_start, year_end, replace=replace)

        print("On propage les valeurs de ", year_start, " à ", year_end)

    # On redéfini les variables
    def _set_used_as_input_variables(self):
        # ne garde pas toutes les variables par défaut de ErfsFprSurveyScenario :
        # on retire les colonnes à zéro de l'ERFS-FPR
        inherited_but_ignored_variables = [
            "loyer",  # génèrera un Warning: You cannot set a value for the variable loyer, as it has been neutralized
            "f4ba",
            "primes_fonction_publique",
            "statut_occupation_logement",  # génèrera un Warning: You cannot set a value for the variable statut_occupation_logement, as it has been neutralized
            "taxe_habitation",
            "traitement_indiciaire_brut",
            "zone_apl",
        ]
        self.used_as_input_variables = list(
            set(self.used_as_input_variables) - set(inherited_but_ignored_variables)
        )

        # variables leximpact :
        leximpact_variables = None

        variables_to_keep_or_to_calculate = ["age", "rfr", "nbptr", "veuf"]

        # ajout des variables d'entités aux variables principales métier d'input et calculées
        wprms = [
            # "weight_ini",  # poids initial leximpact ?
            "weight_familles",
            "weight_foyers",
            "weight_individus",
            "weight_menages",
            "wprm",
        ]
        ids = [
            # TODO Choisir une clef d'identification unique entre openfisca-survey-manager et openfisca-france-data ? :'-D
            # sachant que les valeurs existent dans openfisca-core.populations.group_population.GroupPopulation.members_entity_id
            # openfisca-survey-manager :
            "foyer_fiscal_id",
            "famille_id",
            "menage_id",
            # openfisca-france-data :
            "idfoy",
            "idfam",
            "idmen",
        ]
        roles = [
            "noindiv",  # source : ERFS-FPR
            "quifoy",
            "quifam",
            "quimen",
            # "famille_role", , en doublon avec quifam ? Déjà dans openfisca-core.populations.group_population.GroupPopulation.members_role
            # "famille_position", déjà dans openfisca-core.populations.group_population.GroupPopulation.members_position
            # "foyer_fiscal_role", en doublon avec quifoy ? Déjà dans openfisca-core.populations.group_population.GroupPopulation.members_role
            # "foyer_fiscal_position", déjà dans openfisca-core.populations.group_population.GroupPopulation.members_position
            # "menage_role", en doublon avec quimen ? Déjà dans openfisca-core.populations.group_population.GroupPopulation.members_role
            # "menage_position", déjà dans openfisca-core.populations.group_population.GroupPopulation.members_position
        ]

        # ajout des variables qui seront injectées par Monte-Carlo
        future_monte_carlo_variables = [
            "assiette_csg_plus_values",
            "f4ba",
            "rente_viagere_titre_onereux_net",
            "revenus_capitaux_prelevement_bareme",
            "revenus_capitaux_prelevement_forfaitaire_unique_ir",
            "revenus_capitaux_prelevement_liberatoire",
        ]

        leximpact_variables = (
            variables_to_keep_or_to_calculate
            + wprms
            + ids
            + roles
            + future_monte_carlo_variables
        )
        self.used_as_input_variables = list(
            set(self.used_as_input_variables + leximpact_variables)
        )

    def inflator(self, year_start, year_end, variables=None):
        """
            Fonction d'inflation

        :year_start:   Année de départ (array qui sera copié et inflaté)
        :year_end:     Année d'arrivée (dans laquelle on colle l'array inflaté)
        :variables:    Variables à inflater (par défaut seulement les `used_as_input_variables`)
        """
        assert year_end > year_start

        if variables is None:
            variables = self.used_as_input_variables
        # Inflation des variables
        inflator_by_variable = inflation_coefs(variables, year_start, year_end)
        print("Inflation de ", year_start, " à ", year_end, " : ", inflator_by_variable)
        self.inflate(
            inflator_by_variable=inflator_by_variable,
            period=year_end,
            target_by_variable=None,  # On n'utilise pas le target_by_variable car le code nous force à se baser sur le baseline_simulation
        )

    def deflator(self, year_start, year_end, variables=None):
        """
            Fonction de déflation (pour propager une calibration en arrière dans le temps)

        :year_start:   Année de départ (array qui sera copié et inflaté)
        :year_end:     Année d'arrivée (dans laquelle on colle l'array inflaté)
        :variables:    Variables à inflater (par défaut seulement les `used_as_input_variables`)
        """
        assert year_end < year_start
        if variables is None:
            variables = self.used_as_input_variables

        inflator_by_variable = inflation_coefs(variables, year_end, year_start)
        for key in inflator_by_variable.keys():
            inflator_by_variable[key] = 1 / inflator_by_variable[key]
        print(
            "Déflation de ", year_start, " vers ", year_end, " : ", inflator_by_variable
        )
        self.inflate(
            inflator_by_variable=inflator_by_variable,
            period=year_end,
            target_by_variable=None,
        )

    def inflator_foyers(self, year_start, year_end):
        # Etat initial
        initial_ff_weight = self.simulation.calculate_add(
            "weight_foyers", period=year_start
        ).sum()
        inflator_ff = inflation_foyers_fiscaux(year_end) / initial_ff_weight
        print("Inflation des personnes : ", inflator_ff)
        # Injection dans la base
        for entity_label in self.weight_variable_by_entity.values():
            array = self.simulation.calculate_add(entity_label, period=year_start)
            self.simulation.delete_arrays(
                entity_label, period=year_end
            )  # delete existing arrays
            self.simulation.set_input(
                entity_label, year_end, inflator_ff * array
            )  # insert inflated array
            print("Inflation des foyers propagée aux poids de : ", entity_label)

    def monte_carlo_injector(self, reference_var, var, res_queue: Queue):
        """
        Task for parallel processing
        Arg:
            scenario: scenario instance
            var: variable to inject
        return an array with the data
        """
        print(f"Injection de {var}")
        # On supprime la variable si elle existe déjà
        self.simulation.delete_arrays(var, period=annee_pote)
        # Injection sur la base du RFR
        # leximpact_survey_scenario.injector(annee_pote, "rfr", var)
        # Calcul de la nouvelle variable par Monte-Carlo à partir des copules de POTE
        result_array = monte_carlo_computer(self, annee_pote, reference_var, var)
        # total = leximpact_survey_scenario.calculate_variable(var, period=annee_pote).sum()
        # print(f"Injection de {total} € pour {var}")
        res_queue.put({var: result_array})

    def injector(self, year, reference_var, variables_to_inject):
        """
        Fonction d'injection de variable par l'algorithme de Monte-Carlo

        :year:              Année d'injection (année de la base ET des copules)
        :reference_var:     Variable de référence pour les copules (par exemple, RFR)
        :new_var:           Variables à ajouter dans la base (elles doivent toutes avoir la même `reference_var`)
        """
        # TODO : Permettre l'injection de variables individus SI un jour on utilise d'autres bases que POTE
        # "L'injection se fait avec des copules de POTE, donc uniquement d'une variable en foyers à une variable en foyers"

        processes = []
        results = []
        q = Queue()
        # On génère la liste des appels à lancer
        for v in variables_to_inject:
            p = Process(target=self.monte_carlo_injector, args=(reference_var, v, q))
            processes.append(p)
        # On lance les appels
        [p.start() for p in processes]
        # On récupère les résultats
        [results.append(q.get()) for p in processes]
        # On traite les résultats
        for res in results:
            for variable, array in res.items():
                self.simulation.delete_arrays(variable, annee_pote)
                self.simulation.set_input(variable, annee_pote, array)

        # Sans multiprocessing, fonctionne bien, mais long
        # for v in variables_to_inject:
        #     result = self.monte_carlo_injector(reference_var, v)
        #     self.simulation.delete_arrays(v, annee_pote)
        #     self.simulation.set_input(v, annee_pote, result[v])

        # create a process pool that uses all CPUs to compute all variables in parallel.
        # Time without multiprocessing : 7min 47s, with: 2min 54s
        # scenario = Value('scenario', self)
        # variables = Array('variables_to_inject', variables_to_inject)

    #         processes = []

    #         for v in variables_to_inject:
    #             p = Process(target=self.monte_carlo_injector, args=(v,))
    #             processes.append(p)

    #         [x.start() for x in processes]

    #         with Pool() as pool:
    #             # call the function for each item in parallel
    #             monte_carlo_results = pool.map(self.monte_carlo_injector, variables_to_inject)
    #         # On sauve les résultats dans la simulation
    #         # On ne peut pas le faire dans la partie multiprocessing : les données seraient perdues !
    #         for monte_carlo in monte_carlo_results:
    #             for variable, of_vector in monte_carlo.items():
    #                 leximpact_survey_scenario.simulation.delete_arrays(variable, annee_pote)
    #                 leximpact_survey_scenario.simulation.set_input(variable, annee_pote, of_vector)
    # total = self.calculate_variable(var, period=year).sum()
    # total = self.compute_aggregate(var, period=annee_pote)
    # print(f"Somme de {new_var} après injection : {total} pour {year}")

    def calibrator(self, year, variables=None, calmar_parameters=None):
        """
            Fonction de calibration
        :year       Année de calibration
        """

        # Choix des variables à calibrer
        if variables is None:
            variables = (
                self.used_as_input_variables
            )  # Par défaut on prend les variables de référence

        # Choix des marges
        target_margin_by_variable = get_margins(self, year, variables)

        # On choisi nos parametres - À rendre parametrable ?
        # Pour le moment on met les paramètres par défaut de Mahdi,
        # il faudra les adapter avec des tests sur les résultats de la calibration
        # peut-etre une boucle d'asservissement en machine learning ?
        if calmar_parameters is None:
            calmar_parameters = {
                "use_proportions": True,
                "initial_weight": None,
                "method": "raking ratio",  # Options: None, 'linear', 'raking ratio', 'logit'
                "up": 3,
                "lo": 1 / 3,
            }

        calibration = LexCalibration(self, year)
        calibration.calibrate(target_margin_by_variable, calmar_parameters)

    def save_current_survey(
        self,
        collection: str = "openfisca_erfs_fpr",
        survey_name: str = "input",
        period: str = None,
    ):
        """
        Récupére tous les dataframes de toutes les entités et les sauve dans le survey scenario
        """
        if period is None:
            for year in range(self.annee_erfs, self.final_year + 1):
                # merge=True permet d'obtenir les variables clef pour pouvoir faire des merge ensuite avec les données.
                data_frame_by_entity = self.create_data_frame_by_entity(
                    variables=self.used_as_input_variables, period=year, index=True
                )

                for entity, input_dataframe in data_frame_by_entity.items():
                    assert type(input_dataframe) == pd.DataFrame
                    print(
                        f"set_table_in_survey of {entity} for {year} in {collection}.{survey_name}"
                    )

                    # TODO Ajouter idfoy_original de ErfsFprSurveyScenario ? Serait l'identifiant de l'enquête ERFS-FPR.
                    if entity == "foyer_fiscal":
                        # ISSUE idfoy d'individus ne correspond pas à la cardinalité d'un dataframe de foyers fiscaux :
                        # assert "idfoy" in input_dataframe.columns.to_list()
                        # FIX : si dataframe et openfisca indexent bien les individus dans leues foyers fiscaux
                        # de la même manière, on utilise openfisca pour aggréger les idfoy au niveau des foyers fiscaux
                        idfoy = self.simulation.foyer_fiscal.members(
                            "idfoy", self.final_year
                        )
                        idfoy_simulation = self.simulation.populations[
                            "foyer_fiscal"
                        ].members_entity_id  # dimension : individus
                        assert np.array_equal(
                            idfoy, idfoy_simulation
                        ), f"😈 Des indices de {entity} diffèrent entre le dataframe et la simulation : {idfoy - idfoy_simulation}"

                        idfoy_aggreges_foyer_fiscal = self.simulation.populations[
                            "foyer_fiscal"
                        ].ids  # dimension : foyers fiscaux
                        input_dataframe[
                            "idfoy"
                        ] = idfoy_aggreges_foyer_fiscal  # TODO renommer ICI la colonne idfoy (variable d'individus normalement)

                    set_table_in_survey(
                        input_dataframe,
                        entity,
                        period=year,
                        collection=collection,
                        survey_name=survey_name,
                    )
                    # print(f"🎉 A new {survey_name}.h5 is born for {entity} :)")
        else:
            data_frame_by_entity = self.create_data_frame_by_entity(
                variables=self.used_as_input_variables, period=period
            )

            for entity, input_dataframe in data_frame_by_entity.items():
                set_table_in_survey(
                    input_dataframe,
                    entity,
                    period=period,
                    collection=collection,
                    survey_name=survey_name,
                )
                # print(f"🎉 A new {survey_name}.h5 is born for {entity} :)")
