"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.RouteSpec = exports.GrpcRetryEvent = exports.TcpRetryEvent = exports.HttpRetryEvent = exports.HttpRouteProtocol = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const http_route_path_match_1 = require("./http-route-path-match");
const utils_1 = require("./private/utils");
const shared_interfaces_1 = require("./shared-interfaces");
/**
 * Supported :scheme options for HTTP2.
 *
 * @stability stable
 */
var HttpRouteProtocol;
(function (HttpRouteProtocol) {
    HttpRouteProtocol["HTTP"] = "http";
    HttpRouteProtocol["HTTPS"] = "https";
})(HttpRouteProtocol = exports.HttpRouteProtocol || (exports.HttpRouteProtocol = {}));
/**
 * HTTP events on which to retry.
 *
 * @stability stable
 */
var HttpRetryEvent;
(function (HttpRetryEvent) {
    HttpRetryEvent["SERVER_ERROR"] = "server-error";
    HttpRetryEvent["GATEWAY_ERROR"] = "gateway-error";
    HttpRetryEvent["CLIENT_ERROR"] = "client-error";
    HttpRetryEvent["STREAM_ERROR"] = "stream-error";
})(HttpRetryEvent = exports.HttpRetryEvent || (exports.HttpRetryEvent = {}));
/**
 * TCP events on which you may retry
 */
var TcpRetryEvent;
(function (TcpRetryEvent) {
    TcpRetryEvent["CONNECTION_ERROR"] = "connection-error";
})(TcpRetryEvent = exports.TcpRetryEvent || (exports.TcpRetryEvent = {}));
/**
 * gRPC events.
 *
 * @stability stable
 */
var GrpcRetryEvent;
(function (GrpcRetryEvent) {
    GrpcRetryEvent["CANCELLED"] = "cancelled";
    GrpcRetryEvent["DEADLINE_EXCEEDED"] = "deadline-exceeded";
    GrpcRetryEvent["INTERNAL_ERROR"] = "internal";
    GrpcRetryEvent["RESOURCE_EXHAUSTED"] = "resource-exhausted";
    GrpcRetryEvent["UNAVAILABLE"] = "unavailable";
})(GrpcRetryEvent = exports.GrpcRetryEvent || (exports.GrpcRetryEvent = {}));
/**
 * Used to generate specs with different protocols for a RouteSpec.
 *
 * @stability stable
 */
class RouteSpec {
    /**
     * Creates an HTTP Based RouteSpec.
     *
     * @stability stable
     */
    static http(options) {
        jsiiDeprecationWarnings._aws_cdk_aws_appmesh_HttpRouteSpecOptions(options);
        return new HttpRouteSpec(options, shared_interfaces_1.Protocol.HTTP);
    }
    /**
     * Creates an HTTP2 Based RouteSpec.
     *
     * @stability stable
     */
    static http2(options) {
        jsiiDeprecationWarnings._aws_cdk_aws_appmesh_HttpRouteSpecOptions(options);
        return new HttpRouteSpec(options, shared_interfaces_1.Protocol.HTTP2);
    }
    /**
     * Creates a TCP Based RouteSpec.
     *
     * @stability stable
     */
    static tcp(options) {
        jsiiDeprecationWarnings._aws_cdk_aws_appmesh_TcpRouteSpecOptions(options);
        return new TcpRouteSpec(options);
    }
    /**
     * Creates a GRPC Based RouteSpec.
     *
     * @stability stable
     */
    static grpc(options) {
        jsiiDeprecationWarnings._aws_cdk_aws_appmesh_GrpcRouteSpecOptions(options);
        return new GrpcRouteSpec(options);
    }
}
exports.RouteSpec = RouteSpec;
_a = JSII_RTTI_SYMBOL_1;
RouteSpec[_a] = { fqn: "@aws-cdk/aws-appmesh.RouteSpec", version: "1.143.0" };
class HttpRouteSpec extends RouteSpec {
    constructor(props, protocol) {
        var _b, _c;
        super();
        this.protocol = protocol;
        this.match = props.match;
        this.weightedTargets = props.weightedTargets;
        this.timeout = props.timeout;
        this.priority = props.priority;
        if (props.retryPolicy) {
            const httpRetryEvents = (_b = props.retryPolicy.httpRetryEvents) !== null && _b !== void 0 ? _b : [];
            const tcpRetryEvents = (_c = props.retryPolicy.tcpRetryEvents) !== null && _c !== void 0 ? _c : [];
            if (httpRetryEvents.length + tcpRetryEvents.length === 0) {
                throw new Error('You must specify one value for at least one of `httpRetryEvents` or `tcpRetryEvents`');
            }
            this.retryPolicy = {
                ...props.retryPolicy,
                httpRetryEvents: httpRetryEvents.length > 0 ? httpRetryEvents : undefined,
                tcpRetryEvents: tcpRetryEvents.length > 0 ? tcpRetryEvents : undefined,
            };
        }
    }
    bind(scope) {
        var _b, _c, _d, _e, _f, _g;
        const pathMatchConfig = ((_c = (_b = this.match) === null || _b === void 0 ? void 0 : _b.path) !== null && _c !== void 0 ? _c : http_route_path_match_1.HttpRoutePathMatch.startsWith('/')).bind(scope);
        // Set prefix path match to '/' if none of path matches are defined.
        const headers = (_d = this.match) === null || _d === void 0 ? void 0 : _d.headers;
        const queryParameters = (_e = this.match) === null || _e === void 0 ? void 0 : _e.queryParameters;
        utils_1.validateHttpMatchArrayLength(headers, queryParameters);
        const httpConfig = {
            action: {
                weightedTargets: renderWeightedTargets(this.weightedTargets),
            },
            match: {
                prefix: pathMatchConfig.prefixPathMatch,
                path: pathMatchConfig.wholePathMatch,
                headers: headers === null || headers === void 0 ? void 0 : headers.map(header => header.bind(scope).headerMatch),
                method: (_f = this.match) === null || _f === void 0 ? void 0 : _f.method,
                scheme: (_g = this.match) === null || _g === void 0 ? void 0 : _g.protocol,
                queryParameters: queryParameters === null || queryParameters === void 0 ? void 0 : queryParameters.map(queryParameter => queryParameter.bind(scope).queryParameterMatch),
            },
            timeout: renderTimeout(this.timeout),
            retryPolicy: this.retryPolicy ? renderHttpRetryPolicy(this.retryPolicy) : undefined,
        };
        return {
            priority: this.priority,
            httpRouteSpec: this.protocol === shared_interfaces_1.Protocol.HTTP ? httpConfig : undefined,
            http2RouteSpec: this.protocol === shared_interfaces_1.Protocol.HTTP2 ? httpConfig : undefined,
        };
    }
}
class TcpRouteSpec extends RouteSpec {
    constructor(props) {
        super();
        this.weightedTargets = props.weightedTargets;
        this.timeout = props.timeout;
        this.priority = props.priority;
    }
    bind(_scope) {
        return {
            priority: this.priority,
            tcpRouteSpec: {
                action: {
                    weightedTargets: renderWeightedTargets(this.weightedTargets),
                },
                timeout: renderTimeout(this.timeout),
            },
        };
    }
}
class GrpcRouteSpec extends RouteSpec {
    constructor(props) {
        var _b, _c, _d;
        super();
        this.weightedTargets = props.weightedTargets;
        this.match = props.match;
        this.timeout = props.timeout;
        this.priority = props.priority;
        if (props.retryPolicy) {
            const grpcRetryEvents = (_b = props.retryPolicy.grpcRetryEvents) !== null && _b !== void 0 ? _b : [];
            const httpRetryEvents = (_c = props.retryPolicy.httpRetryEvents) !== null && _c !== void 0 ? _c : [];
            const tcpRetryEvents = (_d = props.retryPolicy.tcpRetryEvents) !== null && _d !== void 0 ? _d : [];
            if (grpcRetryEvents.length + httpRetryEvents.length + tcpRetryEvents.length === 0) {
                throw new Error('You must specify one value for at least one of `grpcRetryEvents`, `httpRetryEvents` or `tcpRetryEvents`');
            }
            this.retryPolicy = {
                ...props.retryPolicy,
                grpcRetryEvents: grpcRetryEvents.length > 0 ? grpcRetryEvents : undefined,
                httpRetryEvents: httpRetryEvents.length > 0 ? httpRetryEvents : undefined,
                tcpRetryEvents: tcpRetryEvents.length > 0 ? tcpRetryEvents : undefined,
            };
        }
    }
    bind(scope) {
        const serviceName = this.match.serviceName;
        const methodName = this.match.methodName;
        const metadata = this.match.metadata;
        utils_1.validateGrpcRouteMatch(this.match);
        utils_1.validateGrpcMatchArrayLength(metadata);
        if (methodName && !serviceName) {
            throw new Error('If you specify a method name, you must also specify a service name');
        }
        return {
            priority: this.priority,
            grpcRouteSpec: {
                action: {
                    weightedTargets: renderWeightedTargets(this.weightedTargets),
                },
                match: {
                    serviceName: serviceName,
                    methodName: methodName,
                    metadata: metadata === null || metadata === void 0 ? void 0 : metadata.map(singleMetadata => singleMetadata.bind(scope).headerMatch),
                },
                timeout: renderTimeout(this.timeout),
                retryPolicy: this.retryPolicy ? renderGrpcRetryPolicy(this.retryPolicy) : undefined,
            },
        };
    }
}
/**
 * Utility method to add weighted route targets to an existing route
 */
function renderWeightedTargets(weightedTargets) {
    const renderedTargets = [];
    for (const t of weightedTargets) {
        renderedTargets.push({
            virtualNode: t.virtualNode.virtualNodeName,
            weight: t.weight || 1,
        });
    }
    return renderedTargets;
}
/**
 * Utility method to construct a route timeout object
 */
function renderTimeout(timeout) {
    return timeout
        ? {
            idle: (timeout === null || timeout === void 0 ? void 0 : timeout.idle) !== undefined
                ? {
                    unit: 'ms',
                    value: timeout === null || timeout === void 0 ? void 0 : timeout.idle.toMilliseconds(),
                }
                : undefined,
            perRequest: (timeout === null || timeout === void 0 ? void 0 : timeout.perRequest) !== undefined
                ? {
                    unit: 'ms',
                    value: timeout === null || timeout === void 0 ? void 0 : timeout.perRequest.toMilliseconds(),
                }
                : undefined,
        }
        : undefined;
}
function renderHttpRetryPolicy(retryPolicy) {
    return {
        maxRetries: retryPolicy.retryAttempts,
        perRetryTimeout: {
            unit: 'ms',
            value: retryPolicy.retryTimeout.toMilliseconds(),
        },
        httpRetryEvents: retryPolicy.httpRetryEvents,
        tcpRetryEvents: retryPolicy.tcpRetryEvents,
    };
}
function renderGrpcRetryPolicy(retryPolicy) {
    return {
        ...renderHttpRetryPolicy(retryPolicy),
        grpcRetryEvents: retryPolicy.grpcRetryEvents,
    };
}
//# sourceMappingURL=data:application/json;base64,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