/**
 * \file writer.h
 * \rst
 *
 * Writer
 * ------
 *
 * A writer writes packets to a given arena using the AlephZero transport.
 *
 * \endrst
 */

#ifndef A0_WRITER_H
#define A0_WRITER_H

#include <a0/arena.h>
#include <a0/err.h>
#include <a0/inline.h>
#include <a0/middleware.h>
#include <a0/packet.h>

#ifdef __cplusplus
extern "C" {
#endif

/** \addtogroup WRITER
 *  @{
 */

struct a0_writer_s {
  a0_middleware_t _action;
  a0_writer_t* _next;
};

/// Initializes a writer.
a0_err_t a0_writer_init(a0_writer_t*, a0_arena_t);
/// Closes the given writer.
a0_err_t a0_writer_close(a0_writer_t*);
/// Serializes the given packet into the writer's arena.
a0_err_t a0_writer_write(a0_writer_t*, a0_packet_t);

/// Modifies the writer to include the given middleware.
///
/// The middleware is owned by the writer and will be closed when the writer is closed.
a0_err_t a0_writer_push(a0_writer_t*, a0_middleware_t);

/**
 * Wraps a writer with a middleware as a new writer.
 *
 * The middleware is owned by the new writer and will be closed when the new writer is closed.
 *
 * The new writer does NOT own the old writer. The old writer may be reused.
 * The caller is responsible for closing the old writer AFTER the new writer is closed.
 */
a0_err_t a0_writer_wrap(a0_writer_t* in, a0_middleware_t, a0_writer_t* out);

/** @}*/

#ifdef __cplusplus
}
#endif

#endif  // A0_WRITER_H
