# -*- coding: utf-8 -*-
from setuptools import setup

package_dir = \
{'': 'src'}

packages = \
['ec2_spot_price']

package_data = \
{'': ['*']}

install_requires = \
['boto3>=1.17.12,<2.0.0', 'pandas>=1.2.2,<2.0.0', 'rich>=9.11.1,<10.0.0']

extras_require = \
{':python_version < "3.8"': ['importlib-metadata>=3.7.0,<4.0.0']}

entry_points = \
{'console_scripts': ['ec2_spot_price = ec2_spot_price.ec2_spot_price:main']}

setup_kwargs = {
    'name': 'ec2-spot-price',
    'version': '0.2.2',
    'description': 'ec2-spot-price: retrieve Amazon EC2 spot instance price',
    'long_description': '# ec2-spot-price: Retrieve Amazon EC2 spot instance price\n\nThis Python module provides simple functions and commands to retrieve [Amazon EC2 spot instance price](https://aws.amazon.com/ec2/spot/pricing/) by AWS API.\n\n![](https://raw.githubusercontent.com/susumuota/ec2-spot-price/master/img/demo.gif)\n\n## Install\n\nInstall by `pip`. Or [`pipx`](https://pipxproject.github.io/pipx/) may be convenient to use as a CLI application.\n\n```sh\npip install ec2-spot-price\n```\n\n## Setup\n\nYou need to setup [AWS authentication credentials](https://boto3.amazonaws.com/v1/documentation/api/latest/guide/quickstart.html#configuration) which have permissions to access [`ec2:DescribeSpotPriceHistory`](https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeSpotPriceHistory.html) and [`ec2:DescribeRegions`](https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeRegions.html) APIs.\n\nA simple way to do is to [create a new IAM user](https://docs.aws.amazon.com/IAM/latest/UserGuide/id_users_create.html) and attach `AmazonEC2ReadOnlyAccess` policy directly to the user. Or you can use existent credentials which have permissions to access `ec2:DescribeSpotPriceHistory` and `ec2:DescribeRegions` APIs.\n\n### Create a new IAM user\n\nCreate a new IAM user like the following.\n\n- Go to [IAM console](https://console.aws.amazon.com/iam/home)\n- Click `Users` and `Add user`\n- User name: `myuser` (whatever you want)\n- Access type: `Programmatic access`\n- Click `Next: Permissions`\n- Set permissions: `Attach existing policies directly`\n- Policy name: `AmazonEC2ReadOnlyAccess`\n- Click `Next: Tags`\n- Click `Next: Review`\n- Click `Create user`\n- Click `Download .csv`\n- Click `Close`\n\n#### Note: custom policy\n\nIf you don\'t want to attach `AmazonEC2ReadOnlyAccess` policy, you can [create a new policy](https://docs.aws.amazon.com/IAM/latest/UserGuide/access_policies_create-console.html) that only allows to access `ec2:DescribeSpotPriceHistory` and `ec2:DescribeRegions` APIs. Then attach this policy instead of `AmazonEC2ReadOnlyAccess` policy.\n\n```yaml\n{\n    "Version": "2012-10-17",\n    "Statement": [\n        {\n            "Effect": "Allow",\n            "Action": [\n                "ec2:DescribeSpotPriceHistory",\n                "ec2:DescribeRegions"\n            ],\n            "Resource": "*"\n        }\n    ]\n}\n```\n\n### Edit ~/.aws/credentials\n\nYou can use "[named profile](https://docs.aws.amazon.com/cli/latest/userguide/cli-configure-profiles.html)" to have multiple credentials settings.\n\n```ini\n[myprofile]  # whatever you want, or [default]\naws_access_key_id=[copy from csv]\naws_secret_access_key=[copy from csv]\nregion=us-east-2  # wherever you want\n```\n\nIf you use "named profile", you need to specify `AWS_PROFILE` environment variable.\n\n```sh\nexport AWS_PROFILE=myprofile\n```\n\n\n## Usage\n\n### `ec2_spot_price` command\n\nYou can run `ec2_spot_price` (or `python /path/to/ec2_spot_price.py`) command to retrieve spot instance prices.\n\n`-h` option shows help message.\n\n```sh\n% ec2_spot_price -h\nusage: ec2_spot_price [-h] [-r REGION_NAMES] [-i INSTANCE_TYPES] [-o OS_TYPES]\n                      [-csv]\n\nretrieve Amazon EC2 spot instance price\n\noptional arguments:\n  -h, --help            show this help message and exit\n  -r REGION_NAMES, --region_names REGION_NAMES\n                        filter regions. if "" is specified, retrieve all of\n                        the regions. (default: "us-east-1,us-east-2,us-\n                        west-1,us-west-2")\n  -i INSTANCE_TYPES, --instance_types INSTANCE_TYPES\n                        filter instance types e.g. "g3.4xlarge,p2.xlarge".\n                        (default: retrieve all of the instance types)\n  -o OS_TYPES, --os_types OS_TYPES\n                        filter OS types. (default: "Linux/UNIX")\n  -csv, --csv           output CSV format. (default: False)\n```\n\nYou can specify region names by `-r`, instance types by `-i` and OS types by `-o`. For example, the following command shows table of `us-east-1,us-east-2` regions, `c5.xlarge,c5d.xlarge` instances and `Linux/UNIX` OS (default value).\n\n```sh\n% ec2_spot_price -r "us-east-1,us-east-2" -i "c5.xlarge,c5d.xlarge"\n ─────────────────────────────────────────────────────────────────────── \n  Price      Zone         Instance     OS           Timestamp            \n ─────────────────────────────────────────────────────────────────────── \n  0.038000   us-east-2a   c5.xlarge    Linux/UNIX   2021-02-24 03:59:14  \n  0.038000   us-east-2a   c5d.xlarge   Linux/UNIX   2021-02-24 18:23:40  \n  0.038000   us-east-2b   c5.xlarge    Linux/UNIX   2021-02-24 03:59:14  \n  0.038000   us-east-2b   c5d.xlarge   Linux/UNIX   2021-02-24 20:49:32  \n  0.038000   us-east-2c   c5.xlarge    Linux/UNIX   2021-02-24 03:59:14  \n  0.038000   us-east-2c   c5d.xlarge   Linux/UNIX   2021-02-24 08:06:21  \n  0.066500   us-east-1c   c5.xlarge    Linux/UNIX   2021-02-24 15:56:41  \n  0.066800   us-east-1f   c5d.xlarge   Linux/UNIX   2021-02-24 18:38:06  \n  0.069600   us-east-1a   c5d.xlarge   Linux/UNIX   2021-02-24 17:55:16  \n  0.069700   us-east-1d   c5.xlarge    Linux/UNIX   2021-02-24 19:28:40  \n  0.069900   us-east-1b   c5.xlarge    Linux/UNIX   2021-02-24 21:10:11  \n  0.072100   us-east-1a   c5.xlarge    Linux/UNIX   2021-02-24 16:55:41  \n  0.079600   us-east-1c   c5d.xlarge   Linux/UNIX   2021-02-24 21:01:42  \n  0.079600   us-east-1d   c5d.xlarge   Linux/UNIX   2021-02-24 14:31:55  \n  0.083700   us-east-1f   c5.xlarge    Linux/UNIX   2021-02-24 14:48:41  \n  0.090300   us-east-1b   c5d.xlarge   Linux/UNIX   2021-02-24 03:58:17  \n ─────────────────────────────────────────────────────────────────────── \n```\n\nIn this case, you should use `c5.xlarge` at `us-east-2` region.\n\nAnother example to retrieve all of the spot prices in all regions with verbose option.\n\n```sh\n% ec2_spot_price -r "" -i "" -o "" -csv > spot_prices.csv\n% wc -l spot_prices.csv\n   49822 spot_prices.csv\n% head spot_prices.csv\nSpotPrice,AvailabilityZone,InstanceType,ProductDescription,Timestamp\n0.000800,ap-south-1a,t4g.nano,Linux/UNIX,2021-02-24 17:41:52\n0.000800,ap-south-1a,t4g.nano,SUSE Linux,2021-02-24 17:41:52\n0.000800,ap-south-1b,t4g.nano,Linux/UNIX,2021-02-24 17:41:52\n0.000800,ap-south-1b,t4g.nano,SUSE Linux,2021-02-24 17:41:52\n0.000800,ap-south-1c,t4g.nano,Linux/UNIX,2021-02-24 17:41:52\n0.000800,ap-south-1c,t4g.nano,SUSE Linux,2021-02-24 17:41:52\n0.000900,ap-south-1a,t3a.nano,Linux/UNIX,2021-02-24 17:06:18\n0.000900,ap-south-1a,t3a.nano,SUSE Linux,2021-02-24 17:06:18\n0.000900,ap-south-1b,t3a.nano,Linux/UNIX,2021-02-24 17:06:18\n```\n\nThen open `spot_prices.csv` with spread sheet application like Excel.\n\n\n### `ec2_spot_price` module\n\nThere are three functions.\n\nFunction `get_spot_prices` retrieves spot prices as list.\n\nFunction `print_csv` prints spot prices with CSV format.\n\nFunction `print_table` prints spot prices with table format.\n\n```python\n% python\n>>> import ec2_spot_price as ec2sp\n>>> df = ec2sp.get_spot_prices([\'us-east-1\', \'us-east-2\'], [\'c5.xlarge\', \'c5d.xlarge\'], [\'Linux/UNIX\'])\n>>> len(df)\n16\n>>> ec2sp.print_table(df)\n ─────────────────────────────────────────────────────────────────────── \n  Price      Zone         Instance     OS           Timestamp            \n ─────────────────────────────────────────────────────────────────────── \n  0.038000   us-east-2a   c5.xlarge    Linux/UNIX   2021-02-24 03:59:14  \n  0.038000   us-east-2a   c5d.xlarge   Linux/UNIX   2021-02-24 18:23:40  \n  0.038000   us-east-2b   c5.xlarge    Linux/UNIX   2021-02-24 03:59:14  \n  0.038000   us-east-2b   c5d.xlarge   Linux/UNIX   2021-02-24 20:49:32  \n  0.038000   us-east-2c   c5.xlarge    Linux/UNIX   2021-02-24 03:59:14  \n  0.038000   us-east-2c   c5d.xlarge   Linux/UNIX   2021-02-24 08:06:21  \n  0.066500   us-east-1c   c5.xlarge    Linux/UNIX   2021-02-24 15:56:41  \n  0.066800   us-east-1f   c5d.xlarge   Linux/UNIX   2021-02-24 18:38:06  \n  0.069600   us-east-1a   c5d.xlarge   Linux/UNIX   2021-02-24 17:55:16  \n  0.069700   us-east-1d   c5.xlarge    Linux/UNIX   2021-02-24 19:28:40  \n  0.069900   us-east-1b   c5.xlarge    Linux/UNIX   2021-02-24 21:10:11  \n  0.072100   us-east-1a   c5.xlarge    Linux/UNIX   2021-02-24 16:55:41  \n  0.079600   us-east-1c   c5d.xlarge   Linux/UNIX   2021-02-24 21:01:42  \n  0.079600   us-east-1d   c5d.xlarge   Linux/UNIX   2021-02-24 14:31:55  \n  0.082500   us-east-1f   c5.xlarge    Linux/UNIX   2021-02-24 21:35:41  \n  0.090300   us-east-1b   c5d.xlarge   Linux/UNIX   2021-02-24 03:58:17  \n ─────────────────────────────────────────────────────────────────────── \n>>> ec2sp.print_csv(df)\nSpotPrice,AvailabilityZone,InstanceType,ProductDescription,Timestamp\n0.038000,us-east-2a,c5.xlarge,Linux/UNIX,2021-02-24 03:59:14\n0.038000,us-east-2a,c5d.xlarge,Linux/UNIX,2021-02-24 18:23:40\n0.038000,us-east-2b,c5.xlarge,Linux/UNIX,2021-02-24 03:59:14\n0.038000,us-east-2b,c5d.xlarge,Linux/UNIX,2021-02-24 20:49:32\n0.038000,us-east-2c,c5.xlarge,Linux/UNIX,2021-02-24 03:59:14\n0.038000,us-east-2c,c5d.xlarge,Linux/UNIX,2021-02-24 08:06:21\n0.066500,us-east-1c,c5.xlarge,Linux/UNIX,2021-02-24 15:56:41\n0.066800,us-east-1f,c5d.xlarge,Linux/UNIX,2021-02-24 18:38:06\n0.069600,us-east-1a,c5d.xlarge,Linux/UNIX,2021-02-24 17:55:16\n0.069700,us-east-1d,c5.xlarge,Linux/UNIX,2021-02-24 19:28:40\n0.069900,us-east-1b,c5.xlarge,Linux/UNIX,2021-02-24 21:10:11\n0.072100,us-east-1a,c5.xlarge,Linux/UNIX,2021-02-24 16:55:41\n0.079600,us-east-1c,c5d.xlarge,Linux/UNIX,2021-02-24 21:01:42\n0.079600,us-east-1d,c5d.xlarge,Linux/UNIX,2021-02-24 14:31:55\n0.082500,us-east-1f,c5.xlarge,Linux/UNIX,2021-02-24 21:35:41\n0.090300,us-east-1b,c5d.xlarge,Linux/UNIX,2021-02-24 03:58:17\n```\n\nAnother example to retrieve all of the spot prices in all regions.\nYou can pass spot prices to `pd.DataFrame` and filter them.\n\n```python\n>>> import ec2_spot_price as ec2sp\n>>> df = ec2sp.get_spot_prices([], [], [])\n>>> len(df)\n49817\n>>> df = df.query(\'ProductDescription == "Linux/UNIX"\')\n>>> df = df.drop([\'Timestamp\', \'ProductDescription\'], axis=1)\n>>> df = df.sort_values(by=[\'SpotPrice\', \'AvailabilityZone\', \'InstanceType\'])\n>>> df = df.query(\'InstanceType.str.match("c5.?\\.xlarge")\')\n>>> len(df)\n187\n>>> df.head(20)\n      SpotPrice AvailabilityZone InstanceType\n42061  0.038000       us-east-2a    c5.xlarge\n40121  0.038000       us-east-2a   c5d.xlarge\n40650  0.038000       us-east-2a   c5n.xlarge\n42060  0.038000       us-east-2b    c5.xlarge\n39630  0.038000       us-east-2b   c5d.xlarge\n40649  0.038000       us-east-2b   c5n.xlarge\n42059  0.038000       us-east-2c    c5.xlarge\n41712  0.038000       us-east-2c   c5d.xlarge\n40648  0.038000       us-east-2c   c5n.xlarge\n39716  0.042400       us-east-2c   c5a.xlarge\n39592  0.044100       us-east-2b   c5a.xlarge\n3105   0.052800      ap-south-1c   c5a.xlarge\n3104   0.053200      ap-south-1b   c5a.xlarge\n2505   0.053500      ap-south-1a   c5a.xlarge\n2831   0.054000      ap-south-1a   c5n.xlarge\n3430   0.054000      ap-south-1b   c5d.xlarge\n3768   0.054100      ap-south-1a   c5d.xlarge\n1765   0.054100      ap-south-1b   c5n.xlarge\n1727   0.054100      ap-south-1c   c5d.xlarge\n2981   0.054100      ap-south-1c   c5n.xlarge\n```\n\n## Links\n\n- https://aws.amazon.com/ec2/spot/pricing/\n- https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/using-spot-instances-history.html\n- https://docs.aws.amazon.com/cli/latest/reference/ec2/describe-spot-price-history.html\n- https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeSpotPriceHistory.html\n- https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeRegions.html\n- https://boto3.amazonaws.com/v1/documentation/api/latest/reference/services/ec2.html?highlight=describe%20spot#EC2.Client.describe_spot_price_history\n- https://docs.aws.amazon.com/IAM/latest/UserGuide/id_users_create.html\n- https://boto3.amazonaws.com/v1/documentation/api/latest/guide/quickstart.html#configuration\n\n\n## Author\n\nSusumu OTA\n',
    'author': 'Susumu OTA',
    'author_email': '1632335+susumuota@users.noreply.github.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/susumuota/ec2-spot-price',
    'package_dir': package_dir,
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'extras_require': extras_require,
    'entry_points': entry_points,
    'python_requires': '>=3.7.1,<4.0.0',
}


setup(**setup_kwargs)
