"use strict";
var _a, _b, _c, _d;
Object.defineProperty(exports, "__esModule", { value: true });
exports.VirtualClusterInput = exports.ReleaseLabel = exports.Classification = exports.EmrContainersStartJobRun = void 0;
const jsiiDeprecationWarnings = require("../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const path = require("path");
const iam = require("@aws-cdk/aws-iam");
const lambda = require("@aws-cdk/aws-lambda");
const logs = require("@aws-cdk/aws-logs");
const s3 = require("@aws-cdk/aws-s3");
const sfn = require("@aws-cdk/aws-stepfunctions");
const cdk = require("@aws-cdk/core");
const cr = require("@aws-cdk/custom-resources");
const awscli = require("@aws-cdk/lambda-layer-awscli");
const task_utils_1 = require("../private/task-utils");
/**
 * Starts a job run.
 *
 * A job is a unit of work that you submit to Amazon EMR on EKS for execution.
 * The work performed by the job can be defined by a Spark jar, PySpark script, or SparkSQL query.
 * A job run is an execution of the job on the virtual cluster.
 *
 * @see https://docs.aws.amazon.com/step-functions/latest/dg/connect-emr-eks.html
 */
class EmrContainersStartJobRun extends sfn.TaskStateBase {
    constructor(scope, id, props) {
        var _e, _f;
        super(scope, id, props);
        this.props = props;
        /**
         * Render the EMR Containers ConfigurationProperty as JSON
         */
        this.applicationConfigPropertyToJson = (property) => {
            return {
                Classification: cdk.stringToCloudFormation(property.classification.classificationStatement),
                Properties: property.properties ? cdk.objectToCloudFormation(property.properties) : undefined,
                Configurations: property.nestedConfig ? cdk.listMapper(this.applicationConfigPropertyToJson)(property.nestedConfig) : undefined,
            };
        };
        this.assignLogGroup = () => {
            var _e, _f, _g;
            if ((_e = this.props.monitoring) === null || _e === void 0 ? void 0 : _e.logGroup) {
                return ((_f = this.props.monitoring) === null || _f === void 0 ? void 0 : _f.logGroup);
            }
            else {
                return (((_g = this.props.monitoring) === null || _g === void 0 ? void 0 : _g.logging) ? new logs.LogGroup(this, 'Monitoring Log Group') : undefined);
            }
        };
        this.assignLogBucket = () => {
            var _e, _f, _g;
            if ((_e = this.props.monitoring) === null || _e === void 0 ? void 0 : _e.logBucket) {
                return ((_f = this.props.monitoring) === null || _f === void 0 ? void 0 : _f.logBucket);
            }
            else {
                return (((_g = this.props.monitoring) === null || _g === void 0 ? void 0 : _g.logging) ? new s3.Bucket(this, 'Monitoring Bucket') : undefined);
            }
        };
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_stepfunctions_tasks_EmrContainersStartJobRunProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.constructor);
            }
            throw error;
        }
        this.integrationPattern = (_e = props.integrationPattern) !== null && _e !== void 0 ? _e : sfn.IntegrationPattern.RUN_JOB;
        task_utils_1.validatePatternSupported(this.integrationPattern, EmrContainersStartJobRun.SUPPORTED_INTEGRATION_PATTERNS);
        if (this.props.applicationConfig) {
            this.validateAppConfig(this.props.applicationConfig);
        }
        if (this.props.jobDriver.sparkSubmitJobDriver) {
            this.validateSparkSubmitJobDriver(props.jobDriver.sparkSubmitJobDriver);
        }
        if (this.props.executionRole === undefined
            && sfn.JsonPath.isEncodedJsonPath(props.virtualCluster.id)) {
            throw new Error('Execution role cannot be undefined when the virtual cluster ID is not a concrete value. Provide an execution role with the correct trust policy');
        }
        this.logGroup = this.assignLogGroup();
        this.logBucket = this.assignLogBucket();
        this.role = (_f = this.props.executionRole) !== null && _f !== void 0 ? _f : this.createJobExecutionRole();
        this.grantPrincipal = this.role;
        this.grantMonitoringPolicies();
        this.taskPolicies = this.createPolicyStatements();
    }
    /**
     * @internal
     */
    _renderTask() {
        var _e, _f, _g, _h, _j;
        return {
            Resource: task_utils_1.integrationResourceArn('emr-containers', 'startJobRun', this.integrationPattern),
            Parameters: sfn.FieldUtils.renderObject({
                VirtualClusterId: this.props.virtualCluster.id,
                Name: this.props.jobName,
                ExecutionRoleArn: this.role.roleArn,
                ReleaseLabel: this.props.releaseLabel.label,
                JobDriver: {
                    SparkSubmitJobDriver: {
                        EntryPoint: (_e = this.props.jobDriver.sparkSubmitJobDriver) === null || _e === void 0 ? void 0 : _e.entryPoint.value,
                        EntryPointArguments: (_g = (_f = this.props.jobDriver.sparkSubmitJobDriver) === null || _f === void 0 ? void 0 : _f.entryPointArguments) === null || _g === void 0 ? void 0 : _g.value,
                        SparkSubmitParameters: (_h = this.props.jobDriver.sparkSubmitJobDriver) === null || _h === void 0 ? void 0 : _h.sparkSubmitParameters,
                    },
                },
                ConfigurationOverrides: {
                    ApplicationConfiguration: cdk.listMapper(this.applicationConfigPropertyToJson)(this.props.applicationConfig),
                    MonitoringConfiguration: {
                        CloudWatchMonitoringConfiguration: this.logGroup ? {
                            LogGroupName: this.logGroup.logGroupName,
                            LogStreamNamePrefix: this.props.monitoring.logStreamNamePrefix,
                        } : undefined,
                        PersistentAppUI: (((_j = this.props.monitoring) === null || _j === void 0 ? void 0 : _j.persistentAppUI) === false)
                            ? 'DISABLED'
                            : 'ENABLED',
                        S3MonitoringConfiguration: this.logBucket ? {
                            LogUri: this.logBucket.s3UrlForObject(),
                        } : undefined,
                    },
                },
                Tags: this.props.tags,
            }),
        };
    }
    validateAppConfigPropertiesLength(appConfig) {
        if ((appConfig === null || appConfig === void 0 ? void 0 : appConfig.properties) === undefined) {
            return;
        }
        else if (Object.keys(appConfig.properties).length > 100) {
            throw new Error(`Application configuration properties must have 100 or fewer entries. Received ${Object.keys(appConfig.properties).length}`);
        }
    }
    validatePropertiesNestedAppConfigBothNotUndefined(appConfig) {
        if ((appConfig === null || appConfig === void 0 ? void 0 : appConfig.properties) === undefined && (appConfig === null || appConfig === void 0 ? void 0 : appConfig.nestedConfig) === undefined) {
            throw new Error('Application configuration must have either properties or nested app configurations defined.');
        }
    }
    validateAppConfig(config) {
        if (config === undefined) {
            return;
        }
        else if (config.length > 100) {
            throw new Error(`Application configuration array must have 100 or fewer entries. Received ${config.length}`);
        }
        else {
            config.forEach(element => this.validateAppConfig(element.nestedConfig));
            config.forEach(element => this.validateAppConfigPropertiesLength(element));
            config.forEach(element => this.validatePropertiesNestedAppConfigBothNotUndefined(element));
        }
    }
    isArrayOfStrings(value) {
        return Array.isArray(value) && value.every(item => typeof item === 'string');
    }
    validateEntryPointArguments(entryPointArguments) {
        if (typeof entryPointArguments.value === 'string' && !sfn.JsonPath.isEncodedJsonPath(entryPointArguments.value)) {
            throw new Error(`Entry point arguments must be a string array or encoded JSON path, but received a non JSON path string');
      .`);
        }
        if (!this.isArrayOfStrings(entryPointArguments.value)) {
            throw new Error(`Entry point arguments must be a string array or encoded JSON path but received ${typeof entryPointArguments.value}.`);
        }
    }
    validateEntryPointArgumentsLength(entryPointArguments) {
        if (this.isArrayOfStrings(entryPointArguments.value)
            && (entryPointArguments.value.length > 10280 || entryPointArguments.value.length < 1)) {
            throw new Error(`Entry point arguments must be a string array between 1 and 10280 in length. Received ${entryPointArguments.value.length}.`);
        }
    }
    validateSparkSubmitParametersLength(sparkSubmitParameters) {
        if (sparkSubmitParameters.length > 102400 || sparkSubmitParameters.length < 1) {
            throw new Error(`Spark submit parameters must be between 1 and 102400 characters in length. Received ${sparkSubmitParameters.length}.`);
        }
    }
    validateEntryPoint(entryPoint) {
        if (!sfn.JsonPath.isEncodedJsonPath(entryPoint.value) && (entryPoint.value.length > 256 || entryPoint.value.length < 1)) {
            throw new Error(`Entry point must be between 1 and 256 characters in length. Received ${entryPoint.value.length}.`);
        }
    }
    validateSparkSubmitJobDriver(driver) {
        this.validateEntryPoint(driver.entryPoint);
        if (driver.entryPointArguments) {
            this.validateEntryPointArguments(driver.entryPointArguments);
            this.validateEntryPointArgumentsLength(driver.entryPointArguments);
        }
        if (driver.sparkSubmitParameters) {
            this.validateSparkSubmitParametersLength(driver.sparkSubmitParameters);
        }
    }
    // https://docs.aws.amazon.com/emr/latest/EMR-on-EKS-DevelopmentGuide/creating-job-execution-role.html
    createJobExecutionRole() {
        var _e, _f, _g;
        const jobExecutionRole = new iam.Role(this, 'Job-Execution-Role', {
            assumedBy: new iam.CompositePrincipal(new iam.ServicePrincipal('emr-containers.amazonaws.com'), new iam.ServicePrincipal('states.amazonaws.com')),
        });
        (_e = this.logBucket) === null || _e === void 0 ? void 0 : _e.grantReadWrite(jobExecutionRole);
        (_f = this.logGroup) === null || _f === void 0 ? void 0 : _f.grantWrite(jobExecutionRole);
        (_g = this.logGroup) === null || _g === void 0 ? void 0 : _g.grant(jobExecutionRole, 'logs:DescribeLogStreams');
        jobExecutionRole.addToPrincipalPolicy(new iam.PolicyStatement({
            resources: [
                'arn:aws:logs:*:*:*',
            ],
            actions: [
                'logs:DescribeLogGroups',
            ],
        }));
        this.updateRoleTrustPolicy(jobExecutionRole);
        return jobExecutionRole;
    }
    grantMonitoringPolicies() {
        var _e, _f, _g;
        (_e = this.logBucket) === null || _e === void 0 ? void 0 : _e.grantReadWrite(this.role);
        (_f = this.logGroup) === null || _f === void 0 ? void 0 : _f.grantWrite(this.role);
        (_g = this.logGroup) === null || _g === void 0 ? void 0 : _g.grant(this.role, 'logs:DescribeLogStreams');
        this.role.addToPrincipalPolicy(new iam.PolicyStatement({
            resources: [
                'arn:aws:logs:*:*:*',
            ],
            actions: [
                'logs:DescribeLogGroups',
            ],
        }));
    }
    /**
     * If an execution role is not provided by user, the automatically generated job execution role must create a trust relationship
     * between itself and the identity of the EMR managed service account in order to run jobs on the Kubernetes namespace.
     *
     * This cannot occur if the user provided virtualClusterId is within an encoded JSON path.
     *
     * The trust relationship can be created by updating the trust policy of the job execution role.
     *
     * @param role the automatically generated job execution role
     */
    updateRoleTrustPolicy(role) {
        const eksClusterInfo = new cr.AwsCustomResource(this, 'GetEksClusterInfo', {
            onCreate: {
                service: 'EMRcontainers',
                action: 'describeVirtualCluster',
                parameters: {
                    id: this.props.virtualCluster.id,
                },
                outputPaths: ['virtualCluster.containerProvider.info.eksInfo.namespace', 'virtualCluster.containerProvider.id'],
                physicalResourceId: cr.PhysicalResourceId.of('id'),
            },
            policy: cr.AwsCustomResourcePolicy.fromSdkCalls({
                resources: cr.AwsCustomResourcePolicy.ANY_RESOURCE,
            }),
        });
        /* We make use of custom resources to call update-roll-trust-policy as this command is only available through
         * aws cli because this is only used during the initial setup and is not available through the sdk.
         * https://awscli.amazonaws.com/v2/documentation/api/latest/reference/emr-containers/update-role-trust-policy.html
         * Commands available through SDK: https://docs.aws.amazon.com/AWSJavaScriptSDK/latest/AWS/EMRcontainers.html
         * Commands available through CLI: https://awscli.amazonaws.com/v2/documentation/api/latest/reference/emr-containers/index.html
        */
        const cliLayer = new awscli.AwsCliLayer(this, 'awsclilayer');
        const shellCliLambda = new lambda.SingletonFunction(this, 'Call Update-Role-Trust-Policy', {
            uuid: '8693BB64-9689-44B6-9AAF-B0CC9EB8757C',
            runtime: lambda.Runtime.PYTHON_3_6,
            handler: 'index.handler',
            code: lambda.Code.fromAsset(path.join(__dirname, 'utils/role-policy')),
            timeout: cdk.Duration.seconds(30),
            memorySize: 256,
            layers: [cliLayer],
        });
        shellCliLambda.addToRolePolicy(new iam.PolicyStatement({
            resources: [
                cdk.Stack.of(this).formatArn({
                    service: 'eks',
                    resource: 'cluster',
                    resourceName: eksClusterInfo.getResponseField('virtualCluster.containerProvider.id'),
                }),
            ],
            actions: [
                'eks:DescribeCluster',
            ],
        }));
        shellCliLambda.addToRolePolicy(new iam.PolicyStatement({
            resources: [role.roleArn],
            actions: [
                'iam:GetRole',
                'iam:UpdateAssumeRolePolicy',
            ],
        }));
        const provider = new cr.Provider(this, 'CustomResourceProvider', {
            onEventHandler: shellCliLambda,
        });
        new cdk.CustomResource(this, 'Custom Resource', {
            properties: {
                eksNamespace: eksClusterInfo.getResponseField('virtualCluster.containerProvider.info.eksInfo.namespace'),
                eksClusterId: eksClusterInfo.getResponseField('virtualCluster.containerProvider.id'),
                roleName: role.roleName,
            },
            serviceToken: provider.serviceToken,
        });
    }
    createPolicyStatements() {
        const policyStatements = [
            new iam.PolicyStatement({
                resources: [
                    cdk.Stack.of(this).formatArn({
                        arnFormat: cdk.ArnFormat.SLASH_RESOURCE_SLASH_RESOURCE_NAME,
                        service: 'emr-containers',
                        resource: 'virtualclusters',
                        resourceName: sfn.JsonPath.isEncodedJsonPath(this.props.virtualCluster.id) ? '*' : this.props.virtualCluster.id,
                    }),
                ],
                actions: ['emr-containers:StartJobRun'],
                conditions: {
                    StringEquals: {
                        'emr-containers:ExecutionRoleArn': this.role.roleArn,
                    },
                },
            }),
        ];
        if (this.integrationPattern === sfn.IntegrationPattern.RUN_JOB) {
            policyStatements.push(new iam.PolicyStatement({
                resources: [
                    cdk.Stack.of(this).formatArn({
                        arnFormat: cdk.ArnFormat.SLASH_RESOURCE_SLASH_RESOURCE_NAME,
                        service: 'emr-containers',
                        resource: 'virtualclusters',
                        resourceName: sfn.JsonPath.isEncodedJsonPath(this.props.virtualCluster.id) ? '*' : `${this.props.virtualCluster.id}/jobruns/*`,
                    }),
                ],
                actions: [
                    'emr-containers:DescribeJobRun',
                    'emr-containers:CancelJobRun',
                ],
            }));
        }
        return policyStatements;
    }
}
exports.EmrContainersStartJobRun = EmrContainersStartJobRun;
_a = JSII_RTTI_SYMBOL_1;
EmrContainersStartJobRun[_a] = { fqn: "@aws-cdk/aws-stepfunctions-tasks.EmrContainersStartJobRun", version: "1.154.0" };
EmrContainersStartJobRun.SUPPORTED_INTEGRATION_PATTERNS = [
    sfn.IntegrationPattern.REQUEST_RESPONSE,
    sfn.IntegrationPattern.RUN_JOB,
];
/**
 * The classification within a EMR Containers application configuration.
 * Class can be extended to add other classifications.
 * For example, new Classification('xxx-yyy');
 */
class Classification {
    /**
     * Creates a new Classification
     *
     * @param classificationStatement A literal string in case a new EMR classification is released, if not already defined.
     */
    constructor(classificationStatement) {
        this.classificationStatement = classificationStatement;
    }
}
exports.Classification = Classification;
_b = JSII_RTTI_SYMBOL_1;
Classification[_b] = { fqn: "@aws-cdk/aws-stepfunctions-tasks.Classification", version: "1.154.0" };
/**
 * Sets the maximizeResourceAllocation property to true or false.
 * When true, Amazon EMR automatically configures spark-defaults properties based on cluster hardware configuration.
 *
 * For more info:
 * @see https://docs.aws.amazon.com/emr/latest/ReleaseGuide/emr-spark-configure.html#emr-spark-maximizeresourceallocation
 */
Classification.SPARK = new Classification('spark');
/**
 * Sets values in the spark-defaults.conf file.
 *
 * For more info:
 * @see https://spark.apache.org/docs/latest/configuration.html
 */
Classification.SPARK_DEFAULTS = new Classification('spark-defaults');
/**
 * Sets values in the spark-env.sh file.
 *
 * For more info:
 * @see https://spark.apache.org/docs/latest/configuration.html#environment-variables
 */
Classification.SPARK_ENV = new Classification('spark-env');
/**
 * Sets values in the hive-site.xml for Spark.
 */
Classification.SPARK_HIVE_SITE = new Classification('spark-hive-site');
/**
 * Sets values in the log4j.properties file.
 *
 * For more settings and info:
 * @see https://github.com/apache/spark/blob/master/conf/log4j.properties.template
 */
Classification.SPARK_LOG4J = new Classification('spark-log4j');
/**
 * Sets values in the metrics.properties file.
 *
 * For more settings and info:
 * @see https://github.com/apache/spark/blob/master/conf/metrics.properties.template
 */
Classification.SPARK_METRICS = new Classification('spark-metrics');
/**
 * The Amazon EMR release version to use for the job run.
 *
 * Can be extended to include new EMR releases
 *
 * For example, `new ReleaseLabel('emr-x.xx.x-latest');`
 */
class ReleaseLabel {
    /**
     * Initializes the label string.
     *
     * @param label A literal string that contains the release-version ex. 'emr-x.x.x-latest'
     */
    constructor(label) {
        this.label = label;
    }
}
exports.ReleaseLabel = ReleaseLabel;
_c = JSII_RTTI_SYMBOL_1;
ReleaseLabel[_c] = { fqn: "@aws-cdk/aws-stepfunctions-tasks.ReleaseLabel", version: "1.154.0" };
/**
 * EMR Release version 5.32.0
 */
ReleaseLabel.EMR_5_32_0 = new ReleaseLabel('emr-5.32.0-latest');
/**
 * EMR Release version 5.33.0
 */
ReleaseLabel.EMR_5_33_0 = new ReleaseLabel('emr-5.33.0-latest');
/**
 * EMR Release version 6.2.0
 */
ReleaseLabel.EMR_6_2_0 = new ReleaseLabel('emr-6.2.0-latest');
/**
 * EMR Release version 6.3.0
 */
ReleaseLabel.EMR_6_3_0 = new ReleaseLabel('emr-6.3.0-latest');
/**
 * Class that returns a virtual cluster's id depending on input type
 */
class VirtualClusterInput {
    /**
     * Initializes the virtual cluster ID.
     *
     * @param id The VirtualCluster Id
     */
    constructor(id) {
        this.id = id;
    }
    /**
     * Input for a virtualClusterId from a Task Input
     */
    static fromTaskInput(taskInput) {
        return new VirtualClusterInput(taskInput.value);
    }
    /**
     * Input for virtualClusterId from a literal string
     */
    static fromVirtualClusterId(virtualClusterId) {
        return new VirtualClusterInput(virtualClusterId);
    }
}
exports.VirtualClusterInput = VirtualClusterInput;
_d = JSII_RTTI_SYMBOL_1;
VirtualClusterInput[_d] = { fqn: "@aws-cdk/aws-stepfunctions-tasks.VirtualClusterInput", version: "1.154.0" };
//# sourceMappingURL=data:application/json;base64,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