"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.BatchSubmitJob = void 0;
const jsiiDeprecationWarnings = require("../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const iam = require("@aws-cdk/aws-iam");
const sfn = require("@aws-cdk/aws-stepfunctions");
const core_1 = require("@aws-cdk/core");
const task_utils_1 = require("../private/task-utils");
/**
 * Task to submits an AWS Batch job from a job definition.
 *
 * @see https://docs.aws.amazon.com/step-functions/latest/dg/connect-batch.html
 */
class BatchSubmitJob extends sfn.TaskStateBase {
    constructor(scope, id, props) {
        var _b, _c;
        super(scope, id, props);
        this.props = props;
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_stepfunctions_tasks_BatchSubmitJobProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.constructor);
            }
            throw error;
        }
        this.integrationPattern = (_b = props.integrationPattern) !== null && _b !== void 0 ? _b : sfn.IntegrationPattern.RUN_JOB;
        task_utils_1.validatePatternSupported(this.integrationPattern, BatchSubmitJob.SUPPORTED_INTEGRATION_PATTERNS);
        // validate arraySize limits
        core_1.withResolved(props.arraySize, (arraySize) => {
            if (arraySize !== undefined && (arraySize < 2 || arraySize > 10000)) {
                throw new Error(`arraySize must be between 2 and 10,000. Received ${arraySize}.`);
            }
        });
        // validate dependency size
        if (props.dependsOn && props.dependsOn.length > 20) {
            throw new Error(`dependencies must be 20 or less. Received ${props.dependsOn.length}.`);
        }
        // validate attempts
        core_1.withResolved(props.attempts, (attempts) => {
            if (attempts !== undefined && (attempts < 1 || attempts > 10)) {
                throw new Error(`attempts must be between 1 and 10. Received ${attempts}.`);
            }
        });
        // validate timeout
        props.timeout !== undefined && core_1.withResolved(props.timeout.toSeconds(), (timeout) => {
            if (timeout < 60) {
                throw new Error(`attempt duration must be greater than 60 seconds. Received ${timeout} seconds.`);
            }
        });
        // This is required since environment variables must not start with AWS_BATCH;
        // this naming convention is reserved for variables that are set by the AWS Batch service.
        if ((_c = props.containerOverrides) === null || _c === void 0 ? void 0 : _c.environment) {
            Object.keys(props.containerOverrides.environment).forEach(key => {
                if (key.match(/^AWS_BATCH/)) {
                    throw new Error(`Invalid environment variable name: ${key}. Environment variable names starting with 'AWS_BATCH' are reserved.`);
                }
            });
        }
        this.taskPolicies = this.configurePolicyStatements();
    }
    /**
     * @internal
     */
    _renderTask() {
        var _b;
        return {
            Resource: task_utils_1.integrationResourceArn('batch', 'submitJob', this.integrationPattern),
            Parameters: sfn.FieldUtils.renderObject({
                JobDefinition: this.props.jobDefinitionArn,
                JobName: this.props.jobName,
                JobQueue: this.props.jobQueueArn,
                Parameters: (_b = this.props.payload) === null || _b === void 0 ? void 0 : _b.value,
                ArrayProperties: this.props.arraySize !== undefined
                    ? { Size: this.props.arraySize }
                    : undefined,
                ContainerOverrides: this.props.containerOverrides
                    ? this.configureContainerOverrides(this.props.containerOverrides)
                    : undefined,
                DependsOn: this.props.dependsOn
                    ? this.props.dependsOn.map(jobDependency => ({
                        JobId: jobDependency.jobId,
                        Type: jobDependency.type,
                    }))
                    : undefined,
                RetryStrategy: this.props.attempts !== undefined
                    ? { Attempts: this.props.attempts }
                    : undefined,
                Timeout: this.props.timeout
                    ? { AttemptDurationSeconds: this.props.timeout.toSeconds() }
                    : undefined,
            }),
            TimeoutSeconds: undefined,
        };
    }
    configurePolicyStatements() {
        return [
            // Resource level access control for job-definition requires revision which batch does not support yet
            // Using the alternative permissions as mentioned here:
            // https://docs.aws.amazon.com/batch/latest/userguide/batch-supported-iam-actions-resources.html
            new iam.PolicyStatement({
                resources: [
                    core_1.Stack.of(this).formatArn({
                        service: 'batch',
                        resource: 'job-definition',
                        resourceName: '*',
                    }),
                    this.props.jobQueueArn,
                ],
                actions: ['batch:SubmitJob'],
            }),
            new iam.PolicyStatement({
                resources: [
                    core_1.Stack.of(this).formatArn({
                        service: 'events',
                        resource: 'rule/StepFunctionsGetEventsForBatchJobsRule',
                    }),
                ],
                actions: ['events:PutTargets', 'events:PutRule', 'events:DescribeRule'],
            }),
        ];
    }
    configureContainerOverrides(containerOverrides) {
        var _b;
        let environment;
        if (containerOverrides.environment) {
            environment = Object.entries(containerOverrides.environment).map(([key, value]) => ({
                Name: key,
                Value: value,
            }));
        }
        let resources = [];
        if (containerOverrides.gpuCount) {
            resources.push({
                Type: 'GPU',
                Value: `${containerOverrides.gpuCount}`,
            });
        }
        if (containerOverrides.memory) {
            resources.push({
                Type: 'MEMORY',
                Value: `${containerOverrides.memory.toMebibytes()}`,
            });
        }
        if (containerOverrides.vcpus) {
            resources.push({
                Type: 'VCPU',
                Value: `${containerOverrides.vcpus}`,
            });
        }
        return {
            Command: containerOverrides.command,
            Environment: environment,
            InstanceType: (_b = containerOverrides.instanceType) === null || _b === void 0 ? void 0 : _b.toString(),
            ResourceRequirements: resources.length ? resources : undefined,
        };
    }
}
exports.BatchSubmitJob = BatchSubmitJob;
_a = JSII_RTTI_SYMBOL_1;
BatchSubmitJob[_a] = { fqn: "@aws-cdk/aws-stepfunctions-tasks.BatchSubmitJob", version: "1.154.0" };
BatchSubmitJob.SUPPORTED_INTEGRATION_PATTERNS = [
    sfn.IntegrationPattern.REQUEST_RESPONSE,
    sfn.IntegrationPattern.RUN_JOB,
];
//# sourceMappingURL=data:application/json;base64,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