"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.SkillEndpointPermission = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const constructs_1 = require("constructs");
const aws_lambda_1 = require("aws-cdk-lib/aws-lambda");
const skill_1 = require("./skill");
const custom_resources_1 = require("aws-cdk-lib/custom-resources");
const aws_iam_1 = require("aws-cdk-lib/aws-iam");
/**
 * Class for configuring and managing Skill Endpoint Permissions.
 */
class SkillEndpointPermission extends constructs_1.Construct {
    /**
     * Creates an instance of the Skill Endpoint Permission.
     * @param scope - The construct scope.
     * @param id - The construct ID.
     * @param props - The Skill Endpoint properties.
     */
    constructor(scope, id, props) {
        super(scope, id);
        this.principals = {
            [skill_1.SkillType.CUSTOM]: 'alexa-appkit.amazon.com',
            [skill_1.SkillType.SMART_HOME]: 'alexa-connectedhome.amazon.com',
        };
        this.skillType = props.skillType;
        this.permission = new aws_lambda_1.CfnPermission(this, 'InitialSkillPermission', {
            functionName: props.handler.functionArn,
            principal: this.principals[props.skillType],
            action: 'lambda:InvokeFunction',
        });
        this.policy = custom_resources_1.AwsCustomResourcePolicy.fromStatements([
            new aws_iam_1.PolicyStatement({
                actions: ['lambda:RemovePermission', 'lambda:AddPermission'],
                resources: [props.handler.functionArn],
            }),
        ]);
        this.handler = props.handler;
    }
    /**
     * Configures a Skill Endpoint Permission for a specific skill.
     * This replaces the initially created permission with a new permission checking for the SkillId.
     * @param scope - The construct scope.
     * @param id - The construct ID.
     * @param skill - The skill to configure the permission for.
     * @returns An IDependable object representing the configured permission.
     */
    configureSkillId(scope, id, skill) {
        if (this.skillType !== skill.skillType) {
            throw new Error(`Permission was intended for skillType ${this.skillType}, but skill has ${skill.skillType}`);
        }
        const parent = new constructs_1.Construct(scope, id);
        new custom_resources_1.AwsCustomResource(parent, 'RemovePermission', {
            policy: this.policy,
            onCreate: this.removePermissionCall(parent, skill),
            onDelete: this.addPermissionCall(parent),
            installLatestAwsSdk: false,
        });
        new custom_resources_1.AwsCustomResource(parent, 'AddPermission', {
            policy: this.policy,
            onCreate: this.addPermissionCall(parent, skill),
            onDelete: this.removePermissionCall(parent, skill),
            installLatestAwsSdk: false,
        });
        parent.node.addDependency(this.permission, this.handler);
        return parent;
    }
    removePermissionCall(scope, skill) {
        return {
            service: 'Lambda',
            action: 'removePermission',
            parameters: {
                FunctionName: this.handler.functionArn,
                StatementId: this.permission.attrId,
            },
            physicalResourceId: custom_resources_1.PhysicalResourceId.of(`RemovePermission-${scope.node.addr}-${skill.skillId}`),
        };
    }
    addPermissionCall(scope, skill) {
        return {
            service: 'Lambda',
            action: 'addPermission',
            parameters: {
                FunctionName: this.handler.functionArn,
                StatementId: this.permission.attrId,
                Principal: this.principals[skill?.skillType ?? this.skillType],
                Action: 'lambda:InvokeFunction',
                EventSourceToken: skill?.skillId,
            },
            physicalResourceId: custom_resources_1.PhysicalResourceId.of(`AddPermission-${scope.node.addr}-${skill?.skillId ?? 'no-skill'}`),
        };
    }
}
exports.SkillEndpointPermission = SkillEndpointPermission;
_a = JSII_RTTI_SYMBOL_1;
SkillEndpointPermission[_a] = { fqn: "cdk-skill-management.SkillEndpointPermission", version: "1.0.11" };
//# sourceMappingURL=data:application/json;base64,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