# Copyright 2019 British Broadcasting Corporation
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

import re

from six import string_types
from flask import abort
from flask import request as flask_request
from socket import getfqdn
from fnmatch import fnmatch
from authlib.jose import JWTClaims
from authlib.jose.errors import InvalidClaimError, MissingClaimError

from ..logger import Logger as defaultLogger

logger = defaultLogger("token_validator")


def generate_claims_options(nmos_api):
    """Helper function to generate the required claims within a valid JWT"""

    claims_options = {
        "iss": {"essential": True},
        "sub": {"essential": True},
        "aud": {"essential": True},
        "exp": {"essential": True},
        "iat": {"essential": False},
        "nbf": {"essential": False},
        "client_id": {"essential": True},
        "scope": {"essential": False},
    }
    if nmos_api:
        if isinstance(nmos_api, str):
            # Essential: False - as a scope claim is sufficient to access base resources (e.g. /x-nmos/query)
            claims_options["x-nmos-{}".format(nmos_api)] = {"essential": False}
        elif isinstance(nmos_api, list):
            for api_name in nmos_api:
                claims_options["x-nmos-{}".format(api_name)] = {"essential": False}
        claims_options["scope"]["value"] = nmos_api
    return claims_options


class JWTClaimsValidator(JWTClaims):

    def __init__(self, payload, header, options=None, params=None):
        super(JWTClaimsValidator, self).__init__(payload, header, options, params)

    def validate_iss(self):  # placeholder
        super(JWTClaimsValidator, self).validate_iss()

    def validate_sub(self):  # placeholder
        super(JWTClaimsValidator, self).validate_sub()

    def validate_aud(self):
        super(JWTClaimsValidator, self).validate_aud()
        claim_name = "aud"
        fqdn = getfqdn()  # Fully qualified domain name of Resource Server
        actual_claim_value = self.get(claim_name)  # actual claim value in JWT
        if isinstance(actual_claim_value, string_types):
            actual_claim_value = [actual_claim_value]

        for aud in actual_claim_value:
            if fnmatch(fqdn, aud):
                return
        raise InvalidClaimError(
            "Hostname '{}' does not match aud claim value of '{}'".format(fqdn, actual_claim_value))

    def validate_nmos(self, request=None):

        if not request:
            request = flask_request

        # actual 'x-nmos-*' claims in JWT
        nmos_token_claims = {
            key: val for key, val in self.items() if key.startswith("x-nmos")
        }

        # set of valid claims generated by generate_claims_options()
        valid_claim_option = {
            key: val for key, val in self.options.items() if key.startswith("x-nmos")
        }

        if not valid_claim_option:  # No options given for validation
            logger.writeWarning("No x-nmos claim value given for validation in claims options")
            return

        def _validate_permissions_object(claim_name):

            access_permission_object = nmos_token_claims.get(claim_name)

            if not access_permission_object:
                raise InvalidClaimError("{}. Missing from token claims: '{}'.".format(
                    claim_name, list(nmos_token_claims.keys())))

            if request.method in ["GET", "OPTIONS", "HEAD"]:
                access_right = "read"
            elif request.method in ["PUT", "POST", "PATCH", "DELETE"]:
                access_right = "write"
            else:
                abort(405)

            url_access_list = access_permission_object.get(access_right)
            if not url_access_list:
                raise InvalidClaimError(
                    "{}. No entry in permissions object for '{}'".format(claim_name, access_right))

            pattern = re.compile(r'/x-nmos/[a-z]+/v[0-9]+\.[0-9]+/(.*)')  # Capture path after namespace
            sub_path = pattern.match(request.path).group(1)
            for wildcard_url in url_access_list:
                if fnmatch(sub_path, wildcard_url):
                    return

            raise InvalidClaimError("{}. No matching paths in token claim: {} for URL path: '{}'".format(
                claim_name, url_access_list, sub_path))

        # base resources do not require authorization
        if request.path in ['/', '/x-nmos', '/x-nmos/']:
            return

        # if request path is /x-nmos/api_name[/version/], then permit access, providing claim or scope is present
        path_regex = re.search(r'^/x-nmos/([a-z]+)/?(v[0-9]+\.[0-9]+)?/?$', request.path)
        if path_regex:
            # api_name = path_regex.group(1)  # e.g. query, registration, etc.
            scope_option = self.options.get("scope").get("value")
            # scope in claim_options must be in token or x-nmos-<api_name> claim must exist
            if scope_option not in self.get("scope") and \
                    "x-nmos-{}".format(scope_option) not in list(nmos_token_claims.keys()):
                raise MissingClaimError(
                    "Missing {} from scope claim or 'x-nmos-{}' claim from token".format(scope_option, scope_option))
            else:
                return

        # Check that all x-nmos claims in "actual_claims" are in "valid_claims"
        if not all(claim_name in nmos_token_claims.keys() for claim_name in valid_claim_option.keys()):
            raise MissingClaimError("One of: {} is missing from 'x-nmos' token claims: '{}'".format(
                list(valid_claim_option.keys()), list(nmos_token_claims.keys())))

        for claim_name in valid_claim_option.keys():
            _validate_permissions_object(claim_name)

    def validate(self, req, now=None, leeway=0):
        super(JWTClaimsValidator, self).validate()
        self.validate_nmos(req)
