"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.FargateProfile = void 0;
const ec2 = require("@aws-cdk/aws-ec2");
const iam = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const consts_1 = require("./cluster-resource-handler/consts");
const cluster_resource_provider_1 = require("./cluster-resource-provider");
// v2 - keep this import as a separate section to reduce merge conflict when forward merging with the v2 branch.
// eslint-disable-next-line
const core_2 = require("@aws-cdk/core");
/**
 * Fargate profiles allows an administrator to declare which pods run on Fargate.
 *
 * This declaration is done through the profile’s selectors. Each
 * profile can have up to five selectors that contain a namespace and optional
 * labels. You must define a namespace for every selector. The label field
 * consists of multiple optional key-value pairs. Pods that match a selector (by
 * matching a namespace for the selector and all of the labels specified in the
 * selector) are scheduled on Fargate. If a namespace selector is defined
 * without any labels, Amazon EKS will attempt to schedule all pods that run in
 * that namespace onto Fargate using the profile. If a to-be-scheduled pod
 * matches any of the selectors in the Fargate profile, then that pod is
 * scheduled on Fargate.
 *
 * If a pod matches multiple Fargate profiles, Amazon EKS picks one of the
 * matches at random. In this case, you can specify which profile a pod should
 * use by adding the following Kubernetes label to the pod specification:
 * eks.amazonaws.com/fargate-profile: profile_name. However, the pod must still
 * match a selector in that profile in order to be scheduled onto Fargate.
 *
 * @stability stable
 */
class FargateProfile extends core_2.Construct {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        var _a, _b;
        super(scope, id);
        const provider = cluster_resource_provider_1.ClusterResourceProvider.getOrCreate(this, {
            adminRole: props.cluster.adminRole,
        });
        this.podExecutionRole = (_a = props.podExecutionRole) !== null && _a !== void 0 ? _a : new iam.Role(this, 'PodExecutionRole', {
            assumedBy: new iam.ServicePrincipal('eks-fargate-pods.amazonaws.com'),
            managedPolicies: [iam.ManagedPolicy.fromAwsManagedPolicyName('AmazonEKSFargatePodExecutionRolePolicy')],
        });
        this.podExecutionRole.grantPassRole(props.cluster.adminRole);
        let subnets;
        if (props.vpc) {
            const selection = (_b = props.subnetSelection) !== null && _b !== void 0 ? _b : { subnetType: ec2.SubnetType.PRIVATE };
            subnets = props.vpc.selectSubnets(selection).subnetIds;
        }
        if (props.selectors.length < 1) {
            throw new Error('Fargate profile requires at least one selector');
        }
        if (props.selectors.length > 5) {
            throw new Error('Fargate profile supports up to five selectors');
        }
        this.tags = new core_1.TagManager(core_1.TagType.MAP, 'AWS::EKS::FargateProfile');
        const resource = new core_1.CustomResource(this, 'Resource', {
            serviceToken: provider.serviceToken,
            resourceType: consts_1.FARGATE_PROFILE_RESOURCE_TYPE,
            properties: {
                AssumeRoleArn: props.cluster.adminRole.roleArn,
                Config: {
                    clusterName: props.cluster.clusterName,
                    fargateProfileName: props.fargateProfileName,
                    podExecutionRoleArn: this.podExecutionRole.roleArn,
                    selectors: props.selectors,
                    subnets,
                    tags: core_1.Lazy.any({ produce: () => this.tags.renderTags() }),
                },
            },
        });
        this.fargateProfileArn = resource.getAttString('fargateProfileArn');
        this.fargateProfileName = resource.ref;
        // Fargate profiles must be created sequentially. If other profile(s) already
        // exist on the same cluster, create a dependency to force sequential creation.
        const clusterFargateProfiles = props.cluster._attachFargateProfile(this);
        if (clusterFargateProfiles.length > 1) {
            const previousProfile = clusterFargateProfiles[clusterFargateProfiles.length - 2];
            resource.node.addDependency(previousProfile);
        }
        // map the fargate pod execution role to the relevant groups in rbac
        // see https://github.com/aws/aws-cdk/issues/7981
        props.cluster.awsAuth.addRoleMapping(this.podExecutionRole, {
            username: 'system:node:{{SessionName}}',
            groups: [
                'system:bootstrappers',
                'system:nodes',
                'system:node-proxier',
            ],
        });
    }
}
exports.FargateProfile = FargateProfile;
//# sourceMappingURL=data:application/json;base64,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