# -*- coding: utf-8 -*-
from setuptools import setup

modules = \
['aiomisc_pytest']
install_requires = \
['aiomisc>=17', 'pytest>=7.2.1,<8.0.0']

entry_points = \
{'pytest11': ['aiomisc = aiomisc_pytest']}

setup_kwargs = {
    'name': 'aiomisc-pytest',
    'version': '1.1.1',
    'description': 'pytest integration for aiomisc',
    'long_description': 'aiomisc pytest plugin\n=====================\n\nThis package contains a plugin for pytest.\n\n## Basic usage\n\nSimple usage example:\n\n```python\nasync def test_sample(event_loop):\n    f = event_loop.crete_future()\n    event_loop.call_soon(f.set_result, True)\n    assert await f\n```\n\nasynchronous fixture example:\n\n\n```python\n\nimport asyncio\nimport pytest\n\n\n@pytest.fixture\nasync def my_fixture(loop):\n    await asyncio.sleep(0)\n\n    # Requires python 3.6+\n    yield\n```\n\nIn case you have to save an instance of an async fixture between tests,\nthe wrong solution is just changing the fixture scope.\nBut why it wouldn\'t work? That\'s because, in the base scenario, the `loop`\nfixture creates a new event loop instance per test which will be closed after\ntest teardown. When you have to use an async fixture any caller of\n`asyncio.get_event_loop()` will get the current event loop instance which\nwill be closed and the next test will run in another event loop.\nSo the solution is to redefine the `loop` fixture with the required scope\nand custom fixture with the required scope.\n\n```python\nimport asyncio\nimport pytest\nfrom aiomisc import entrypoint\n\n\n@pytest.fixture(scope=\'module\')\ndef loop():\n    with entrypoint() as loop:\n        asyncio.set_event_loop(loop)\n        yield loop\n\n\n@pytest.fixture(scope=\'module\')\nasync def sample_fixture(loop):\n    yield 1\n\n\nLOOP_ID = None\n\n\nasync def test_using_fixture(sample_fixture):\n    global LOOP_ID\n    LOOP_ID = id(asyncio.get_event_loop())\n    assert sample_fixture == 1\n\n\nasync def test_not_using_fixture(loop):\n    assert id(loop) == LOOP_ID\n```\n\n## pytest markers\n\nPackage contains some useful markers for pytest:\n\n* `catch_loop_exceptions` - uncaught event loop exceptions will failling test.\n* `forbid_get_event_loop` - forbids call `asyncio.get_event_loop`\n  during test case.\n\n```python\nimport asyncio\nimport pytest\n\n\n@pytest.mark.forbid_get_event_loop\nasync def test_with_get_loop():\n    def switch_context():\n        loop = asyncio.get_event_loop()\n        future = loop.create_future()\n        loop.call_soon(future.set_result, True)\n        return future\n\n    with pytest.raises(Exception):\n        await switch_context()\n\n\n# Test will be failed\n@pytest.mark.catch_loop_exceptions\nasync def test_with_errors(loop):\n    async def fail():\n        # switch context\n        await asyncio.sleep(0)\n        raise Exception()\n\n    loop.create_task(fail())\n    await asyncio.sleep(0.1)\n    return\n```\n\n## Passing default context\n\n```python\nimport pytest\n\n@pytest.fixture\ndef default_context():\n    return {\n        \'foo\': \'bar\',\n        \'bar\': \'foo\',\n    }\n```\n\n## Testing services\n\nRedefine `services` fixture in your test module:\n\n```python\nimport aiomisc\nimport pytest\n\n\nclass SimpleServie(aiomisc.Service):\n    async def start(self) -> None:\n        pass\n\n    \n@pytest.fixture\ndef services():\n    return [SimpleServie()]\n```\n\n## Event loop policy overriding\n\n```python\nimport asyncio\nimport pytest\nimport tokio\nimport uvloop\n\npolicy_ids = (\'uvloop\', \'asyncio\', \'tokio\')\npolicies = (uvloop.EventLoopPolicy(),\n            asyncio.DefaultEventLoopPolicy(),\n            tokio.EventLoopPolicy())\n\n@pytest.fixture(params=policies, ids=policy_ids)\ndef event_loop_policy(request):\n    return request.param\n```\n\n## Thread pool overriding\n\n```python\nimport pytest\nfrom aiomisc.thread_pool import ThreadPoolExecutor\nimport concurrent.futures\n\nthread_pool_ids = (\'aiomisc pool\', \'default pool\')\nthread_pool_implementation = (ThreadPoolExecutor,\n                              concurrent.futures.ThreadPoolExecutor)\n\n\n@pytest.fixture(params=thread_pool_implementation, ids=thread_pool_ids)\ndef thread_pool_executor(request):\n    return request.param\n```\n\n## entrypoint arguments\n\n```python\nimport pytest\n\n@pytest.fixture\ndef entrypoint_kwargs() -> dict:\n    return dict(log_config=False)\n```\n\n## aiohttp test client\n\n\n```python\n\nimport pytest\nfrom myapp.services.rest import REST\n\n\n@pytest.fixture\ndef rest_port(aiomisc_unused_port_factory):\n    return aiomisc_unused_port_factory()\n\n\n@pytest.fixture\ndef rest_service(rest_port):\n    return REST(port=rest_port)\n\n\n@pytest.fixture\ndef services(rest_service):\n    return [rest_service]\n\n\n@pytest.fixture\ndef api_client(api_service):\n    test_srv = TestServer(\n        app=rest_service.app,\n        port=arguments.port,\n    )\n\n    return TestClient(test_srv)\n\n...\n```\n\n\n## TCPProxy\n\nSimple TCP proxy for emulate network problems. Available as fixture `tcp_proxy`\n\n\nExamples:\n\n```python\nimport asyncio\nimport time\n\nimport pytest\n\nimport aiomisc\n\n\nclass EchoServer(aiomisc.service.TCPServer):\n    async def handle_client(\n            self, reader: asyncio.StreamReader,\n            writer: asyncio.StreamWriter\n    ):\n        chunk = await reader.read(65534)\n        while chunk:\n            writer.write(chunk)\n            chunk = await reader.read(65534)\n\n        writer.close()\n        await writer.wait_closed()\n\n\n@pytest.fixture()\ndef server_port(aiomisc_unused_port_factory) -> int:\n    return aiomisc_unused_port_factory()\n\n\n@pytest.fixture()\ndef services(server_port, localhost):\n    return [EchoServer(port=server_port, address=localhost)]\n\n\n@pytest.fixture()\nasync def proxy(tcp_proxy, localhost, server_port):\n    async with tcp_proxy(localhost, server_port) as proxy:\n        yield proxy\n\n\nasync def test_proxy_client_close(proxy):\n    reader, writer = await proxy.create_client()\n    payload = b"Hello world"\n\n    writer.write(payload)\n    response = await asyncio.wait_for(reader.read(1024), timeout=1)\n\n    assert response == payload\n\n    assert not reader.at_eof()\n    await proxy.disconnect_all()\n\n    assert await asyncio.wait_for(reader.read(), timeout=1) == b""\n    assert reader.at_eof()\n\n\nasync def test_proxy_client_slow(proxy):\n    read_delay = 0.1\n    write_delay = 0.2\n\n    # Emulation of asymmetric and slow ISP\n    with proxy.slowdown(read_delay, write_delay):\n        reader, writer = await proxy.create_client()\n        payload = b"Hello world"\n\n        delta = -time.monotonic()\n\n        writer.write(payload)\n        await asyncio.wait_for(reader.read(1024), timeout=2)\n\n        delta += time.monotonic()\n\n        assert delta >= read_delay + write_delay\n\n\nasync def test_proxy_client_with_processor(proxy):\n    processed_request = b"Never say hello"\n\n    # Patching protocol functions\n    proxy.set_content_processors(\n        # Process data from client to server\n        lambda _: processed_request,\n\n        # Process data from server to client\n        lambda chunk: chunk[::-1],\n    )\n\n    reader, writer = await proxy.create_client()\n    writer.write(b\'nevermind\')\n\n    response = await reader.read(16)\n\n    assert response == processed_request[::-1]\n```\n',
    'author': 'Dmitry Orlov',
    'author_email': 'me@mosquito.su',
    'maintainer': 'None',
    'maintainer_email': 'None',
    'url': 'None',
    'py_modules': modules,
    'install_requires': install_requires,
    'entry_points': entry_points,
    'python_requires': '>=3.7,<4.0',
}


setup(**setup_kwargs)
