# AUTOGENERATED! DO NOT EDIT! File to edit: nbs/21_reward.ipynb (unless otherwise specified).

__all__ = ['Reward', 'RewardCallback', 'GenericRewardCallback', 'RewardModification', 'NoveltyReward',
           'ContrastiveReward']

# Cell

from ..imports import *
from ..core import *
from .callback import *
from ..torch_imports import *
from ..torch_core import *

# Cell

class Reward():
    '''
    Reward - wrapper for `reward_function`. Handles batching
    and value lookup

    Inputs:

    - `reward_function Callable`: function with the format
    `rewards = reward_function(samples)`

    - `weight float`: weight to scale rewards

    - `bs Optional[int]`: if given, samples will be batched into
    chunks of size `bs` and sent to `reward_function` as batches

    - `log bool`: if True, keeps aa lookup table of
    `sample : reward` values to avoid repeat computation
    '''
    def __init__(self, reward_function, weight=1, bs=None, log=True):

        self.reward_function = reward_function
        self.weight = weight
        self.bs = bs
        self.score_log = {}
        self.log = log

    def load_data(self, samples, values):
        for i in range(len(samples)):
            self.score_log[samples[i]] = values[i]

    def __call__(self, samples):

        rewards = np.array([0. for i in samples])

        to_score = []
        to_score_idxs = []

        for i in range(len(samples)):

            if self.log:
                if samples[i] in self.score_log:
                    rewards[i] = self.score_log[samples[i]]
                else:
                    to_score.append(samples[i])
                    to_score_idxs.append(i)

            else:
                to_score.append(samples[i])
                to_score_idxs.append(i)

        if to_score:
            new_rewards = self.compute_batched_reward(to_score)

            for i in range(len(to_score)):
                batch_idx = to_score_idxs[i]
                reward = new_rewards[i]
                rewards[batch_idx] = reward

                if self.log:
                    self.score_log[to_score[i]] = reward

        rewards = to_device(torch.tensor(rewards).float()).squeeze()
        rewards = rewards * self.weight

        return rewards

    def _compute_reward(self, samples):
        return self.reward_function(samples)

    def compute_batched_reward(self, samples):
        if self.bs is not None:
            sample_chunks = chunk_list(samples, self.bs)
            rewards = []
            for chunk in sample_chunks:
                rewards_iter = self._compute_reward(chunk)
                if isinstance(rewards_iter, torch.Tensor):
                    rewards_iter = rewards_iter.detach().cpu()

                rewards += list(rewards_iter)

        else:
            rewards = self._compute_reward(samples)
            if isinstance(rewards, torch.Tensor):
                rewards = rewards.detach().cpu()

        return rewards

    def add_data_to_log(self, samples, rewards):
        for i in range(len(samples)):
            self.score_log[samples[i]] = rewards[i]


# Cell

class RewardCallback(Callback):
    '''
    RewardCallback - callback wrapper for `Reward`
    used during `compute_reward` event

    Inputs:

    - `reward Reward`: reward to use

    - `name str`: reward name

    - `sample_name str`: sample name to grab from
    `BatchState` to send to `reward`

    - `order int`: callback order

    - `track bool`: if metrics should be tracked
    from this callback
    '''
    def __init__(self, reward, name, sample_name='samples',
                order=10, track=True):
        super().__init__(name=name, order=order)

        self.reward = reward
        self.sample_name = sample_name
        self.track = track

    def setup(self):
        log = self.environment.log
        log.add_log(self.name)
        if self.track:
            log.add_metric(self.name)

    def compute_reward(self):
        env = self.environment
        batch_state = env.batch_state
        samples = batch_state[self.sample_name]

        if samples:
            rewards = self.reward(samples)
        else:
            rewards = to_device(torch.tensor(0.))

        batch_state.rewards += rewards
        batch_state[self.name] = rewards

        if self.track:
            env.log.update_metric(self.name, rewards.mean().detach().cpu().numpy())

# Cell

class GenericRewardCallback(RewardCallback):
    '''
    GenericRewardCallback - generic reward
    wrapper

    Inputs:

    - `reward Callable`: reward function. Reward
    will be passed the entire batch state

    - `name str`: reward name

    - `order int`: callback order

    - `track bool`: if metrics should be tracked
    from this callback
    '''
    def __init__(self, reward, name,
                order=10, track=True):
        super().__init__(reward,
                         name,
                         order=order,
                         track=track
                        )

    def compute_reward(self):
        env = self.environment
        batch_state = env.batch_state
        rewards = self.reward(batch_state)

        batch_state.rewards += rewards
        batch_state[self.name] = rewards

        if self.track:
            env.log.update_metric(self.name, rewards.mean().detach().cpu().numpy())

# Cell

class RewardModification(Callback):
    '''
    RewardModification - callback wrapper for `Reward`
    used during `reward_modification` event

    Inputs:

    - `reward Reward`: reward to use

    - `name str`: reward name

    - `sample_name str`: sample name to grab from
    `BatchState` to send to `reward`

    - `order int`: callback order

    - `track bool`: if metrics should be tracked
    from this callback
    '''
    def __init__(self, reward, name, sample_name='samples',
                order=10, track=True):
        super().__init__(name=name, order=order)

        self.reward = reward
        self.sample_name = sample_name
        self.track = track

    def setup(self):
        log = self.environment.log
        log.add_log(self.name)
        if self.track:
            log.add_metric(self.name)

    def reward_modification(self):
        env = self.environment
        batch_state = env.batch_state
        samples = batch_state[self.sample_name]

        if samples:
            rewards = self.reward(samples)
        else:
            rewards = 0.

        batch_state.rewards_final += rewards
        batch_state[self.name] = rewards

        if self.track:
            env.log.update_metric(self.name, rewards.mean().detach().cpu().numpy())

# Cell

class NoveltyReward(Callback):
    '''
    NoveltyReward - gives a reward bonus
    for new samples. Rewards are given a
    bonus of `weight`

    Inputs:

    - `weight float`: novelty score weight

    - `track bool`: if metrics should be tracked
    from this callback
    '''
    def __init__(self, weight=1., track=True):
        super().__init__(name='novel')

        self.weight = weight
        self.track = track

    def setup(self):
        log = self.environment.log
        log.add_log(self.name)
        if self.track:
            log.add_metric(self.name)

    def reward_modification(self):
        env = self.environment
        batch_state = env.batch_state
        samples = batch_state.samples

        df = env.log.df
        new = (~pd.Series(samples).isin(df.samples)).values

        rewards = np.array([float(i) for i in new])*self.weight
        rewards = to_device(torch.from_numpy(rewards).float())

        batch_state.rewards_final += rewards
        batch_state[self.name] = rewards

        if self.track:
            env.log.update_metric(self.name, rewards.mean().detach().cpu().numpy())

# Cell

class ContrastiveReward(RewardCallback):
    '''
    ContrastiveReward - contrastive wrapper for
    reward callbacks

    Inputs:

    - `base_reward RewardCallback`: base reward callback

    - `max_score Optional[float]`: maximum possible score.
    If given, contrastive rewards are scaled following
    `reward = (target_reward - source_reward)/(max_reward - source_reward)`
    '''
    def __init__(self, base_reward, max_score=None):
        super().__init__(reward = base_reward.reward,
                         name = base_reward.name,
                         sample_name = base_reward.sample_name,
                         order = base_reward.order,
                         track = base_reward.track)

        self.base_reward = base_reward
        self.max_score = max_score

    def setup(self):
        self.base_reward.environment = self.environment

    def __call__(self, event_name):

        event = getattr(self, event_name, None)

        if event is not None:
            output = event()
        else:
            output = None

        if not event_name=='compute_reward':
            _ = self.base_reward(event_name)

        return output

    def compute_and_clean(self, samples):
        rewards = self.base_reward.reward(samples)
        if isinstance(rewards, torch.Tensor):
            rewards = rewards.detach().cpu()

        rewards = np.array(rewards)
        return rewards

    def _compute_reward(self, samples):
        source_samples = [i[0] for i in samples]
        target_samples = [i[1] for i in samples]

        source_rewards = self.compute_and_clean(source_samples)
        target_rewards = self.compute_and_clean(target_samples)

        rewards = target_rewards - source_rewards
        if self.max_score is not None:
            rewards = rewards / (self.max_score-source_rewards)

        rewards = to_device(torch.from_numpy(rewards).float())

        return rewards

    def compute_reward(self):
        env = self.environment
        batch_state = env.batch_state
        samples = batch_state[self.sample_name]

        rewards = self._compute_reward(samples)

        batch_state.rewards += rewards
        batch_state[self.name] = rewards

        if self.track:
            env.log.update_metric(self.name, rewards.mean().detach().cpu().numpy())
