"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const core_1 = require("@aws-cdk/core");
const ec2 = require("../lib");
let app;
let stack;
beforeEach(() => {
    app = new core_1.App();
    stack = new core_1.Stack(app, 'Stack', {
        env: { account: '1234', region: 'testregion' },
    });
});
test('can make and use a Windows image', () => {
    // WHEN
    const image = new ec2.GenericWindowsImage({
        testregion: 'ami-1234',
    });
    // THEN
    const details = image.getImage(stack);
    expect(details.imageId).toEqual('ami-1234');
    expect(details.osType).toEqual(ec2.OperatingSystemType.WINDOWS);
});
test('WindowsImage retains userdata if given', () => {
    // WHEN
    const ud = ec2.UserData.forWindows();
    const image = new ec2.GenericWindowsImage({
        testregion: 'ami-1234',
    }, {
        userData: ud,
    });
    // THEN
    const details = image.getImage(stack);
    expect(details.userData).toEqual(ud);
});
test('WindowsImage creates UserData if not given', () => {
    // WHEN
    const image = new ec2.GenericWindowsImage({
        testregion: 'ami-1234',
    });
    // THEN
    const details = image.getImage(stack);
    expect(isWindowsUserData(details.userData)).toBeTruthy();
});
test('LookupMachineImage default search', () => {
    // GIVEN
    // WHEN
    new ec2.LookupMachineImage({ name: 'bla*', owners: ['amazon'] }).getImage(stack);
    // THEN
    const missing = app.synth().manifest.missing || [];
    expect(missing).toEqual([
        {
            key: 'ami:account=1234:filters.image-type.0=machine:filters.name.0=bla*:filters.state.0=available:owners.0=amazon:region=testregion',
            props: {
                account: '1234',
                region: 'testregion',
                owners: ['amazon'],
                filters: {
                    'name': ['bla*'],
                    'state': ['available'],
                    'image-type': ['machine'],
                },
            },
            provider: 'ami',
        },
    ]);
});
test('LookupMachineImage creates correct type of UserData', () => {
    // WHEN
    const linuxDetails = new ec2.LookupMachineImage({ name: 'bla*', owners: ['amazon'] }).getImage(stack);
    const windowsDetails = new ec2.LookupMachineImage({ name: 'bla*', owners: ['amazon'], windows: true }).getImage(stack);
    // THEN
    expect(isWindowsUserData(windowsDetails.userData)).toBeTruthy();
    expect(isLinuxUserData(linuxDetails.userData)).toBeTruthy();
});
function isWindowsUserData(ud) {
    return ud.render().indexOf('powershell') > -1;
}
function isLinuxUserData(ud) {
    return ud.render().indexOf('bash') > -1;
}
//# sourceMappingURL=data:application/json;base64,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