"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.EmrAddStep = exports.ActionOnFailure = void 0;
const jsiiDeprecationWarnings = require("../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const iam = require("@aws-cdk/aws-iam");
const sfn = require("@aws-cdk/aws-stepfunctions");
const core_1 = require("@aws-cdk/core");
const task_utils_1 = require("../private/task-utils");
/**
 * The action to take when the cluster step fails.
 * @see https://docs.aws.amazon.com/emr/latest/APIReference/API_StepConfig.html
 *
 * Here, they are named as TERMINATE_JOB_FLOW, TERMINATE_CLUSTER, CANCEL_AND_WAIT, and CONTINUE respectively.
 *
 * @default CONTINUE
 *
 */
var ActionOnFailure;
(function (ActionOnFailure) {
    /**
     * Terminate the Cluster on Step Failure
     */
    ActionOnFailure["TERMINATE_CLUSTER"] = "TERMINATE_CLUSTER";
    /**
     * Cancel Step execution and enter WAITING state
     */
    ActionOnFailure["CANCEL_AND_WAIT"] = "CANCEL_AND_WAIT";
    /**
     * Continue to the next Step
     */
    ActionOnFailure["CONTINUE"] = "CONTINUE";
})(ActionOnFailure = exports.ActionOnFailure || (exports.ActionOnFailure = {}));
/**
 * A Step Functions Task to add a Step to an EMR Cluster
 *
 * The StepConfiguration is defined as Parameters in the state machine definition.
 *
 * OUTPUT: the StepId
 *
 */
class EmrAddStep extends sfn.TaskStateBase {
    constructor(scope, id, props) {
        var _b, _c;
        super(scope, id, props);
        this.props = props;
        jsiiDeprecationWarnings._aws_cdk_aws_stepfunctions_tasks_EmrAddStepProps(props);
        this.actionOnFailure = (_b = props.actionOnFailure) !== null && _b !== void 0 ? _b : ActionOnFailure.CONTINUE;
        this.integrationPattern = (_c = props.integrationPattern) !== null && _c !== void 0 ? _c : sfn.IntegrationPattern.RUN_JOB;
        task_utils_1.validatePatternSupported(this.integrationPattern, EmrAddStep.SUPPORTED_INTEGRATION_PATTERNS);
        this.taskPolicies = this.createPolicyStatements();
    }
    /**
     * @internal
     */
    _renderTask() {
        return {
            Resource: task_utils_1.integrationResourceArn('elasticmapreduce', 'addStep', this.integrationPattern),
            Parameters: sfn.FieldUtils.renderObject({
                ClusterId: this.props.clusterId,
                Step: {
                    Name: this.props.name,
                    ActionOnFailure: this.actionOnFailure.valueOf(),
                    HadoopJarStep: {
                        Jar: this.props.jar,
                        MainClass: this.props.mainClass,
                        Args: this.props.args,
                        Properties: (this.props.properties === undefined) ?
                            undefined :
                            Object.entries(this.props.properties).map(kv => ({
                                Key: kv[0],
                                Value: kv[1],
                            })),
                    },
                },
            }),
        };
    }
    /**
     * This generates the PolicyStatements required by the Task to call AddStep.
     */
    createPolicyStatements() {
        const stack = core_1.Stack.of(this);
        const policyStatements = [
            new iam.PolicyStatement({
                actions: [
                    'elasticmapreduce:AddJobFlowSteps',
                    'elasticmapreduce:DescribeStep',
                    'elasticmapreduce:CancelSteps',
                ],
                resources: [
                    stack.formatArn({
                        service: 'elasticmapreduce',
                        resource: 'cluster',
                        resourceName: '*',
                    }),
                ],
            }),
        ];
        if (this.integrationPattern === sfn.IntegrationPattern.RUN_JOB) {
            policyStatements.push(new iam.PolicyStatement({
                actions: ['events:PutTargets', 'events:PutRule', 'events:DescribeRule'],
                resources: [stack.formatArn({
                        service: 'events',
                        resource: 'rule',
                        resourceName: 'StepFunctionsGetEventForEMRAddJobFlowStepsRule',
                    })],
            }));
        }
        return policyStatements;
    }
}
exports.EmrAddStep = EmrAddStep;
_a = JSII_RTTI_SYMBOL_1;
EmrAddStep[_a] = { fqn: "@aws-cdk/aws-stepfunctions-tasks.EmrAddStep", version: "1.148.0" };
EmrAddStep.SUPPORTED_INTEGRATION_PATTERNS = [
    sfn.IntegrationPattern.REQUEST_RESPONSE,
    sfn.IntegrationPattern.RUN_JOB,
];
//# sourceMappingURL=data:application/json;base64,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