from __future__ import unicode_literals
import sys
import argparse
import logging

from boardgamegeek.api import BoardGameGeek, BoardGameGeekNetworkAPI, HOT_ITEM_CHOICES

log = logging.getLogger("boardgamegeek")
log_fmt = "[%(levelname)s] %(message)s"

def brief_game_stats(game):

    try:
        desc = '''"{}",{},{}-{},{},{},{},{},"{}","{}"'''.format(game.name, game.year,
               game.min_players, game.max_players,
               game.playing_time,
               game.rating_average, game.rating_average_weight, game.users_rated,
               " / ".join(game.categories).lower(),
               " / ".join(game.mechanics).lower())

        print >>sys.stderr, "{}".format(desc)
        sys.stdout.flush()
    except Exception as e:
        pass

    return

    log.info("Name        : {}".format(game.name))
    log.info("Categories  : {}".format(game.categories))
    log.info("Mechanics   : {}".format(game.mechanics))
    log.info("Players     : {}-{}".format(game.min_players, game.max_players))
    log.info("Age         : {}".format(game.min_age))
    log.info("Play time   : {}".format(game.playing_time))
    log.info("Game weight : {}".format(game.rating_average_weight))
    log.info("Score       : {}".format(game.rating_average))
    log.info("Votes       : {}".format(game.users_rated))
    log.info("MY SCORE    : {}".format(my_score))


def main():
    p = argparse.ArgumentParser(prog="boardgamegeek")

    p.add_argument("-u", "--user", help="Query by user name")
    p.add_argument("-g", "--game", help="Query by game name")
    p.add_argument("--most-recent", help="get the most recent game when querying by name (default)", action="store_true")
    p.add_argument("--most-popular", help="get the most popular (top ranked) game when querying by name", action="store_true")

    p.add_argument("-i", "--id", help="Query by game id", type=int)
    p.add_argument("--game-stats", help="Return brief statistics about the game")
    p.add_argument("-G", "--guild", help="Query by guild id")
    p.add_argument("-c", "--collection", help="Query user's collection")
    p.add_argument("-p", "--plays", help="Query user's play list")
    p.add_argument("-P", "--plays-by-game", help="Query a game's plays")
    p.add_argument("-H", "--hot-items", help="List all hot items by type", choices=HOT_ITEM_CHOICES)
    p.add_argument("-S", "--search", help="search and return results")
    p.add_argument("--debug", action="store_true")
    p.add_argument("--retries", help="number of retries to perform in case of timeout or API HTTP 202 code",
                   type=int,
                   default=5)
    p.add_argument("--timeout", help="Timeout for API operations", type=int, default=10)

    args = p.parse_args()

    # configure logging
    if args.debug:
        log_level = logging.DEBUG
    else:
        # make requests shush
        logging.getLogger("requests").setLevel(logging.WARNING)
        log_level = logging.INFO

    log.setLevel(log_level)
    stdout = logging.StreamHandler()
    stdout.setLevel(log_level)

    fmt = logging.Formatter(log_fmt)
    stdout.setFormatter(fmt)
    log.addHandler(stdout)

    def progress_cb(items, total):
        log.debug("fetching items: {}% complete".format(items*100/total))

    if not any([args.user, args.game, args.id, args.guild, args.collection,
                args.plays, args.plays_by_game, args.hot_items, args.search]):
        p.error("no action specified!")

    bgg = BoardGameGeek(timeout=args.timeout, retries=args.retries)

    if args.user:
        user = bgg.user(args.user, progress=progress_cb)
        if user:
            user._format(log)

    # query by game id
    if args.id:
        game = bgg.game(game_id=args.id)
        if game:
            game._format(log)

    # query by game name
    if args.game:
        # fetch the most popular
        if args.most_popular:
            game = bgg.game(args.game, choose="best-rank")
        else:
        # fetch the most recent one
            game = bgg.game(args.game, choose="recent")
        if game:
            game._format(log)

    if args.game_stats:
        game = bgg.game(args.game_stats)
        if game:
            brief_game_stats(game)

    if args.guild:
        guild = bgg.guild(args.guild, progress=progress_cb)
        if guild:
            guild._format(log)

    if args.collection:
        collection = bgg.collection(args.collection)
        if collection:
            collection._format(log)

    if args.plays:
        plays = bgg.plays(name=args.plays, progress=progress_cb)
        if plays:
            plays._format(log)

    if args.plays_by_game:
        try:
            game_id = int(args.plays_by_game)
        except:
            game_id = bgg.get_game_id(args.plays_by_game)

        plays = bgg.plays(game_id=game_id, progress=progress_cb)
        if plays:
            plays._format(log)

    if args.hot_items:
        hot_items = bgg.hot_items(args.hot_items)
        for item in hot_items:
            item._format(log)
            log.info("")

    if args.search:
        results = bgg.search(args.search)
        if results:
            for r in results:
                r._format(log)
                log.info("")

if __name__ == "__main__":
    main()
