"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.GitHubRunners = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cdk = require("aws-cdk-lib");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_lambda_1 = require("aws-cdk-lib/aws-lambda");
const constructs_1 = require("constructs");
const codebuild_1 = require("./providers/codebuild");
const fargate_1 = require("./providers/fargate");
const lambda_1 = require("./providers/lambda");
const secrets_1 = require("./secrets");
const utils_1 = require("./utils");
const webhook_1 = require("./webhook");
/**
 * Create all the required infrastructure to provide self-hosted GitHub runners. It creates a webhook, secrets, and a step function to orchestrate all runs. Secrets are not automatically filled. See README.md for instructions on how to setup GitHub integration.
 *
 * By default, this will create a runner provider of each available type with the defaults. This is good enough for the initial setup stage when you just want to get GitHub integration working.
 *
 * ```typescript
 * new GitHubRunners(this, 'runners');
 * ```
 *
 * Usually you'd want to configure the runner providers so the runners can run in a certain VPC or have certain permissions.
 *
 * ```typescript
 * const vpc = ec2.Vpc.fromLookup(this, 'vpc', { vpcId: 'vpc-1234567' });
 * const runnerSg = new ec2.SecurityGroup(this, 'runner security group', { vpc: vpc });
 * const dbSg = ec2.SecurityGroup.fromSecurityGroupId(this, 'database security group', 'sg-1234567');
 * const bucket = new s3.Bucket(this, 'runner bucket');
 *
 * // create a custom CodeBuild provider
 * const myProvider = new CodeBuildRunner(
 *   this, 'codebuild runner',
 *   {
 *      label: 'my-codebuild',
 *      vpc: vpc,
 *      securityGroup: runnerSg,
 *   },
 * );
 * // grant some permissions to the provider
 * bucket.grantReadWrite(myProvider);
 * dbSg.connections.allowFrom(runnerSg, ec2.Port.tcp(3306), 'allow runners to connect to MySQL database');
 *
 * // create the runner infrastructure
 * new GitHubRunners(
 *   this,
 *   'runners',
 *   {
 *     providers: [myProvider],
 *   }
 * );
 * ```
 */
class GitHubRunners extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.extraLambdaEnv = {};
        this.props = props ?? {};
        this.secrets = new secrets_1.Secrets(this, 'Secrets');
        this.extraLambdaProps = {
            vpc: this.props.vpc,
            vpcSubnets: this.props.vpcSubnets,
            allowPublicSubnet: this.props.allowPublicSubnet,
            securityGroups: this.props.securityGroup ? [this.props.securityGroup] : undefined,
            layers: this.props.extraCertificates ? [new aws_cdk_lib_1.aws_lambda.LayerVersion(scope, 'Certificate Layer', {
                    description: 'Layer containing GitHub Enterprise Server certificate for cdk-github-runners',
                    code: aws_cdk_lib_1.aws_lambda.Code.fromAsset(this.props.extraCertificates),
                })] : undefined,
        };
        if (this.props.extraCertificates) {
            this.extraLambdaEnv.NODE_EXTRA_CA_CERTS = '/opt/certs.pem';
        }
        if (this.props.providers) {
            this.providers = this.props.providers;
        }
        else {
            this.providers = [
                new codebuild_1.CodeBuildRunner(this, 'CodeBuild', {}),
                new lambda_1.LambdaRunner(this, 'Lambda', {}),
                new fargate_1.FargateRunner(this, 'Fargate', {}),
            ];
        }
        this.orchestrator = this.stateMachine(props);
        this.webhook = new webhook_1.GithubWebhookHandler(this, 'Webhook Handler', {
            orchestrator: this.orchestrator,
            secrets: this.secrets,
        });
        this.setupUrl = this.setupFunction();
        this.statusFunction();
    }
    stateMachine(props) {
        const tokenRetrieverTask = new aws_cdk_lib_1.aws_stepfunctions_tasks.LambdaInvoke(this, 'Get Runner Token', {
            lambdaFunction: this.tokenRetriever(),
            payloadResponseOnly: true,
            resultPath: '$.runner',
        });
        let deleteRunnerFunction = this.deleteRunner();
        const deleteRunnerTask = new aws_cdk_lib_1.aws_stepfunctions_tasks.LambdaInvoke(this, 'Delete Runner', {
            lambdaFunction: deleteRunnerFunction,
            payloadResponseOnly: true,
            resultPath: '$.delete',
            payload: aws_cdk_lib_1.aws_stepfunctions.TaskInput.fromObject({
                runnerName: aws_cdk_lib_1.aws_stepfunctions.JsonPath.stringAt('$$.Execution.Name'),
                owner: aws_cdk_lib_1.aws_stepfunctions.JsonPath.stringAt('$.owner'),
                repo: aws_cdk_lib_1.aws_stepfunctions.JsonPath.stringAt('$.repo'),
                runId: aws_cdk_lib_1.aws_stepfunctions.JsonPath.stringAt('$.runId'),
                installationId: aws_cdk_lib_1.aws_stepfunctions.JsonPath.stringAt('$.installationId'),
                idleOnly: false,
            }),
        });
        deleteRunnerTask.addRetry({
            errors: [
                'RunnerBusy',
            ],
            interval: cdk.Duration.minutes(1),
            backoffRate: 1,
            maxAttempts: 60,
        });
        const waitForIdleRunner = new aws_cdk_lib_1.aws_stepfunctions.Wait(this, 'Wait', {
            time: aws_cdk_lib_1.aws_stepfunctions.WaitTime.duration(props?.idleTimeout ?? cdk.Duration.minutes(10)),
        });
        const deleteIdleRunnerTask = new aws_cdk_lib_1.aws_stepfunctions_tasks.LambdaInvoke(this, 'Delete Idle Runner', {
            lambdaFunction: deleteRunnerFunction,
            payloadResponseOnly: true,
            resultPath: '$.delete',
            payload: aws_cdk_lib_1.aws_stepfunctions.TaskInput.fromObject({
                runnerName: aws_cdk_lib_1.aws_stepfunctions.JsonPath.stringAt('$$.Execution.Name'),
                owner: aws_cdk_lib_1.aws_stepfunctions.JsonPath.stringAt('$.owner'),
                repo: aws_cdk_lib_1.aws_stepfunctions.JsonPath.stringAt('$.repo'),
                runId: aws_cdk_lib_1.aws_stepfunctions.JsonPath.stringAt('$.runId'),
                installationId: aws_cdk_lib_1.aws_stepfunctions.JsonPath.stringAt('$.installationId'),
                idleOnly: true,
            }),
        });
        const providerChooser = new aws_cdk_lib_1.aws_stepfunctions.Choice(this, 'Choose provider');
        for (const provider of this.providers) {
            const providerTask = provider.getStepFunctionTask({
                runnerTokenPath: aws_cdk_lib_1.aws_stepfunctions.JsonPath.stringAt('$.runner.token'),
                runnerNamePath: aws_cdk_lib_1.aws_stepfunctions.JsonPath.stringAt('$$.Execution.Name'),
                githubDomainPath: aws_cdk_lib_1.aws_stepfunctions.JsonPath.stringAt('$.runner.domain'),
                ownerPath: aws_cdk_lib_1.aws_stepfunctions.JsonPath.stringAt('$.owner'),
                repoPath: aws_cdk_lib_1.aws_stepfunctions.JsonPath.stringAt('$.repo'),
            });
            providerChooser.when(aws_cdk_lib_1.aws_stepfunctions.Condition.and(...provider.labels.map(label => aws_cdk_lib_1.aws_stepfunctions.Condition.isPresent(`$.labels.${label}`))), providerTask);
        }
        providerChooser.otherwise(new aws_cdk_lib_1.aws_stepfunctions.Succeed(this, 'Unknown label'));
        const work = tokenRetrieverTask.next(new aws_cdk_lib_1.aws_stepfunctions.Parallel(this, 'Error Catcher', { resultPath: '$.result' })
            .branch(providerChooser)
            .branch(waitForIdleRunner.next(deleteIdleRunnerTask))
            .addCatch(deleteRunnerTask
            .next(new aws_cdk_lib_1.aws_stepfunctions.Fail(this, 'Runner Failed')), {
            resultPath: '$.error',
        }));
        const check = new aws_cdk_lib_1.aws_stepfunctions.Choice(this, 'Is self hosted?')
            .when(aws_cdk_lib_1.aws_stepfunctions.Condition.isNotPresent('$.labels.self-hosted'), new aws_cdk_lib_1.aws_stepfunctions.Succeed(this, 'No'))
            .otherwise(work);
        const stateMachine = new aws_cdk_lib_1.aws_stepfunctions.StateMachine(this, 'Runner Orchestrator', {
            definition: check,
        });
        for (const provider of this.providers) {
            provider.grantStateMachine(stateMachine);
        }
        return stateMachine;
    }
    tokenRetriever() {
        const func = new utils_1.BundledNodejsFunction(this, 'token-retriever', {
            description: 'Get token from GitHub Actions used to start new self-hosted runner',
            environment: {
                GITHUB_SECRET_ARN: this.secrets.github.secretArn,
                GITHUB_PRIVATE_KEY_SECRET_ARN: this.secrets.githubPrivateKey.secretArn,
                ...this.extraLambdaEnv,
            },
            timeout: cdk.Duration.seconds(30),
            ...this.extraLambdaProps,
        });
        this.secrets.github.grantRead(func);
        this.secrets.githubPrivateKey.grantRead(func);
        return func;
    }
    deleteRunner() {
        const func = new utils_1.BundledNodejsFunction(this, 'delete-runner', {
            description: 'Delete GitHub Actions runner on error',
            environment: {
                GITHUB_SECRET_ARN: this.secrets.github.secretArn,
                GITHUB_PRIVATE_KEY_SECRET_ARN: this.secrets.githubPrivateKey.secretArn,
                ...this.extraLambdaEnv,
            },
            timeout: cdk.Duration.seconds(30),
            ...this.extraLambdaProps,
        });
        this.secrets.github.grantRead(func);
        this.secrets.githubPrivateKey.grantRead(func);
        return func;
    }
    statusFunction() {
        const statusFunction = new utils_1.BundledNodejsFunction(this, 'status', {
            description: 'Provide user with status about self-hosted GitHub Actions runners',
            environment: {
                WEBHOOK_SECRET_ARN: this.secrets.webhook.secretArn,
                GITHUB_SECRET_ARN: this.secrets.github.secretArn,
                GITHUB_PRIVATE_KEY_SECRET_ARN: this.secrets.githubPrivateKey.secretArn,
                SETUP_SECRET_ARN: this.secrets.setup.secretArn,
                WEBHOOK_URL: this.webhook.url,
                WEBHOOK_HANDLER_ARN: this.webhook.handler.latestVersion.functionArn,
                STEP_FUNCTION_ARN: this.orchestrator.stateMachineArn,
                SETUP_FUNCTION_URL: this.setupUrl,
                ...this.extraLambdaEnv,
            },
            timeout: cdk.Duration.minutes(3),
            ...this.extraLambdaProps,
        });
        const providers = this.providers.map(provider => provider.status(statusFunction));
        // expose providers as stack metadata as it's too big for Lambda environment variables
        // specifically integration testing got an error because lambda update request was >5kb
        const stack = cdk.Stack.of(this);
        const f = statusFunction.node.defaultChild;
        f.addPropertyOverride('Environment.Variables.LOGICAL_ID', f.logicalId);
        f.addPropertyOverride('Environment.Variables.STACK_NAME', stack.stackName);
        f.addMetadata('providers', providers);
        statusFunction.addToRolePolicy(new aws_cdk_lib_1.aws_iam.PolicyStatement({
            actions: ['cloudformation:DescribeStackResource'],
            resources: [stack.stackId],
        }));
        this.secrets.webhook.grantRead(statusFunction);
        this.secrets.github.grantRead(statusFunction);
        this.secrets.githubPrivateKey.grantRead(statusFunction);
        this.secrets.setup.grantRead(statusFunction);
        this.orchestrator.grantRead(statusFunction);
        new cdk.CfnOutput(this, 'status command', {
            value: `aws --region ${stack.region} lambda invoke --function-name ${statusFunction.functionName} status.json`,
        });
    }
    setupFunction() {
        const setupFunction = new utils_1.BundledNodejsFunction(this, 'setup', {
            description: 'Setup GitHub Actions integration with self-hosted runners',
            environment: {
                SETUP_SECRET_ARN: this.secrets.setup.secretArn,
                WEBHOOK_SECRET_ARN: this.secrets.webhook.secretArn,
                GITHUB_SECRET_ARN: this.secrets.github.secretArn,
                GITHUB_PRIVATE_KEY_SECRET_ARN: this.secrets.githubPrivateKey.secretArn,
                WEBHOOK_URL: this.webhook.url,
                ...this.extraLambdaEnv,
            },
            timeout: cdk.Duration.minutes(3),
            ...this.extraLambdaProps,
        });
        // this.secrets.webhook.grantRead(setupFunction);
        this.secrets.webhook.grantWrite(setupFunction);
        this.secrets.github.grantRead(setupFunction);
        this.secrets.github.grantWrite(setupFunction);
        // this.secrets.githubPrivateKey.grantRead(setupFunction);
        this.secrets.githubPrivateKey.grantWrite(setupFunction);
        this.secrets.setup.grantRead(setupFunction);
        this.secrets.setup.grantWrite(setupFunction);
        return setupFunction.addFunctionUrl({ authType: aws_lambda_1.FunctionUrlAuthType.NONE }).url;
    }
}
exports.GitHubRunners = GitHubRunners;
_a = JSII_RTTI_SYMBOL_1;
GitHubRunners[_a] = { fqn: "@cloudsnorkel/cdk-github-runners.GitHubRunners", version: "0.6.0" };
//# sourceMappingURL=data:application/json;base64,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