"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.AmiBuilder = void 0;
const cdk = require("aws-cdk-lib");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const utils_1 = require("../../utils");
const common_1 = require("../common");
const common_2 = require("./common");
const linux_components_1 = require("./linux-components");
const windows_components_1 = require("./windows-components");
/**
 * Image builder recipe for Amazon Machine Image (AMI).
 */
class AmiRecipe extends common_2.ImageBuilderObjectBase {
    constructor(scope, id, props) {
        super(scope, id);
        const name = common_2.uniqueImageBuilderName(this);
        let components = props.components.map(component => {
            return {
                componentArn: component.arn,
            };
        });
        let parentAmi;
        let workingDirectory;
        if (props.platform == 'Linux') {
            let archUrl;
            if (props.architecture.is(common_1.Architecture.X86_64)) {
                archUrl = 'amd64';
            }
            else if (props.architecture.is(common_1.Architecture.ARM64)) {
                archUrl = 'arm64';
            }
            else {
                throw new Error(`Unsupported architecture for parent AMI: ${props.architecture.name}`);
            }
            parentAmi = aws_cdk_lib_1.aws_ec2.MachineImage.fromSsmParameter(`/aws/service/canonical/ubuntu/server/focal/stable/current/${archUrl}/hvm/ebs-gp2/ami-id`, {
                os: aws_cdk_lib_1.aws_ec2.OperatingSystemType.LINUX,
            }).getImage(this).imageId;
            workingDirectory = '/home/runner';
        }
        else if (props.platform == 'Windows') {
            parentAmi = aws_cdk_lib_1.aws_ec2.MachineImage.latestWindows(aws_cdk_lib_1.aws_ec2.WindowsVersion.WINDOWS_SERVER_2022_ENGLISH_FULL_CONTAINERSLATEST).getImage(this).imageId;
            workingDirectory = 'C:/'; // must exist or Image Builder fails and must not be empty or git will stall installing from the default windows\system32
        }
        else {
            throw new Error(`Unsupported AMI recipe platform: ${props.platform}`);
        }
        const recipe = new aws_cdk_lib_1.aws_imagebuilder.CfnImageRecipe(this, 'Recipe', {
            name: name,
            version: this.version('ImageRecipe', name, {
                platform: props.platform,
                components,
            }),
            parentImage: parentAmi,
            components,
            workingDirectory,
        });
        this.arn = recipe.attrArn;
        this.name = name;
    }
}
/**
 * An AMI builder that uses AWS Image Builder to build AMIs pre-baked with all the GitHub Actions runner requirements. Builders can be used with {@link Ec2Runner}.
 *
 * Each builder re-runs automatically at a set interval to make sure the AMIs contain the latest versions of everything.
 *
 * You can create an instance of this construct to customize the AMI used to spin-up runners. Some runner providers may require custom components. Check the runner provider documentation.
 *
 * For example, to set a specific runner version, rebuild the image every 2 weeks, and add a few packages for the EC2 provider, use:
 *
 * ```
 * const builder = new AmiBuilder(this, 'Builder', {
 *     runnerVersion: RunnerVersion.specific('2.293.0'),
 *     rebuildInterval: Duration.days(14),
 * });
 * builder.addComponent(new ImageBuilderComponent(scope, id, {
 *   platform: 'Linux',
 *   displayName: 'p7zip',
 *   description: 'Install some more packages',
 *   commands: [
 *     'set -ex',
 *     'apt-get install p7zip',
 *   ],
 * }));
 * new Ec2Runner(this, 'EC2 provider', {
 *     label: 'custom-ec2',
 *     amiBuilder: builder,
 * });
 * ```
 */
class AmiBuilder extends common_2.ImageBuilderBase {
    constructor(scope, id, props) {
        super(scope, id, {
            os: props?.os,
            supportedOs: [common_1.Os.LINUX, common_1.Os.WINDOWS],
            architecture: props?.architecture,
            supportedArchitectures: [common_1.Architecture.X86_64, common_1.Architecture.ARM64],
            instanceType: props?.instanceType,
            vpc: props?.vpc,
            securityGroup: props?.securityGroup,
            subnetSelection: props?.subnetSelection,
            logRemovalPolicy: props?.logRemovalPolicy,
            logRetention: props?.logRetention,
            runnerVersion: props?.runnerVersion,
            rebuildInterval: props?.rebuildInterval,
            imageTypeName: 'AMI',
        });
        // add all basic components
        if (this.os.is(common_1.Os.WINDOWS)) {
            this.addBaseWindowsComponents();
        }
        else if (this.os.is(common_1.Os.LINUX)) {
            this.addBaseLinuxComponents();
        }
    }
    addBaseWindowsComponents() {
        this.addComponent(windows_components_1.WindowsComponents.cloudwatchAgent(this, 'CloudWatch agent'));
        this.addComponent(windows_components_1.WindowsComponents.awsCli(this, 'AWS CLI'));
        this.addComponent(windows_components_1.WindowsComponents.githubCli(this, 'GitHub CLI'));
        this.addComponent(windows_components_1.WindowsComponents.git(this, 'git'));
        this.addComponent(windows_components_1.WindowsComponents.githubRunner(this, 'GitHub Actions Runner', this.runnerVersion));
        this.addComponent(windows_components_1.WindowsComponents.docker(this, 'Docker'));
    }
    addBaseLinuxComponents() {
        this.addComponent(linux_components_1.LinuxUbuntuComponents.requiredPackages(this, 'Upgrade packages and install basics', this.architecture));
        this.addComponent(linux_components_1.LinuxUbuntuComponents.runnerUser(this, 'User', this.architecture));
        this.addComponent(linux_components_1.LinuxUbuntuComponents.awsCli(this, 'AWS CLI', this.architecture));
        this.addComponent(linux_components_1.LinuxUbuntuComponents.githubCli(this, 'GitHub CLI', this.architecture));
        this.addComponent(linux_components_1.LinuxUbuntuComponents.git(this, 'git', this.architecture));
        this.addComponent(linux_components_1.LinuxUbuntuComponents.githubRunner(this, 'GitHub Actions Runner', this.runnerVersion, this.architecture));
        this.addComponent(linux_components_1.LinuxUbuntuComponents.docker(this, 'Docker', this.architecture));
    }
    /**
     * Add a component to be installed before any other components. Useful for required system settings like certificates or proxy settings.
     * @param component
     */
    prependComponent(component) {
        if (this.boundAmi) {
            throw new Error('AMI is already bound. Use this method before passing the builder to a runner provider.');
        }
        if (component.platform != this.platform) {
            throw new Error('Component platform doesn\'t match builder platform');
        }
        this.components = [component].concat(this.components);
    }
    /**
     * Add a component to be installed.
     * @param component
     */
    addComponent(component) {
        if (this.boundAmi) {
            throw new Error('AMI is already bound. Use this method before passing the builder to a runner provider.');
        }
        if (component.platform != this.platform) {
            throw new Error('Component platform doesn\'t match builder platform');
        }
        this.components.push(component);
    }
    /**
     * Add extra trusted certificates. This helps deal with self-signed certificates for GitHub Enterprise Server.
     *
     * @param path path to directory containing a file called certs.pem containing all the required certificates
     */
    addExtraCertificates(path) {
        this.prependComponent(new common_2.ImageBuilderComponent(this, 'Extra Certs', {
            platform: this.platform,
            displayName: 'GitHub Actions Runner',
            description: 'Install latest version of GitHub Actions Runner',
            commands: [
                '$ErrorActionPreference = \'Stop\'',
                'Import-Certificate -FilePath certs\\certs.pem -CertStoreLocation Cert:\\LocalMachine\\Root',
            ],
            assets: [
                {
                    path: 'certs',
                    asset: new aws_cdk_lib_1.aws_s3_assets.Asset(this, 'Extra Certs Asset', { path }),
                },
            ],
        }));
    }
    /**
     * Called by IRunnerProvider to finalize settings and create the AMI builder.
     */
    bind() {
        if (this.boundAmi) {
            return this.boundAmi;
        }
        const launchTemplate = new aws_cdk_lib_1.aws_ec2.LaunchTemplate(this, 'Launch template');
        const stackName = cdk.Stack.of(this).stackName;
        const builderName = this.node.path;
        const dist = new aws_cdk_lib_1.aws_imagebuilder.CfnDistributionConfiguration(this, 'Distribution', {
            name: common_2.uniqueImageBuilderName(this),
            description: this.description,
            distributions: [
                {
                    region: aws_cdk_lib_1.Stack.of(this).region,
                    amiDistributionConfiguration: {
                        Name: `${cdk.Names.uniqueResourceName(this, {
                            maxLength: 100,
                            separator: '-',
                            allowedSpecialCharacters: '_-',
                        })}-{{ imagebuilder:buildDate }}`,
                        AmiTags: {
                            'Name': this.node.id,
                            'GitHubRunners:Stack': stackName,
                            'GitHubRunners:Builder': builderName,
                        },
                    },
                    launchTemplateConfigurations: [
                        {
                            launchTemplateId: launchTemplate.launchTemplateId,
                        },
                    ],
                },
            ],
        });
        const recipe = new AmiRecipe(this, 'Ami Recipe', {
            platform: this.platform,
            components: this.components,
            architecture: this.architecture,
        });
        const log = this.createLog(recipe.name);
        const infra = this.createInfrastructure([
            aws_cdk_lib_1.aws_iam.ManagedPolicy.fromAwsManagedPolicyName('AmazonSSMManagedInstanceCore'),
            aws_cdk_lib_1.aws_iam.ManagedPolicy.fromAwsManagedPolicyName('EC2InstanceProfileForImageBuilder'),
        ]);
        this.createImage(infra, dist, log, recipe.arn, undefined);
        this.createPipeline(infra, dist, log, recipe.arn, undefined);
        this.boundAmi = {
            launchTemplate: launchTemplate,
            architecture: this.architecture,
            os: this.os,
            logGroup: log,
            runnerVersion: this.runnerVersion,
        };
        this.imageCleaner(launchTemplate, stackName, builderName);
        return this.boundAmi;
    }
    imageCleaner(launchTemplate, stackName, builderName) {
        const deleter = utils_1.BundledNodejsFunction.singleton(this, 'delete-ami', {
            initialPolicy: [
                new aws_cdk_lib_1.aws_iam.PolicyStatement({
                    actions: ['ec2:DescribeLaunchTemplateVersions', 'ec2:DescribeImages', 'ec2:DeregisterImage'],
                    resources: ['*'],
                }),
            ],
            timeout: cdk.Duration.minutes(5),
        });
        // delete old AMIs on schedule
        const eventRule = new aws_cdk_lib_1.aws_events.Rule(this, 'Delete AMI Schedule', {
            schedule: aws_cdk_lib_1.aws_events.Schedule.rate(cdk.Duration.days(1)),
            description: `Delete old AMIs for ${builderName}`,
        });
        eventRule.addTarget(new aws_cdk_lib_1.aws_events_targets.LambdaFunction(deleter, {
            event: aws_cdk_lib_1.aws_events.RuleTargetInput.fromObject({
                RequestType: 'Scheduled',
                LaunchTemplateId: launchTemplate.launchTemplateId,
                StackName: stackName,
                BuilderName: builderName,
            }),
        }));
        // delete all AMIs when this construct is removed
        new aws_cdk_lib_1.CustomResource(this, 'AMI Deleter', {
            serviceToken: deleter.functionArn,
            resourceType: 'Custom::AmiDeleter',
            properties: {
                StackName: stackName,
                BuilderName: builderName,
            },
        });
    }
}
exports.AmiBuilder = AmiBuilder;
//# sourceMappingURL=data:application/json;base64,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