"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.FargateRunner = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const path = require("path");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_logs_1 = require("aws-cdk-lib/aws-logs");
const aws_stepfunctions_1 = require("aws-cdk-lib/aws-stepfunctions");
const common_1 = require("./common");
const codebuild_1 = require("./image-builders/codebuild");
/**
 * Our special launch target that can use spot instances and set EnableExecuteCommand.
 */
class EcsFargateLaunchTarget {
    constructor(props) {
        this.props = props;
    }
    /**
     * Called when the Fargate launch type configured on RunTask
     */
    bind(_task, launchTargetOptions) {
        if (!launchTargetOptions.taskDefinition.isFargateCompatible) {
            throw new Error('Supplied TaskDefinition is not compatible with Fargate');
        }
        return {
            parameters: {
                EnableExecuteCommand: this.props.enableExecute,
                CapacityProviderStrategy: [
                    {
                        CapacityProvider: this.props.spot ? 'FARGATE_SPOT' : 'FARGATE',
                    },
                ],
            },
        };
    }
}
/**
 * GitHub Actions runner provider using Fargate to execute jobs.
 *
 * Creates a task definition with a single container that gets started for each job.
 *
 * This construct is not meant to be used by itself. It should be passed in the providers property for GitHubRunners.
 */
class FargateRunner extends common_1.BaseProvider {
    constructor(scope, id, props) {
        super(scope, id);
        this.labels = this.labelsFromProperties('fargate', props.label, props.labels);
        this.vpc = props.vpc ?? aws_cdk_lib_1.aws_ec2.Vpc.fromLookup(this, 'default vpc', { isDefault: true });
        this.subnetSelection = props.subnetSelection;
        this.securityGroup = props.securityGroup ?? new aws_cdk_lib_1.aws_ec2.SecurityGroup(this, 'security group', { vpc: this.vpc });
        this.connections = this.securityGroup.connections;
        this.assignPublicIp = props.assignPublicIp ?? true;
        this.cluster = props.cluster ? props.cluster : new aws_cdk_lib_1.aws_ecs.Cluster(this, 'cluster', {
            vpc: this.vpc,
            enableFargateCapacityProviders: true,
        });
        this.spot = props.spot ?? false;
        const imageBuilder = props.imageBuilder ?? new codebuild_1.CodeBuildImageBuilder(this, 'Image Builder', {
            dockerfilePath: FargateRunner.LINUX_X64_DOCKERFILE_PATH,
        });
        const image = this.image = imageBuilder.bind();
        let arch;
        if (image.architecture.is(common_1.Architecture.ARM64)) {
            arch = aws_cdk_lib_1.aws_ecs.CpuArchitecture.ARM64;
        }
        else if (image.architecture.is(common_1.Architecture.X86_64)) {
            arch = aws_cdk_lib_1.aws_ecs.CpuArchitecture.X86_64;
        }
        else {
            throw new Error(`${image.architecture.name} is not supported on Fargate`);
        }
        let os;
        if (image.os.is(common_1.Os.LINUX)) {
            os = aws_cdk_lib_1.aws_ecs.OperatingSystemFamily.LINUX;
        }
        else if (image.os.is(common_1.Os.WINDOWS)) {
            os = aws_cdk_lib_1.aws_ecs.OperatingSystemFamily.WINDOWS_SERVER_2019_CORE;
            if (props.ephemeralStorageGiB) {
                throw new Error('Ephemeral storage is not supported on Fargate Windows');
            }
        }
        else {
            throw new Error(`${image.os.name} is not supported on Fargate`);
        }
        this.task = new aws_cdk_lib_1.aws_ecs.FargateTaskDefinition(this, 'task', {
            cpu: props.cpu ?? 1024,
            memoryLimitMiB: props.memoryLimitMiB ?? 2048,
            ephemeralStorageGiB: props.ephemeralStorageGiB ?? !image.os.is(common_1.Os.WINDOWS) ? 25 : undefined,
            runtimePlatform: {
                operatingSystemFamily: os,
                cpuArchitecture: arch,
            },
        });
        this.container = this.task.addContainer('runner', {
            image: aws_cdk_lib_1.aws_ecs.AssetImage.fromEcrRepository(image.imageRepository, image.imageTag),
            logging: aws_cdk_lib_1.aws_ecs.AwsLogDriver.awsLogs({
                logGroup: new aws_cdk_lib_1.aws_logs.LogGroup(this, 'logs', {
                    retention: props.logRetention ?? aws_logs_1.RetentionDays.ONE_MONTH,
                    removalPolicy: aws_cdk_lib_1.RemovalPolicy.DESTROY,
                }),
                streamPrefix: 'runner',
            }),
            command: this.runCommand(),
        });
        this.grantPrincipal = new aws_cdk_lib_1.aws_iam.UnknownPrincipal({ resource: this.task.taskRole });
    }
    /**
     * Generate step function task(s) to start a new runner.
     *
     * Called by GithubRunners and shouldn't be called manually.
     *
     * @param parameters workflow job details
     */
    getStepFunctionTask(parameters) {
        return new aws_cdk_lib_1.aws_stepfunctions_tasks.EcsRunTask(this, this.labels.join(', '), {
            integrationPattern: aws_stepfunctions_1.IntegrationPattern.RUN_JOB,
            taskDefinition: this.task,
            cluster: this.cluster,
            launchTarget: new EcsFargateLaunchTarget({
                spot: this.spot,
                enableExecute: this.image.os.is(common_1.Os.LINUX),
            }),
            subnets: this.subnetSelection,
            assignPublicIp: this.assignPublicIp,
            securityGroups: this.securityGroup ? [this.securityGroup] : undefined,
            containerOverrides: [
                {
                    containerDefinition: this.container,
                    environment: [
                        {
                            name: 'RUNNER_TOKEN',
                            value: parameters.runnerTokenPath,
                        },
                        {
                            name: 'RUNNER_NAME',
                            value: parameters.runnerNamePath,
                        },
                        {
                            name: 'RUNNER_LABEL',
                            value: this.labels.join(','),
                        },
                        {
                            name: 'GITHUB_DOMAIN',
                            value: parameters.githubDomainPath,
                        },
                        {
                            name: 'OWNER',
                            value: parameters.ownerPath,
                        },
                        {
                            name: 'REPO',
                            value: parameters.repoPath,
                        },
                    ],
                },
            ],
        });
    }
    grantStateMachine(_) {
    }
    status(statusFunctionRole) {
        this.image.imageRepository.grant(statusFunctionRole, 'ecr:DescribeImages');
        return {
            type: this.constructor.name,
            labels: this.labels,
            vpcArn: this.vpc?.vpcArn,
            securityGroup: this.securityGroup?.securityGroupId,
            roleArn: this.task.taskRole.roleArn,
            image: {
                imageRepository: this.image.imageRepository.repositoryUri,
                imageTag: this.image.imageTag,
                imageBuilderLogGroup: this.image.logGroup?.logGroupName,
            },
        };
    }
    runCommand() {
        let runnerFlags = '';
        if (this.image.runnerVersion.is(common_1.RunnerVersion.latest())) {
            runnerFlags = '--disableupdate';
        }
        if (this.image.os.is(common_1.Os.LINUX)) {
            return [
                'sh', '-c',
                `./config.sh --unattended --url "https://$GITHUB_DOMAIN/$OWNER/$REPO" --token "$RUNNER_TOKEN" --ephemeral --work _work --labels "$RUNNER_LABEL" ${runnerFlags} --name "$RUNNER_NAME" && ./run.sh`,
            ];
        }
        else if (this.image.os.is(common_1.Os.WINDOWS)) {
            return [
                'powershell', '-Command',
                `cd \\actions ; ./config.cmd --unattended --url "https://\${Env:GITHUB_DOMAIN}/\${Env:OWNER}/\${Env:REPO}" --token "\${Env:RUNNER_TOKEN}" --ephemeral --work _work --labels "\${Env:RUNNER_LABEL}" ${runnerFlags} --name "\${Env:RUNNER_NAME}" ; ./run.cmd`,
            ];
        }
        else {
            throw new Error(`Fargate runner doesn't support ${this.image.os.name}`);
        }
    }
}
exports.FargateRunner = FargateRunner;
_a = JSII_RTTI_SYMBOL_1;
FargateRunner[_a] = { fqn: "@cloudsnorkel/cdk-github-runners.FargateRunner", version: "0.6.0" };
/**
 * Path to Dockerfile for Linux x64 with all the requirement for Fargate runner. Use this Dockerfile unless you need to customize it further than allowed by hooks.
 *
 * Available build arguments that can be set in the image builder:
 * * `BASE_IMAGE` sets the `FROM` line. This should be an Ubuntu compatible image.
 * * `EXTRA_PACKAGES` can be used to install additional packages.
 */
FargateRunner.LINUX_X64_DOCKERFILE_PATH = path.join(__dirname, 'docker-images', 'fargate', 'linux-x64');
/**
 * Path to Dockerfile for Linux ARM64 with all the requirement for Fargate runner. Use this Dockerfile unless you need to customize it further than allowed by hooks.
 *
 * Available build arguments that can be set in the image builder:
 * * `BASE_IMAGE` sets the `FROM` line. This should be an Ubuntu compatible image.
 * * `EXTRA_PACKAGES` can be used to install additional packages.
 */
FargateRunner.LINUX_ARM64_DOCKERFILE_PATH = path.join(__dirname, 'docker-images', 'fargate', 'linux-arm64');
//# sourceMappingURL=data:application/json;base64,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