"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Ec2Runner = void 0;
const cdk = require("aws-cdk-lib");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_logs_1 = require("aws-cdk-lib/aws-logs");
const aws_stepfunctions_1 = require("aws-cdk-lib/aws-stepfunctions");
const common_1 = require("./common");
const ami_1 = require("./image-builders/ami");
// this script is specifically made so `poweroff` is absolutely always called
// each `{}` is a variable coming from `params` below
const linuxUserDataTemplate = `#!/bin/bash -x
TASK_TOKEN="{}"
heartbeat () {
  while true; do
    aws stepfunctions send-task-heartbeat --task-token "$TASK_TOKEN"
    sleep 60
  done
}
setup_logs () {
  cat <<EOF > /tmp/log.conf || exit 1
  {
    "logs": {
      "log_stream_name": "unknown",
      "logs_collected": {
        "files": {
          "collect_list": [
            {
              "file_path": "/var/log/runner.log",
              "log_group_name": "{}",
              "log_stream_name": "{}",
              "timezone": "UTC"
            }
          ]
        }
      }
    }
  }
EOF
  /opt/aws/amazon-cloudwatch-agent/bin/amazon-cloudwatch-agent-ctl -a fetch-config -m ec2 -s -c file:/tmp/log.conf || exit 2
}
action () {
  sudo -Hu runner /home/runner/config.sh --unattended --url "https://{}/{}/{}" --token "{}" --ephemeral --work _work --labels "{}" {} --name "{}" || exit 1
  sudo --preserve-env=AWS_REGION -Hu runner /home/runner/run.sh || exit 2
}
heartbeat &
if setup_logs && action | tee /var/log/runner.log 2>&1; then
  aws stepfunctions send-task-success --task-token "$TASK_TOKEN" --task-output '{"ok": true}'
else
  aws stepfunctions send-task-failure --task-token "$TASK_TOKEN"
fi
poweroff
`.replace(/{/g, '\\{').replace(/}/g, '\\}').replace(/\\{\\}/g, '{}');
// this script is specifically made so `poweroff` is absolutely always called
// each `{}` is a variable coming from `params` below and their order should match the linux script
const windowsUserDataTemplate = `<powershell>
$TASK_TOKEN = "{}"
Start-Job -ScriptBlock {
  while (1) {
    aws stepfunctions send-task-heartbeat --task-token "$using:TASK_TOKEN"
    sleep 60
  }
}
function setup_logs () {
  echo '{
    "logs": {
      "log_stream_name": "unknown",
      "logs_collected": {
        "files": {
         "collect_list": [
            {
              "file_path": "/actions/runner.log",
              "log_group_name": "{}",
              "log_stream_name": "{}",
              "timezone": "UTC"
            }
          ]
        }
      }
    }
  }' | Out-File -Encoding ASCII $Env:TEMP/log.conf
  & "C:/Program Files/Amazon/AmazonCloudWatchAgent/amazon-cloudwatch-agent-ctl.ps1" -a fetch-config -m ec2 -s -c file:$Env:TEMP/log.conf
}
function action () {
  cd /actions
  ./config.cmd --unattended --url "https://{}/{}/{}" --token "{}" --ephemeral --work _work --labels "{}" {} --name "{}" 2>&1 | Out-File -Encoding ASCII -Append /actions/runner.log
  if ($LASTEXITCODE -ne 0) { return 1 }
  ./run.cmd 2>&1 | Out-File -Encoding ASCII -Append /actions/runner.log
  if ($LASTEXITCODE -ne 0) { return 2 }
  return 0
}
setup_logs
$r = action
if ($r -eq 0) {
  aws stepfunctions send-task-success --task-token "$TASK_TOKEN" --task-output '{ }'
} else {
  aws stepfunctions send-task-failure --task-token "$TASK_TOKEN"
}
Stop-Computer -ComputerName localhost -Force
</powershell>
`.replace(/{/g, '\\{').replace(/}/g, '\\}').replace(/\\{\\}/g, '{}');
/**
 * GitHub Actions runner provider using EC2 to execute jobs.
 *
 * This construct is not meant to be used by itself. It should be passed in the providers property for GitHubRunners.
 */
class Ec2Runner extends common_1.BaseProvider {
    constructor(scope, id, props) {
        super(scope, id);
        this.labels = props.labels ?? ['ec2'];
        this.securityGroup = props.securityGroup;
        this.subnet = props.subnet;
        this.instanceType = props.instanceType ?? aws_cdk_lib_1.aws_ec2.InstanceType.of(aws_cdk_lib_1.aws_ec2.InstanceClass.M5, aws_cdk_lib_1.aws_ec2.InstanceSize.LARGE);
        this.storageSize = props.storageSize ?? cdk.Size.gibibytes(30); // 30 is the minimum for Windows
        this.spot = props.spot ?? false;
        this.spotMaxPrice = props.spotMaxPrice;
        const amiBuilder = props.amiBuilder ?? new ami_1.AmiBuilder(this, 'Image Builder');
        this.ami = amiBuilder.bind();
        if (!this.ami.architecture.instanceTypeMatch(this.instanceType)) {
            throw new Error(`AMI architecture (${this.ami.architecture.name}) doesn't match runner instance type (${this.instanceType} / ${this.instanceType.architecture})`);
        }
        this.grantPrincipal = this.role = new aws_cdk_lib_1.aws_iam.Role(this, 'Role', {
            assumedBy: new aws_cdk_lib_1.aws_iam.ServicePrincipal('ec2.amazonaws.com'),
            managedPolicies: [
                aws_cdk_lib_1.aws_iam.ManagedPolicy.fromAwsManagedPolicyName('AmazonSSMManagedInstanceCore'),
            ],
        });
        this.grantPrincipal.addToPrincipalPolicy(new aws_cdk_lib_1.aws_iam.PolicyStatement({
            actions: ['states:SendTaskFailure', 'states:SendTaskSuccess', 'states:SendTaskHeartbeat'],
            resources: ['*'],
        }));
        this.logGroup = new aws_cdk_lib_1.aws_logs.LogGroup(this, 'Logs', {
            retention: props.logRetention ?? aws_logs_1.RetentionDays.ONE_MONTH,
            removalPolicy: aws_cdk_lib_1.RemovalPolicy.DESTROY,
        });
        this.logGroup.grantWrite(this);
    }
    /**
     * Generate step function task(s) to start a new runner.
     *
     * Called by GithubRunners and shouldn't be called manually.
     *
     * @param parameters workflow job details
     */
    getStepFunctionTask(parameters) {
        // we need to build user data in two steps because passing the template as the first parameter to stepfunctions.JsonPath.format fails on syntax
        const params = [
            aws_cdk_lib_1.aws_stepfunctions.JsonPath.taskToken,
            this.logGroup.logGroupName,
            parameters.runnerNamePath,
            parameters.githubDomainPath,
            parameters.ownerPath,
            parameters.repoPath,
            parameters.runnerTokenPath,
            this.labels.join(','),
            this.ami.runnerVersion.is(common_1.RunnerVersion.latest()) ? '' : '--disableupdate',
            parameters.runnerNamePath,
        ];
        const passUserData = new aws_cdk_lib_1.aws_stepfunctions.Pass(this, `${this.labels.join(', ')} data`, {
            parameters: {
                userdataTemplate: this.ami.os.is(common_1.Os.WINDOWS) ? windowsUserDataTemplate : linuxUserDataTemplate,
            },
            resultPath: aws_cdk_lib_1.aws_stepfunctions.JsonPath.stringAt('$.ec2'),
        });
        // we can't use fleets because they don't let us override user data, security groups or even disk size
        // we can't use requestSpotInstances because it doesn't support launch templates, and it's deprecated
        const run = new aws_cdk_lib_1.aws_stepfunctions_tasks.CallAwsService(this, this.labels.join(', '), {
            integrationPattern: aws_stepfunctions_1.IntegrationPattern.WAIT_FOR_TASK_TOKEN,
            service: 'ec2',
            action: 'runInstances',
            heartbeat: aws_cdk_lib_1.Duration.minutes(5),
            parameters: {
                LaunchTemplate: {
                    LaunchTemplateId: this.ami.launchTemplate.launchTemplateId,
                },
                MinCount: 1,
                MaxCount: 1,
                InstanceType: this.instanceType.toString(),
                UserData: aws_cdk_lib_1.aws_stepfunctions.JsonPath.base64Encode(aws_cdk_lib_1.aws_stepfunctions.JsonPath.format(aws_cdk_lib_1.aws_stepfunctions.JsonPath.stringAt('$.ec2.userdataTemplate'), ...params)),
                InstanceInitiatedShutdownBehavior: aws_cdk_lib_1.aws_ec2.InstanceInitiatedShutdownBehavior.TERMINATE,
                IamInstanceProfile: {
                    Arn: new aws_cdk_lib_1.aws_iam.CfnInstanceProfile(this, 'Instance Profile', {
                        roles: [this.role.roleName],
                    }).attrArn,
                },
                MetadataOptions: {
                    HttpTokens: 'required',
                },
                SecurityGroupIds: this.securityGroup ? [this.securityGroup.securityGroupId] : undefined,
                SubnetId: this.subnet?.subnetId,
                BlockDeviceMappings: [{
                        DeviceName: '/dev/sda1',
                        Ebs: {
                            DeleteOnTermination: true,
                            VolumeSize: this.storageSize.toGibibytes(),
                        },
                    }],
                InstanceMarketOptions: this.spot ? {
                    MarketType: 'spot',
                    SpotOptions: {
                        MaxPrice: this.spotMaxPrice,
                        SpotInstanceType: 'one-time',
                    },
                } : undefined,
            },
            iamResources: ['*'],
        });
        return passUserData.next(run);
    }
    grantStateMachine(stateMachineRole) {
        stateMachineRole.grantPrincipal.addToPrincipalPolicy(new aws_cdk_lib_1.aws_iam.PolicyStatement({
            actions: ['iam:PassRole'],
            resources: [this.role.roleArn],
            conditions: {
                StringEquals: {
                    'iam:PassedToService': 'ec2.amazonaws.com',
                },
            },
        }));
        stateMachineRole.grantPrincipal.addToPrincipalPolicy(new aws_cdk_lib_1.aws_iam.PolicyStatement({
            actions: ['ec2:createTags'],
            resources: [aws_cdk_lib_1.Stack.of(this).formatArn({
                    service: 'ec2',
                    resource: '*',
                })],
        }));
    }
    status(statusFunctionRole) {
        statusFunctionRole.grantPrincipal.addToPrincipalPolicy(new aws_cdk_lib_1.aws_iam.PolicyStatement({
            actions: ['ec2:DescribeLaunchTemplateVersions'],
            resources: ['*'],
        }));
        return {
            type: this.constructor.name,
            labels: this.labels,
            securityGroup: this.securityGroup?.securityGroupId,
            roleArn: this.role.roleArn,
            ami: {
                launchTemplate: this.ami.launchTemplate.launchTemplateId || 'unknown',
                amiBuilderLogGroup: this.ami.logGroup?.logGroupName,
            },
        };
    }
    /**
     * The network connections associated with this resource.
     */
    get connections() {
        return this.securityGroup?.connections ?? new aws_cdk_lib_1.aws_ec2.Connections();
    }
}
exports.Ec2Runner = Ec2Runner;
//# sourceMappingURL=data:application/json;base64,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