import * as cdk from 'aws-cdk-lib';
import { aws_ec2 as ec2, aws_iam as iam, aws_stepfunctions as stepfunctions } from 'aws-cdk-lib';
import { Construct } from 'constructs';
import { BaseProvider, IAmiBuilder, IRunnerProvider, IRunnerProviderStatus, RunnerProviderProps, RunnerRuntimeParameters } from './common';
/**
 * Properties for {@link Ec2Runner} construct.
 */
export interface Ec2RunnerProps extends RunnerProviderProps {
    /**
     * AMI builder that creates AMIs with GitHub runner pre-configured. On Linux, a user named `runner` is expected to exist with access to Docker.
     *
     * @default AMI builder for Ubuntu Linux
     */
    readonly amiBuilder?: IAmiBuilder;
    /**
     * GitHub Actions labels used for this provider.
     *
     * These labels are used to identify which provider should spawn a new on-demand runner. Every job sends a webhook with the labels it's looking for
     * based on runs-on. We match the labels from the webhook with the labels specified here. If all the labels specified here are present in the
     * job's labels, this provider will be chosen and spawn a new runner.
     *
     * @default ['ec2']
     */
    readonly labels?: string[];
    /**
     * Instance type for launched runner instances.
     *
     * @default m5.large
     */
    readonly instanceType?: ec2.InstanceType;
    /**
     * Size of volume available for launched runner instances. This modifies the boot volume size and doesn't add any additional volumes.
     *
     * @default 30GB
     */
    readonly storageSize?: cdk.Size;
    /**
     * Security Group to assign to launched runner instances.
     *
     * @default account's default security group
     */
    readonly securityGroup?: ec2.ISecurityGroup;
    /**
     * Subnet where the runner instances will be launched.
     *
     * @default default subnet of account's default VPC
     */
    readonly subnet?: ec2.ISubnet;
    /**
     * Use spot instances to save money. Spot instances are cheaper but not always available and can be stopped prematurely.
     *
     * @default false
     */
    readonly spot?: boolean;
    /**
     * Set a maximum price for spot instances.
     *
     * @default no max price (you will pay current spot price)
     */
    readonly spotMaxPrice?: string;
}
/**
 * GitHub Actions runner provider using EC2 to execute jobs.
 *
 * This construct is not meant to be used by itself. It should be passed in the providers property for GitHubRunners.
 */
export declare class Ec2Runner extends BaseProvider implements IRunnerProvider {
    /**
     * Labels associated with this provider.
     */
    readonly labels: string[];
    /**
     * VPC subnet used for hosting launched instances.
     */
    readonly subnet?: ec2.ISubnet;
    /**
     * Security group attached to launched instances.
     */
    readonly securityGroup?: ec2.ISecurityGroup;
    /**
     * Grant principal used to add permissions to the runner role.
     */
    readonly grantPrincipal: iam.IPrincipal;
    private readonly ami;
    private readonly logGroup;
    private readonly role;
    private readonly instanceType;
    private readonly storageSize;
    private readonly spot;
    private readonly spotMaxPrice;
    constructor(scope: Construct, id: string, props: Ec2RunnerProps);
    /**
     * Generate step function task(s) to start a new runner.
     *
     * Called by GithubRunners and shouldn't be called manually.
     *
     * @param parameters workflow job details
     */
    getStepFunctionTask(parameters: RunnerRuntimeParameters): stepfunctions.IChainable;
    grantStateMachine(stateMachineRole: iam.IGrantable): void;
    status(statusFunctionRole: iam.IGrantable): IRunnerProviderStatus;
    /**
     * The network connections associated with this resource.
     */
    get connections(): ec2.Connections;
}
