"use strict";
var _a, _b, _c;
Object.defineProperty(exports, "__esModule", { value: true });
exports.BaseProvider = exports.Os = exports.Architecture = exports.RunnerVersion = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const constructs_1 = require("constructs");
/**
 * Defines desired GitHub Actions runner version.
 */
class RunnerVersion {
    constructor(version) {
        this.version = version;
    }
    /**
     * Use the latest version available at the time the runner provider image is built.
     */
    static latest() {
        return new RunnerVersion('latest');
    }
    /**
     * Use a specific version.
     *
     * @see https://github.com/actions/runner/releases
     *
     * @param version GitHub Runner version
     */
    static specific(version) {
        return new RunnerVersion(version);
    }
    /**
     * Check if two versions are the same.
     *
     * @param other version to compare
     */
    is(other) {
        return this.version == other.version;
    }
}
exports.RunnerVersion = RunnerVersion;
_a = JSII_RTTI_SYMBOL_1;
RunnerVersion[_a] = { fqn: "@cloudsnorkel/cdk-github-runners.RunnerVersion", version: "0.6.0" };
/**
 * CPU architecture enum for an image.
 */
class Architecture {
    constructor(name) {
        this.name = name;
    }
    static of(architecture) {
        return new Architecture(architecture);
    }
    /**
    * Checks if the given architecture is the same as this one.
    *
    * @param arch architecture to compare
    */
    is(arch) {
        return arch.name == this.name;
    }
    /**
     * Checks if this architecture is in a given list.
     *
     * @param arches architectures to check
     */
    isIn(arches) {
        for (const arch of arches) {
            if (this.is(arch)) {
                return true;
            }
        }
        return false;
    }
    /**
     * Checks if a given EC2 instance type matches this architecture.
     *
     * @param instanceType instance type to check
     */
    instanceTypeMatch(instanceType) {
        if (instanceType.architecture == aws_cdk_lib_1.aws_ec2.InstanceArchitecture.X86_64) {
            return this.is(Architecture.X86_64);
        }
        if (instanceType.architecture == aws_cdk_lib_1.aws_ec2.InstanceArchitecture.ARM_64) {
            return this.is(Architecture.ARM64);
        }
        throw new Error('Unknown instance type architecture');
    }
}
exports.Architecture = Architecture;
_b = JSII_RTTI_SYMBOL_1;
Architecture[_b] = { fqn: "@cloudsnorkel/cdk-github-runners.Architecture", version: "0.6.0" };
/**
 * ARM64
 */
Architecture.ARM64 = Architecture.of('ARM64');
/**
 * X86_64
 */
Architecture.X86_64 = Architecture.of('X86_64');
/**
 * OS enum for an image.
 */
class Os {
    constructor(name) {
        this.name = name;
    }
    static of(os) {
        return new Os(os);
    }
    /**
    * Checks if the given OS is the same as this one.
    *
    * @param os OS to compare
    */
    is(os) {
        return os.name == this.name;
    }
    /**
     * Checks if this OS is in a given list.
     *
     * @param oses list of OS to check
     */
    isIn(oses) {
        for (const os of oses) {
            if (this.is(os)) {
                return true;
            }
        }
        return false;
    }
}
exports.Os = Os;
_c = JSII_RTTI_SYMBOL_1;
Os[_c] = { fqn: "@cloudsnorkel/cdk-github-runners.Os", version: "0.6.0" };
/**
* Linux
*/
Os.LINUX = Os.of('Linux');
/**
* Windows
*/
Os.WINDOWS = Os.of('Windows');
/**
 * Base class for all providers with common methods used by all providers.
 *
 * @internal
 */
class BaseProvider extends constructs_1.Construct {
    labelsFromProperties(defaultLabel, propsLabel, propsLabels) {
        if (propsLabels && propsLabel) {
            throw new Error('Must supply either `label` or `labels` in runner properties, but not both. Try removing the `label` property.');
        }
        if (propsLabels) {
            return propsLabels;
        }
        if (propsLabel) {
            return [propsLabel];
        }
        return [defaultLabel];
    }
}
exports.BaseProvider = BaseProvider;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiY29tbW9uLmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsiLi4vLi4vc3JjL3Byb3ZpZGVycy9jb21tb24udHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7Ozs7QUFBQSw2Q0FBbUk7QUFDbkksMkNBQXVDO0FBRXZDOztHQUVHO0FBQ0gsTUFBYSxhQUFhO0lBbUJ4QixZQUErQixPQUFlO1FBQWYsWUFBTyxHQUFQLE9BQU8sQ0FBUTtJQUM5QyxDQUFDO0lBbkJEOztPQUVHO0lBQ0ksTUFBTSxDQUFDLE1BQU07UUFDbEIsT0FBTyxJQUFJLGFBQWEsQ0FBQyxRQUFRLENBQUMsQ0FBQztJQUNyQyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksTUFBTSxDQUFDLFFBQVEsQ0FBQyxPQUFlO1FBQ3BDLE9BQU8sSUFBSSxhQUFhLENBQUMsT0FBTyxDQUFDLENBQUM7SUFDcEMsQ0FBQztJQUtEOzs7O09BSUc7SUFDSSxFQUFFLENBQUMsS0FBb0I7UUFDNUIsT0FBTyxJQUFJLENBQUMsT0FBTyxJQUFJLEtBQUssQ0FBQyxPQUFPLENBQUM7SUFDdkMsQ0FBQzs7QUE3Qkgsc0NBOEJDOzs7QUFFRDs7R0FFRztBQUNILE1BQWEsWUFBWTtJQWV2QixZQUFvQyxJQUFZO1FBQVosU0FBSSxHQUFKLElBQUksQ0FBUTtJQUNoRCxDQUFDO0lBTE8sTUFBTSxDQUFDLEVBQUUsQ0FBQyxZQUFvQjtRQUNwQyxPQUFPLElBQUksWUFBWSxDQUFDLFlBQVksQ0FBQyxDQUFDO0lBQ3hDLENBQUM7SUFLRDs7OztNQUlFO0lBQ0ssRUFBRSxDQUFDLElBQWtCO1FBQzFCLE9BQU8sSUFBSSxDQUFDLElBQUksSUFBSSxJQUFJLENBQUMsSUFBSSxDQUFDO0lBQ2hDLENBQUM7SUFFRDs7OztPQUlHO0lBQ0ksSUFBSSxDQUFDLE1BQXNCO1FBQ2hDLEtBQUssTUFBTSxJQUFJLElBQUksTUFBTSxFQUFFO1lBQ3pCLElBQUksSUFBSSxDQUFDLEVBQUUsQ0FBQyxJQUFJLENBQUMsRUFBRTtnQkFDakIsT0FBTyxJQUFJLENBQUM7YUFDYjtTQUNGO1FBQ0QsT0FBTyxLQUFLLENBQUM7SUFDZixDQUFDO0lBRUQ7Ozs7T0FJRztJQUNJLGlCQUFpQixDQUFDLFlBQThCO1FBQ3JELElBQUksWUFBWSxDQUFDLFlBQVksSUFBSSxxQkFBRyxDQUFDLG9CQUFvQixDQUFDLE1BQU0sRUFBRTtZQUNoRSxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsWUFBWSxDQUFDLE1BQU0sQ0FBQyxDQUFDO1NBQ3JDO1FBQ0QsSUFBSSxZQUFZLENBQUMsWUFBWSxJQUFJLHFCQUFHLENBQUMsb0JBQW9CLENBQUMsTUFBTSxFQUFFO1lBQ2hFLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxZQUFZLENBQUMsS0FBSyxDQUFDLENBQUM7U0FDcEM7UUFDRCxNQUFNLElBQUksS0FBSyxDQUFDLG9DQUFvQyxDQUFDLENBQUM7SUFDeEQsQ0FBQzs7QUF0REgsb0NBdURDOzs7QUF0REM7O0dBRUc7QUFDb0Isa0JBQUssR0FBRyxZQUFZLENBQUMsRUFBRSxDQUFDLE9BQU8sQ0FBQyxDQUFDO0FBRXhEOztHQUVHO0FBQ29CLG1CQUFNLEdBQUcsWUFBWSxDQUFDLEVBQUUsQ0FBQyxRQUFRLENBQUMsQ0FBQztBQWdENUQ7O0dBRUc7QUFDSCxNQUFhLEVBQUU7SUFlYixZQUFvQyxJQUFZO1FBQVosU0FBSSxHQUFKLElBQUksQ0FBUTtJQUNoRCxDQUFDO0lBTE8sTUFBTSxDQUFDLEVBQUUsQ0FBQyxFQUFVO1FBQzFCLE9BQU8sSUFBSSxFQUFFLENBQUMsRUFBRSxDQUFDLENBQUM7SUFDcEIsQ0FBQztJQUtEOzs7O01BSUU7SUFDSyxFQUFFLENBQUMsRUFBTTtRQUNkLE9BQU8sRUFBRSxDQUFDLElBQUksSUFBSSxJQUFJLENBQUMsSUFBSSxDQUFDO0lBQzlCLENBQUM7SUFFRDs7OztPQUlHO0lBQ0ksSUFBSSxDQUFDLElBQVU7UUFDcEIsS0FBSyxNQUFNLEVBQUUsSUFBSSxJQUFJLEVBQUU7WUFDckIsSUFBSSxJQUFJLENBQUMsRUFBRSxDQUFDLEVBQUUsQ0FBQyxFQUFFO2dCQUNmLE9BQU8sSUFBSSxDQUFDO2FBQ2I7U0FDRjtRQUNELE9BQU8sS0FBSyxDQUFDO0lBQ2YsQ0FBQzs7QUF2Q0gsZ0JBd0NDOzs7QUF2Q0M7O0VBRUU7QUFDcUIsUUFBSyxHQUFHLEVBQUUsQ0FBQyxFQUFFLENBQUMsT0FBTyxDQUFDLENBQUM7QUFFOUM7O0VBRUU7QUFDcUIsVUFBTyxHQUFHLEVBQUUsQ0FBQyxFQUFFLENBQUMsU0FBUyxDQUFDLENBQUM7QUFxVHBEOzs7O0dBSUc7QUFDSCxNQUFzQixZQUFhLFNBQVEsc0JBQVM7SUFDeEMsb0JBQW9CLENBQUMsWUFBb0IsRUFBRSxVQUE4QixFQUFFLFdBQWlDO1FBQ3BILElBQUksV0FBVyxJQUFJLFVBQVUsRUFBRTtZQUM3QixNQUFNLElBQUksS0FBSyxDQUFDLCtHQUErRyxDQUFDLENBQUM7U0FDbEk7UUFFRCxJQUFJLFdBQVcsRUFBRTtZQUNmLE9BQU8sV0FBVyxDQUFDO1NBQ3BCO1FBQ0QsSUFBSSxVQUFVLEVBQUU7WUFDZCxPQUFPLENBQUMsVUFBVSxDQUFDLENBQUM7U0FDckI7UUFDRCxPQUFPLENBQUMsWUFBWSxDQUFDLENBQUM7SUFDeEIsQ0FBQztDQUNGO0FBZEQsb0NBY0MiLCJzb3VyY2VzQ29udGVudCI6WyJpbXBvcnQgeyBhd3NfZWMyIGFzIGVjMiwgYXdzX2VjciBhcyBlY3IsIGF3c19pYW0gYXMgaWFtLCBhd3NfbG9ncyBhcyBsb2dzLCBhd3Nfc3RlcGZ1bmN0aW9ucyBhcyBzdGVwZnVuY3Rpb25zIH0gZnJvbSAnYXdzLWNkay1saWInO1xuaW1wb3J0IHsgQ29uc3RydWN0IH0gZnJvbSAnY29uc3RydWN0cyc7XG5cbi8qKlxuICogRGVmaW5lcyBkZXNpcmVkIEdpdEh1YiBBY3Rpb25zIHJ1bm5lciB2ZXJzaW9uLlxuICovXG5leHBvcnQgY2xhc3MgUnVubmVyVmVyc2lvbiB7XG4gIC8qKlxuICAgKiBVc2UgdGhlIGxhdGVzdCB2ZXJzaW9uIGF2YWlsYWJsZSBhdCB0aGUgdGltZSB0aGUgcnVubmVyIHByb3ZpZGVyIGltYWdlIGlzIGJ1aWx0LlxuICAgKi9cbiAgcHVibGljIHN0YXRpYyBsYXRlc3QoKTogUnVubmVyVmVyc2lvbiB7XG4gICAgcmV0dXJuIG5ldyBSdW5uZXJWZXJzaW9uKCdsYXRlc3QnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBVc2UgYSBzcGVjaWZpYyB2ZXJzaW9uLlxuICAgKlxuICAgKiBAc2VlIGh0dHBzOi8vZ2l0aHViLmNvbS9hY3Rpb25zL3J1bm5lci9yZWxlYXNlc1xuICAgKlxuICAgKiBAcGFyYW0gdmVyc2lvbiBHaXRIdWIgUnVubmVyIHZlcnNpb25cbiAgICovXG4gIHB1YmxpYyBzdGF0aWMgc3BlY2lmaWModmVyc2lvbjogc3RyaW5nKSB7XG4gICAgcmV0dXJuIG5ldyBSdW5uZXJWZXJzaW9uKHZlcnNpb24pO1xuICB9XG5cbiAgcHJvdGVjdGVkIGNvbnN0cnVjdG9yKHJlYWRvbmx5IHZlcnNpb246IHN0cmluZykge1xuICB9XG5cbiAgLyoqXG4gICAqIENoZWNrIGlmIHR3byB2ZXJzaW9ucyBhcmUgdGhlIHNhbWUuXG4gICAqXG4gICAqIEBwYXJhbSBvdGhlciB2ZXJzaW9uIHRvIGNvbXBhcmVcbiAgICovXG4gIHB1YmxpYyBpcyhvdGhlcjogUnVubmVyVmVyc2lvbikge1xuICAgIHJldHVybiB0aGlzLnZlcnNpb24gPT0gb3RoZXIudmVyc2lvbjtcbiAgfVxufVxuXG4vKipcbiAqIENQVSBhcmNoaXRlY3R1cmUgZW51bSBmb3IgYW4gaW1hZ2UuXG4gKi9cbmV4cG9ydCBjbGFzcyBBcmNoaXRlY3R1cmUge1xuICAvKipcbiAgICogQVJNNjRcbiAgICovXG4gIHB1YmxpYyBzdGF0aWMgcmVhZG9ubHkgQVJNNjQgPSBBcmNoaXRlY3R1cmUub2YoJ0FSTTY0Jyk7XG5cbiAgLyoqXG4gICAqIFg4Nl82NFxuICAgKi9cbiAgcHVibGljIHN0YXRpYyByZWFkb25seSBYODZfNjQgPSBBcmNoaXRlY3R1cmUub2YoJ1g4Nl82NCcpO1xuXG4gIHByaXZhdGUgc3RhdGljIG9mKGFyY2hpdGVjdHVyZTogc3RyaW5nKSB7XG4gICAgcmV0dXJuIG5ldyBBcmNoaXRlY3R1cmUoYXJjaGl0ZWN0dXJlKTtcbiAgfVxuXG4gIHByaXZhdGUgY29uc3RydWN0b3IocHVibGljIHJlYWRvbmx5IG5hbWU6IHN0cmluZykge1xuICB9XG5cbiAgLyoqXG4gICogQ2hlY2tzIGlmIHRoZSBnaXZlbiBhcmNoaXRlY3R1cmUgaXMgdGhlIHNhbWUgYXMgdGhpcyBvbmUuXG4gICpcbiAgKiBAcGFyYW0gYXJjaCBhcmNoaXRlY3R1cmUgdG8gY29tcGFyZVxuICAqL1xuICBwdWJsaWMgaXMoYXJjaDogQXJjaGl0ZWN0dXJlKTogYm9vbGVhbiB7XG4gICAgcmV0dXJuIGFyY2gubmFtZSA9PSB0aGlzLm5hbWU7XG4gIH1cblxuICAvKipcbiAgICogQ2hlY2tzIGlmIHRoaXMgYXJjaGl0ZWN0dXJlIGlzIGluIGEgZ2l2ZW4gbGlzdC5cbiAgICpcbiAgICogQHBhcmFtIGFyY2hlcyBhcmNoaXRlY3R1cmVzIHRvIGNoZWNrXG4gICAqL1xuICBwdWJsaWMgaXNJbihhcmNoZXM6IEFyY2hpdGVjdHVyZVtdKTogYm9vbGVhbiB7XG4gICAgZm9yIChjb25zdCBhcmNoIG9mIGFyY2hlcykge1xuICAgICAgaWYgKHRoaXMuaXMoYXJjaCkpIHtcbiAgICAgICAgcmV0dXJuIHRydWU7XG4gICAgICB9XG4gICAgfVxuICAgIHJldHVybiBmYWxzZTtcbiAgfVxuXG4gIC8qKlxuICAgKiBDaGVja3MgaWYgYSBnaXZlbiBFQzIgaW5zdGFuY2UgdHlwZSBtYXRjaGVzIHRoaXMgYXJjaGl0ZWN0dXJlLlxuICAgKlxuICAgKiBAcGFyYW0gaW5zdGFuY2VUeXBlIGluc3RhbmNlIHR5cGUgdG8gY2hlY2tcbiAgICovXG4gIHB1YmxpYyBpbnN0YW5jZVR5cGVNYXRjaChpbnN0YW5jZVR5cGU6IGVjMi5JbnN0YW5jZVR5cGUpOiBib29sZWFuIHtcbiAgICBpZiAoaW5zdGFuY2VUeXBlLmFyY2hpdGVjdHVyZSA9PSBlYzIuSW5zdGFuY2VBcmNoaXRlY3R1cmUuWDg2XzY0KSB7XG4gICAgICByZXR1cm4gdGhpcy5pcyhBcmNoaXRlY3R1cmUuWDg2XzY0KTtcbiAgICB9XG4gICAgaWYgKGluc3RhbmNlVHlwZS5hcmNoaXRlY3R1cmUgPT0gZWMyLkluc3RhbmNlQXJjaGl0ZWN0dXJlLkFSTV82NCkge1xuICAgICAgcmV0dXJuIHRoaXMuaXMoQXJjaGl0ZWN0dXJlLkFSTTY0KTtcbiAgICB9XG4gICAgdGhyb3cgbmV3IEVycm9yKCdVbmtub3duIGluc3RhbmNlIHR5cGUgYXJjaGl0ZWN0dXJlJyk7XG4gIH1cbn1cblxuLyoqXG4gKiBPUyBlbnVtIGZvciBhbiBpbWFnZS5cbiAqL1xuZXhwb3J0IGNsYXNzIE9zIHtcbiAgLyoqXG4gICogTGludXhcbiAgKi9cbiAgcHVibGljIHN0YXRpYyByZWFkb25seSBMSU5VWCA9IE9zLm9mKCdMaW51eCcpO1xuXG4gIC8qKlxuICAqIFdpbmRvd3NcbiAgKi9cbiAgcHVibGljIHN0YXRpYyByZWFkb25seSBXSU5ET1dTID0gT3Mub2YoJ1dpbmRvd3MnKTtcblxuICBwcml2YXRlIHN0YXRpYyBvZihvczogc3RyaW5nKSB7XG4gICAgcmV0dXJuIG5ldyBPcyhvcyk7XG4gIH1cblxuICBwcml2YXRlIGNvbnN0cnVjdG9yKHB1YmxpYyByZWFkb25seSBuYW1lOiBzdHJpbmcpIHtcbiAgfVxuXG4gIC8qKlxuICAqIENoZWNrcyBpZiB0aGUgZ2l2ZW4gT1MgaXMgdGhlIHNhbWUgYXMgdGhpcyBvbmUuXG4gICpcbiAgKiBAcGFyYW0gb3MgT1MgdG8gY29tcGFyZVxuICAqL1xuICBwdWJsaWMgaXMob3M6IE9zKSB7XG4gICAgcmV0dXJuIG9zLm5hbWUgPT0gdGhpcy5uYW1lO1xuICB9XG5cbiAgLyoqXG4gICAqIENoZWNrcyBpZiB0aGlzIE9TIGlzIGluIGEgZ2l2ZW4gbGlzdC5cbiAgICpcbiAgICogQHBhcmFtIG9zZXMgbGlzdCBvZiBPUyB0byBjaGVja1xuICAgKi9cbiAgcHVibGljIGlzSW4ob3NlczogT3NbXSk6IGJvb2xlYW4ge1xuICAgIGZvciAoY29uc3Qgb3Mgb2Ygb3Nlcykge1xuICAgICAgaWYgKHRoaXMuaXMob3MpKSB7XG4gICAgICAgIHJldHVybiB0cnVlO1xuICAgICAgfVxuICAgIH1cbiAgICByZXR1cm4gZmFsc2U7XG4gIH1cbn1cblxuLyoqXG4gKiBEZXNjcmlwdGlvbiBvZiBhIERvY2tlciBpbWFnZSBidWlsdCBieSB7QGxpbmsgSUltYWdlQnVpbGRlcn0uXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgUnVubmVySW1hZ2Uge1xuICAvKipcbiAgICogRUNSIHJlcG9zaXRvcnkgY29udGFpbmluZyB0aGUgaW1hZ2UuXG4gICAqL1xuICByZWFkb25seSBpbWFnZVJlcG9zaXRvcnk6IGVjci5JUmVwb3NpdG9yeTtcblxuICAvKipcbiAgICogU3RhdGljIGltYWdlIHRhZyB3aGVyZSB0aGUgaW1hZ2Ugd2lsbCBiZSBwdXNoZWQuXG4gICAqL1xuICByZWFkb25seSBpbWFnZVRhZzogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBBcmNoaXRlY3R1cmUgb2YgdGhlIGltYWdlLlxuICAgKi9cbiAgcmVhZG9ubHkgYXJjaGl0ZWN0dXJlOiBBcmNoaXRlY3R1cmU7XG5cbiAgLyoqXG4gICAqIE9TIHR5cGUgb2YgdGhlIGltYWdlLlxuICAgKi9cbiAgcmVhZG9ubHkgb3M6IE9zO1xuXG4gIC8qKlxuICAgKiBMb2cgZ3JvdXAgd2hlcmUgaW1hZ2UgYnVpbGRzIGFyZSBsb2dnZWQuXG4gICAqL1xuICByZWFkb25seSBsb2dHcm91cD86IGxvZ3MuTG9nR3JvdXA7XG5cbiAgLyoqXG4gICAqIEluc3RhbGxlZCBydW5uZXIgdmVyc2lvbi5cbiAgICovXG4gIHJlYWRvbmx5IHJ1bm5lclZlcnNpb246IFJ1bm5lclZlcnNpb247XG59XG5cbi8qKlxuICogSW50ZXJmYWNlIGZvciBjb25zdHJ1Y3RzIHRoYXQgYnVpbGQgYW4gaW1hZ2UgdGhhdCBjYW4gYmUgdXNlZCBpbiB7QGxpbmsgSVJ1bm5lclByb3ZpZGVyfS5cbiAqXG4gKiBBbnl0aGluZyB0aGF0IGVuZHMgdXAgd2l0aCBhbiBFQ1IgcmVwb3NpdG9yeSBjb250YWluaW5nIGEgRG9ja2VyIGltYWdlIHRoYXQgcnVucyBHaXRIdWIgc2VsZi1ob3N0ZWQgcnVubmVycyBjYW4gYmUgdXNlZC4gQSBzaW1wbGUgaW1wbGVtZW50YXRpb24gY291bGQgZXZlbiBwb2ludCB0byBhbiBleGlzdGluZyBpbWFnZSBhbmQgbm90aGluZyBlbHNlLlxuICpcbiAqIEl0J3MgaW1wb3J0YW50IHRoYXQgdGhlIHNwZWNpZmllZCBpbWFnZSB0YWcgYmUgYXZhaWxhYmxlIGF0IHRoZSB0aW1lIHRoZSByZXBvc2l0b3J5IGlzIGF2YWlsYWJsZS4gUHJvdmlkZXJzIHVzdWFsbHkgYXNzdW1lIHRoZSBpbWFnZSBpcyByZWFkeSBhbmQgd2lsbCBmYWlsIGlmIGl0J3Mgbm90LlxuICpcbiAqIFRoZSBpbWFnZSBjYW4gYmUgZnVydGhlciB1cGRhdGVkIG92ZXIgdGltZSBtYW51YWxseSBvciB1c2luZyBhIHNjaGVkdWxlIGFzIGxvbmcgYXMgaXQgaXMgYWx3YXlzIHdyaXR0ZW4gdG8gdGhlIHNhbWUgdGFnLlxuICovXG5leHBvcnQgaW50ZXJmYWNlIElJbWFnZUJ1aWxkZXIge1xuICAvKipcbiAgICogRmluYWxpemUgYW5kIHJldHVybiBhbGwgcmVxdWlyZWQgaW5mb3JtYXRpb24gYWJvdXQgdGhlIERvY2tlciBpbWFnZSBidWlsdCBieSB0aGlzIGJ1aWxkZXIuXG4gICAqXG4gICAqIFRoaXMgbWV0aG9kIGNhbiBiZSBjYWxsZWQgbXVsdGlwbGUgdGltZXMgaWYgdGhlIGltYWdlIGlzIGJvdW5kIHRvIG11bHRpcGxlIHByb3ZpZGVycy4gTWFrZSBzdXJlIHlvdSBjYWNoZSB0aGUgaW1hZ2Ugd2hlbiBpbXBsZW1lbnRpbmcgb3IgcmV0dXJuIGFuIGVycm9yIGlmIHRoaXMgYnVpbGRlciBkb2Vzbid0IHN1cHBvcnQgcmV1c2luZyBpbWFnZXMuXG4gICAqXG4gICAqIEByZXR1cm4gaW1hZ2VcbiAgICovXG4gIGJpbmQoKTogUnVubmVySW1hZ2U7XG59XG5cbi8qKlxuICogRGVzY3JpcHRpb24gb2YgYSBBTUkgYnVpbHQgYnkge0BsaW5rIElBbWlCdWlsZGVyfS5cbiAqL1xuZXhwb3J0IGludGVyZmFjZSBSdW5uZXJBbWkge1xuICAvKipcbiAgICogTGF1bmNoIHRlbXBsYXRlIHBvaW50aW5nIHRvIHRoZSBsYXRlc3QgQU1JLlxuICAgKi9cbiAgcmVhZG9ubHkgbGF1bmNoVGVtcGxhdGU6IGVjMi5JTGF1bmNoVGVtcGxhdGU7XG5cbiAgLyoqXG4gICAqIEFyY2hpdGVjdHVyZSBvZiB0aGUgaW1hZ2UuXG4gICAqL1xuICByZWFkb25seSBhcmNoaXRlY3R1cmU6IEFyY2hpdGVjdHVyZTtcblxuICAvKipcbiAgICogT1MgdHlwZSBvZiB0aGUgaW1hZ2UuXG4gICAqL1xuICByZWFkb25seSBvczogT3M7XG5cbiAgLyoqXG4gICAqIExvZyBncm91cCB3aGVyZSBpbWFnZSBidWlsZHMgYXJlIGxvZ2dlZC5cbiAgICovXG4gIHJlYWRvbmx5IGxvZ0dyb3VwPzogbG9ncy5Mb2dHcm91cDtcblxuICAvKipcbiAgICogSW5zdGFsbGVkIHJ1bm5lciB2ZXJzaW9uLlxuICAgKi9cbiAgcmVhZG9ubHkgcnVubmVyVmVyc2lvbjogUnVubmVyVmVyc2lvbjtcbn1cblxuLyoqXG4gKiBJbnRlcmZhY2UgZm9yIGNvbnN0cnVjdHMgdGhhdCBidWlsZCBhbiBBTUkgdGhhdCBjYW4gYmUgdXNlZCBpbiB7QGxpbmsgSVJ1bm5lclByb3ZpZGVyfS5cbiAqXG4gKiBBbnl0aGluZyB0aGF0IGVuZHMgdXAgd2l0aCBhIGxhdW5jaCB0ZW1wbGF0ZSBwb2ludGluZyB0byBhbiBBTUkgdGhhdCBydW5zIEdpdEh1YiBzZWxmLWhvc3RlZCBydW5uZXJzIGNhbiBiZSB1c2VkLiBBIHNpbXBsZSBpbXBsZW1lbnRhdGlvbiBjb3VsZCBldmVuIHBvaW50IHRvIGFuIGV4aXN0aW5nIEFNSSBhbmQgbm90aGluZyBlbHNlLlxuICpcbiAqIFRoZSBBTUkgY2FuIGJlIGZ1cnRoZXIgdXBkYXRlZCBvdmVyIHRpbWUgbWFudWFsbHkgb3IgdXNpbmcgYSBzY2hlZHVsZSBhcyBsb25nIGFzIGl0IGlzIGFsd2F5cyB3cml0dGVuIHRvIHRoZSBzYW1lIGxhdW5jaCB0ZW1wbGF0ZS5cbiAqL1xuZXhwb3J0IGludGVyZmFjZSBJQW1pQnVpbGRlciB7XG4gIC8qKlxuICAgKiBGaW5hbGl6ZSBhbmQgcmV0dXJuIGFsbCByZXF1aXJlZCBpbmZvcm1hdGlvbiBhYm91dCB0aGUgQU1JIGJ1aWx0IGJ5IHRoaXMgYnVpbGRlci5cbiAgICpcbiAgICogVGhpcyBtZXRob2QgY2FuIGJlIGNhbGxlZCBtdWx0aXBsZSB0aW1lcyBpZiB0aGUgaW1hZ2UgaXMgYm91bmQgdG8gbXVsdGlwbGUgcHJvdmlkZXJzLiBNYWtlIHN1cmUgeW91IGNhY2hlIHRoZSBpbWFnZSB3aGVuIGltcGxlbWVudGluZyBvciByZXR1cm4gYW4gZXJyb3IgaWYgdGhpcyBidWlsZGVyIGRvZXNuJ3Qgc3VwcG9ydCByZXVzaW5nIGltYWdlcy5cbiAgICpcbiAgICogQHJldHVybiBhbWlcbiAgICovXG4gIGJpbmQoKTogUnVubmVyQW1pO1xufVxuXG4vKipcbiAqIENvbW1vbiBwcm9wZXJ0aWVzIGZvciBhbGwgcnVubmVyIHByb3ZpZGVycy5cbiAqL1xuZXhwb3J0IGludGVyZmFjZSBSdW5uZXJQcm92aWRlclByb3BzIHtcbiAgLyoqXG4gICAqIFRoZSBudW1iZXIgb2YgZGF5cyBsb2cgZXZlbnRzIGFyZSBrZXB0IGluIENsb3VkV2F0Y2ggTG9ncy4gV2hlbiB1cGRhdGluZ1xuICAgKiB0aGlzIHByb3BlcnR5LCB1bnNldHRpbmcgaXQgZG9lc24ndCByZW1vdmUgdGhlIGxvZyByZXRlbnRpb24gcG9saWN5LiBUb1xuICAgKiByZW1vdmUgdGhlIHJldGVudGlvbiBwb2xpY3ksIHNldCB0aGUgdmFsdWUgdG8gYElORklOSVRFYC5cbiAgICpcbiAgICogQGRlZmF1bHQgbG9ncy5SZXRlbnRpb25EYXlzLk9ORV9NT05USFxuICAgKi9cbiAgcmVhZG9ubHkgbG9nUmV0ZW50aW9uPzogbG9ncy5SZXRlbnRpb25EYXlzO1xufVxuXG4vKipcbiAqIFdvcmtmbG93IGpvYiBwYXJhbWV0ZXJzIGFzIHBhcnNlZCBmcm9tIHRoZSB3ZWJob29rIGV2ZW50LiBQYXNzIHRoZXNlIGludG8geW91ciBydW5uZXIgZXhlY3V0b3IgYW5kIHJ1biBzb21ldGhpbmcgbGlrZTpcbiAqXG4gKiBgYGBzaFxuICogLi9jb25maWcuc2ggLS11bmF0dGVuZGVkIC0tdXJsIFwiaHR0cHM6Ly8ke0dJVEhVQl9ET01BSU59LyR7T1dORVJ9LyR7UkVQT31cIiAtLXRva2VuIFwiJHtSVU5ORVJfVE9LRU59XCIgLS1lcGhlbWVyYWwgLS13b3JrIF93b3JrIC0tbGFiZWxzIFwiJHtSVU5ORVJfTEFCRUx9XCIgLS1uYW1lIFwiJHtSVU5ORVJfTkFNRX1cIiAtLWRpc2FibGV1cGRhdGVcbiAqIGBgYFxuICpcbiAqIEFsbCBwYXJhbWV0ZXJzIGFyZSBzcGVjaWZpZWQgYXMgc3RlcCBmdW5jdGlvbiBwYXRocyBhbmQgdGhlcmVmb3JlIG11c3QgYmUgdXNlZCBvbmx5IGluIHN0ZXAgZnVuY3Rpb24gdGFzayBwYXJhbWV0ZXJzLlxuICovXG5leHBvcnQgaW50ZXJmYWNlIFJ1bm5lclJ1bnRpbWVQYXJhbWV0ZXJzIHtcbiAgLyoqXG4gICAqIFBhdGggdG8gcnVubmVyIHRva2VuIHVzZWQgdG8gcmVnaXN0ZXIgdG9rZW4uXG4gICAqL1xuICByZWFkb25seSBydW5uZXJUb2tlblBhdGg6IHN0cmluZztcblxuICAvKipcbiAgICogUGF0aCB0byBkZXNpcmVkIHJ1bm5lciBuYW1lLiBXZSBzcGVjaWZpY2FsbHkgc2V0IHRoZSBuYW1lIHRvIG1ha2UgdHJvdWJsZXNob290aW5nIGVhc2llci5cbiAgICovXG4gIHJlYWRvbmx5IHJ1bm5lck5hbWVQYXRoOiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIFBhdGggdG8gR2l0SHViIGRvbWFpbi4gTW9zdCBvZiB0aGUgdGltZSB0aGlzIHdpbGwgYmUgZ2l0aHViLmNvbSBidXQgZm9yIHNlbGYtaG9zdGVkIEdpdEh1YiBpbnN0YW5jZXMsIHRoaXMgd2lsbCBiZSBkaWZmZXJlbnQuXG4gICAqL1xuICByZWFkb25seSBnaXRodWJEb21haW5QYXRoOiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIFBhdGggdG8gcmVwb3N0aXJveSBvd25lciBuYW1lLlxuICAgKi9cbiAgcmVhZG9ubHkgb3duZXJQYXRoOiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIFBhdGggdG8gcmVwb3NpdG9yeSBuYW1lLlxuICAgKi9cbiAgcmVhZG9ubHkgcmVwb1BhdGg6IHN0cmluZztcbn1cblxuLyoqXG4gKiBJbWFnZSBzdGF0dXMgcmV0dXJuZWQgZnJvbSBydW5uZXIgcHJvdmlkZXJzIHRvIGJlIGRpc3BsYXllZCBpbiBzdGF0dXMuanNvbi5cbiAqL1xuZXhwb3J0IGludGVyZmFjZSBJUnVubmVySW1hZ2VTdGF0dXMge1xuICAvKipcbiAgICogSW1hZ2UgcmVwb3NpdG9yeSB3aGVyZSBpbWFnZSBidWlsZGVyIHB1c2hlcyBydW5uZXIgaW1hZ2VzLlxuICAgKi9cbiAgcmVhZG9ubHkgaW1hZ2VSZXBvc2l0b3J5OiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIFRhZyBvZiBpbWFnZSB0aGF0IHNob3VsZCBiZSB1c2VkLlxuICAgKi9cbiAgcmVhZG9ubHkgaW1hZ2VUYWc6IHN0cmluZztcblxuICAvKipcbiAgICogTG9nIGdyb3VwIG5hbWUgZm9yIHRoZSBpbWFnZSBidWlsZGVyIHdoZXJlIGhpc3Rvcnkgb2YgaW1hZ2UgYnVpbGRzIGNhbiBiZSBhbmFseXplZC5cbiAgICovXG4gIHJlYWRvbmx5IGltYWdlQnVpbGRlckxvZ0dyb3VwPzogc3RyaW5nO1xufVxuXG4vKipcbiAqIEFNSSBzdGF0dXMgcmV0dXJuZWQgZnJvbSBydW5uZXIgcHJvdmlkZXJzIHRvIGJlIGRpc3BsYXllZCBhcyBvdXRwdXQgb2Ygc3RhdHVzIGZ1bmN0aW9uLlxuICovXG5leHBvcnQgaW50ZXJmYWNlIElSdW5uZXJBbWlTdGF0dXMge1xuICAvKipcbiAgICogSWQgb2YgbGF1bmNoIHRlbXBsYXRlIHBvaW50aW5nIHRvIHRoZSBsYXRlc3QgQU1JIGJ1aWx0IGJ5IHRoZSBBTUkgYnVpbGRlci5cbiAgICovXG4gIHJlYWRvbmx5IGxhdW5jaFRlbXBsYXRlOiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIExvZyBncm91cCBuYW1lIGZvciB0aGUgQU1JIGJ1aWxkZXIgd2hlcmUgaGlzdG9yeSBvZiBidWlsZHMgY2FuIGJlIGFuYWx5emVkLlxuICAgKi9cbiAgcmVhZG9ubHkgYW1pQnVpbGRlckxvZ0dyb3VwPzogc3RyaW5nO1xufVxuXG4vKipcbiAqIEludGVyZmFjZSBmb3IgcnVubmVyIGltYWdlIHN0YXR1cyB1c2VkIGJ5IHN0YXR1cy5qc29uLlxuICovXG5leHBvcnQgaW50ZXJmYWNlIElSdW5uZXJQcm92aWRlclN0YXR1cyB7XG4gIC8qKlxuICAgKiBSdW5uZXIgcHJvdmlkZXIgdHlwZS5cbiAgICovXG4gIHJlYWRvbmx5IHR5cGU6IHN0cmluZztcblxuICAvKipcbiAgICogTGFiZWxzIGFzc29jaWF0ZWQgd2l0aCBwcm92aWRlci5cbiAgICovXG4gIHJlYWRvbmx5IGxhYmVsczogc3RyaW5nW107XG5cbiAgLyoqXG4gICAqIFZQQyB3aGVyZSBydW5uZXJzIHdpbGwgYmUgbGF1bmNoZWQuXG4gICAqL1xuICByZWFkb25seSB2cGNBcm4/OiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIFNlY3VyaXR5IGdyb3VwIGF0dGFjaGVkIHRvIHJ1bm5lcnMuXG4gICAqL1xuICByZWFkb25seSBzZWN1cml0eUdyb3VwPzogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBSb2xlIGF0dGFjaGVkIHRvIHJ1bm5lcnMuXG4gICAqL1xuICByZWFkb25seSByb2xlQXJuPzogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBEZXRhaWxzIGFib3V0IERvY2tlciBpbWFnZSB1c2VkIGJ5IHRoaXMgcnVubmVyIHByb3ZpZGVyLlxuICAgKi9cbiAgcmVhZG9ubHkgaW1hZ2U/OiBJUnVubmVySW1hZ2VTdGF0dXM7XG5cbiAgLyoqXG4gICAqIERldGFpbHMgYWJvdXQgQU1JIHVzZWQgYnkgdGhpcyBydW5uZXIgcHJvdmlkZXIuXG4gICAqL1xuICByZWFkb25seSBhbWk/OiBJUnVubmVyQW1pU3RhdHVzO1xufVxuXG4vKipcbiAqIEludGVyZmFjZSBmb3IgYWxsIHJ1bm5lciBwcm92aWRlcnMuIEltcGxlbWVudGF0aW9ucyBjcmVhdGUgYWxsIHJlcXVpcmVkIHJlc291cmNlcyBhbmQgcmV0dXJuIGEgc3RlcCBmdW5jdGlvbiB0YXNrIHRoYXQgc3RhcnRzIHRob3NlIHJlc291cmNlcyBmcm9tIHtAbGluayBnZXRTdGVwRnVuY3Rpb25UYXNrfS5cbiAqL1xuZXhwb3J0IGludGVyZmFjZSBJUnVubmVyUHJvdmlkZXIgZXh0ZW5kcyBlYzIuSUNvbm5lY3RhYmxlLCBpYW0uSUdyYW50YWJsZSB7XG4gIC8qKlxuICAgKiBHaXRIdWIgQWN0aW9ucyBsYWJlbHMgdXNlZCBmb3IgdGhpcyBwcm92aWRlci5cbiAgICpcbiAgICogVGhlc2UgbGFiZWxzIGFyZSB1c2VkIHRvIGlkZW50aWZ5IHdoaWNoIHByb3ZpZGVyIHNob3VsZCBzcGF3biBhIG5ldyBvbi1kZW1hbmQgcnVubmVyLiBFdmVyeSBqb2Igc2VuZHMgYSB3ZWJob29rIHdpdGggdGhlIGxhYmVscyBpdCdzIGxvb2tpbmcgZm9yXG4gICAqIGJhc2VkIG9uIHJ1bnMtb24uIFdlIHVzZSBtYXRjaCB0aGUgbGFiZWxzIGZyb20gdGhlIHdlYmhvb2sgd2l0aCB0aGUgbGFiZWxzIHNwZWNpZmllZCBoZXJlLiBJZiBhbGwgdGhlIGxhYmVscyBzcGVjaWZpZWQgaGVyZSBhcmUgcHJlc2VudCBpbiB0aGVcbiAgICogam9iJ3MgbGFiZWxzLCB0aGlzIHByb3ZpZGVyIHdpbGwgYmUgY2hvc2VuIGFuZCBzcGF3biBhIG5ldyBydW5uZXIuXG4gICAqL1xuICByZWFkb25seSBsYWJlbHM6IHN0cmluZ1tdO1xuXG4gIC8qKlxuICAgKiBWUEMgbmV0d29yayBpbiB3aGljaCBydW5uZXJzIHdpbGwgYmUgcGxhY2VkLlxuICAgKi9cbiAgcmVhZG9ubHkgdnBjPzogZWMyLklWcGM7XG5cbiAgLyoqXG4gICAqIFNlY3VyaXR5IGdyb3VwIGFzc29jaWF0ZWQgd2l0aCBydW5uZXJzLlxuICAgKi9cbiAgcmVhZG9ubHkgc2VjdXJpdHlHcm91cD86IGVjMi5JU2VjdXJpdHlHcm91cDtcblxuICAvKipcbiAgICogR2VuZXJhdGUgc3RlcCBmdW5jdGlvbiB0YXNrcyB0aGF0IGV4ZWN1dGUgdGhlIHJ1bm5lci5cbiAgICpcbiAgICogQ2FsbGVkIGJ5IEdpdGh1YlJ1bm5lcnMgYW5kIHNob3VsZG4ndCBiZSBjYWxsZWQgbWFudWFsbHkuXG4gICAqXG4gICAqIEBwYXJhbSBwYXJhbWV0ZXJzIHNwZWNpZmljIGJ1aWxkIHBhcmFtZXRlcnNcbiAgICovXG4gIGdldFN0ZXBGdW5jdGlvblRhc2socGFyYW1ldGVyczogUnVubmVyUnVudGltZVBhcmFtZXRlcnMpOiBzdGVwZnVuY3Rpb25zLklDaGFpbmFibGU7XG5cbiAgLyoqXG4gICAqIEFuIG9wdGlvbmFsIG1ldGhvZCB0aGF0IG1vZGlmaWVzIHRoZSByb2xlIG9mIHRoZSBzdGF0ZSBtYWNoaW5lIGFmdGVyIGFsbCB0aGUgdGFza3MgaGF2ZSBiZWVuIGdlbmVyYXRlZC4gVGhpcyBjYW4gYmUgdXNlZCB0byBhZGQgYWRkaXRpb25hbCBwb2xpY3lcbiAgICogc3RhdGVtZW50cyB0byB0aGUgc3RhdGUgbWFjaGluZSByb2xlIHRoYXQgYXJlIG5vdCBhdXRvbWF0aWNhbGx5IGFkZGVkIGJ5IHRoZSB0YXNrIHJldHVybmVkIGZyb20ge0BsaW5rIGdldFN0ZXBGdW5jdGlvblRhc2t9LlxuICAgKlxuICAgKiBAcGFyYW0gc3RhdGVNYWNoaW5lUm9sZSByb2xlIGZvciB0aGUgc3RhdGUgbWFjaGluZSB0aGF0IGV4ZWN1dGVzIHRoZSB0YXNrIHJldHVybmVkIGZyb20ge0BsaW5rIGdldFN0ZXBGdW5jdGlvblRhc2t9LlxuICAgKi9cbiAgZ3JhbnRTdGF0ZU1hY2hpbmUoc3RhdGVNYWNoaW5lUm9sZTogaWFtLklHcmFudGFibGUpOiB2b2lkO1xuXG4gIC8qKlxuICAgKiBSZXR1cm4gc3RhdHVzIG9mIHRoZSBydW5uZXIgcHJvdmlkZXIgdG8gYmUgdXNlZCBpbiB0aGUgbWFpbiBzdGF0dXMgZnVuY3Rpb24uIEFsc28gZ2l2ZXMgdGhlIHN0YXR1cyBmdW5jdGlvbiBhbnkgbmVlZGVkIHBlcm1pc3Npb25zIHRvIHF1ZXJ5IHRoZSBEb2NrZXIgaW1hZ2Ugb3IgQU1JLlxuICAgKlxuICAgKiBAcGFyYW0gc3RhdHVzRnVuY3Rpb25Sb2xlIGdyYW50YWJsZSBmb3IgdGhlIHN0YXR1cyBmdW5jdGlvblxuICAgKi9cbiAgc3RhdHVzKHN0YXR1c0Z1bmN0aW9uUm9sZTogaWFtLklHcmFudGFibGUpOiBJUnVubmVyUHJvdmlkZXJTdGF0dXM7XG59XG5cbi8qKlxuICogQmFzZSBjbGFzcyBmb3IgYWxsIHByb3ZpZGVycyB3aXRoIGNvbW1vbiBtZXRob2RzIHVzZWQgYnkgYWxsIHByb3ZpZGVycy5cbiAqXG4gKiBAaW50ZXJuYWxcbiAqL1xuZXhwb3J0IGFic3RyYWN0IGNsYXNzIEJhc2VQcm92aWRlciBleHRlbmRzIENvbnN0cnVjdCB7XG4gIHByb3RlY3RlZCBsYWJlbHNGcm9tUHJvcGVydGllcyhkZWZhdWx0TGFiZWw6IHN0cmluZywgcHJvcHNMYWJlbDogc3RyaW5nIHwgdW5kZWZpbmVkLCBwcm9wc0xhYmVsczogc3RyaW5nW10gfCB1bmRlZmluZWQpOiBzdHJpbmdbXSB7XG4gICAgaWYgKHByb3BzTGFiZWxzICYmIHByb3BzTGFiZWwpIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcignTXVzdCBzdXBwbHkgZWl0aGVyIGBsYWJlbGAgb3IgYGxhYmVsc2AgaW4gcnVubmVyIHByb3BlcnRpZXMsIGJ1dCBub3QgYm90aC4gVHJ5IHJlbW92aW5nIHRoZSBgbGFiZWxgIHByb3BlcnR5LicpO1xuICAgIH1cblxuICAgIGlmIChwcm9wc0xhYmVscykge1xuICAgICAgcmV0dXJuIHByb3BzTGFiZWxzO1xuICAgIH1cbiAgICBpZiAocHJvcHNMYWJlbCkge1xuICAgICAgcmV0dXJuIFtwcm9wc0xhYmVsXTtcbiAgICB9XG4gICAgcmV0dXJuIFtkZWZhdWx0TGFiZWxdO1xuICB9XG59XG4iXX0=