import unittest
import tempfile
import grpc
import concurrent.futures
import contextlib

import garrison


class ServerTestCase(
    unittest.TestCase,
):
    @classmethod
    def setUpClass(
        cls,
    ):
        cls.database_path = tempfile.TemporaryDirectory()

    @classmethod
    def tearDownClass(
        cls,
    ):
        cls.database_path.cleanup()

    @contextlib.contextmanager
    def ClientStub(
        self,
    ):
        garrison_server_servicer = garrison.server.GarrisonServerServicer(
            database_path=self.database_path.name,
        )
        server = grpc.server(
            thread_pool=concurrent.futures.ThreadPoolExecutor(
                max_workers=1,
            ),
        )
        garrison.garrison_pb2_grpc.add_GarrisonServerServicer_to_server(
            servicer=garrison_server_servicer,
            server=server,
        )
        port = server.add_insecure_port(
            address='[::]:0',
        )
        server.start()

        try:
            yield garrison.client.GarrisonClient(
                host='127.0.0.1',
                port=port,
            )
        finally:
            server.stop(None)

    def test_key(
        self,
    ):
        with self.ClientStub() as client:
            client.key_delete(
                key=b'key_1',
            )

            self.assertEqual(
                first=client.key_delete(
                    key=b'key_1',
                ),
                second=False,
            )
            self.assertEqual(
                first=client.key_set(
                    key=b'key_1',
                    value=b'value_1',
                ),
                second=True,
            )
            self.assertEqual(
                first=client.key_get(
                    key=b'key_1',
                ),
                second=b'value_1',
            )
            self.assertEqual(
                first=client.key_get(
                    key=b'key_1',
                ),
                second=b'value_1',
            )
            self.assertEqual(
                first=client.key_set(
                    key=b'key_1',
                    value=b'value_1',
                ),
                second=False,
            )
            self.assertEqual(
                first=client.key_get(
                    key=b'key_1',
                ),
                second=b'value_1',
            )
            self.assertEqual(
                first=client.key_set(
                    key=b'key_1',
                    value=b'value_2',
                ),
                second=False,
            )
            self.assertEqual(
                first=client.key_get(
                    key=b'key_1',
                ),
                second=b'value_2',
            )
            self.assertEqual(
                first=client.key_delete(
                    key=b'key_1',
                ),
                second=True,
            )
            self.assertEqual(
                first=client.key_get(
                    key=b'key_1',
                ),
                second=None,
            )

    def test_queue_single_values(
        self,
    ):
        with self.ClientStub() as client:
            client.queue_delete(
                queue_name='test_queue',
            )

            self.assertEqual(
                first=client.queue_pop(
                    queue_name='test_queue',
                    number_of_items=1,
                ),
                second=[],
            )

            self.assertEqual(
                first=client.queue_push(
                    queue_name='test_queue',
                    items=[
                        b'test_data',
                    ],
                    priority='NORMAL',
                ),
                second=True,
            )
            self.assertEqual(
                first=client.queue_pop(
                    queue_name='test_queue',
                    number_of_items=1,
                ),
                second=[
                    b'test_data',
                ],
            )
            self.assertEqual(
                first=client.queue_pop(
                    queue_name='test_queue',
                    number_of_items=1,
                ),
                second=[],
            )

    def test_queue_multiple_values(
        self,
    ):
        with self.ClientStub() as client:
            client.queue_delete(
                queue_name='test_queue',
            )

            self.assertEqual(
                first=client.queue_push(
                    queue_name='test_queue',
                    items=[
                        b'item_1',
                        b'item_2',
                        b'item_3',
                        b'item_4',
                        b'item_5',
                    ],
                    priority='NORMAL',
                ),
                second=True,
            )
            self.assertEqual(
                first=client.queue_pop(
                    queue_name='test_queue',
                    number_of_items=1,
                ),
                second=[
                    b'item_1',
                ],
            )
            self.assertEqual(
                first=client.queue_pop(
                    queue_name='test_queue',
                    number_of_items=4,
                ),
                second=[
                    b'item_2',
                    b'item_3',
                    b'item_4',
                    b'item_5',
                ],
            )

            self.assertEqual(
                first=client.queue_push(
                    queue_name='test_queue',
                    items=[
                        b'item_1',
                        b'item_2',
                        b'item_3',
                        b'item_4',
                        b'item_5',
                    ],
                    priority='NORMAL',
                ),
                second=True,
            )
            self.assertEqual(
                first=client.queue_pop(
                    queue_name='test_queue',
                    number_of_items=10,
                ),
                second=[
                    b'item_1',
                    b'item_2',
                    b'item_3',
                    b'item_4',
                    b'item_5',
                ],
            )
            self.assertEqual(
                first=client.queue_pop(
                    queue_name='test_queue',
                    number_of_items=4,
                ),
                second=[],
            )

    def test_queue_single_order(
        self,
    ):
        with self.ClientStub() as client:
            client.queue_delete(
                queue_name='test_queue',
            )

            self.assertEqual(
                first=client.queue_push(
                    queue_name='test_queue',
                    items=[
                        b'item_1',
                    ],
                    priority='NORMAL',
                ),
                second=True,
            )
            self.assertEqual(
                first=client.queue_push(
                    queue_name='test_queue',
                    items=[
                        b'item_2',
                    ],
                    priority='NORMAL',
                ),
                second=True,
            )
            self.assertEqual(
                first=client.queue_pop(
                    queue_name='test_queue',
                    number_of_items=1,
                ),
                second=[
                    b'item_1',
                ],
            )
            self.assertEqual(
                first=client.queue_pop(
                    queue_name='test_queue',
                    number_of_items=1,
                ),
                second=[
                    b'item_2',
                ],
            )

            self.assertEqual(
                first=client.queue_push(
                    queue_name='test_queue',
                    items=[
                        b'item_1',
                    ],
                    priority='HIGH',
                ),
                second=True,
            )
            self.assertEqual(
                first=client.queue_push(
                    queue_name='test_queue',
                    items=[
                        b'item_2',
                    ],
                    priority='HIGH',
                ),
                second=True,
            )
            self.assertEqual(
                first=client.queue_pop(
                    queue_name='test_queue',
                    number_of_items=1,
                ),
                second=[
                    b'item_2',
                ],
            )
            self.assertEqual(
                first=client.queue_pop(
                    queue_name='test_queue',
                    number_of_items=1,
                ),
                second=[
                    b'item_1',
                ],
            )

    def test_queue_multiple_order(
        self,
    ):
        with self.ClientStub() as client:
            client.queue_delete(
                queue_name='test_queue',
            )

            self.assertEqual(
                first=client.queue_push(
                    queue_name='test_queue',
                    items=[
                        b'item_1',
                        b'item_2',
                        b'item_3',
                        b'item_4',
                        b'item_5',
                    ],
                    priority='NORMAL',
                ),
                second=True,
            )
            self.assertEqual(
                first=client.queue_pop(
                    queue_name='test_queue',
                    number_of_items=5,
                ),
                second=[
                    b'item_1',
                    b'item_2',
                    b'item_3',
                    b'item_4',
                    b'item_5',
                ],
            )

            self.assertEqual(
                first=client.queue_push(
                    queue_name='test_queue',
                    items=[
                        b'item_1',
                        b'item_2',
                        b'item_3',
                        b'item_4',
                        b'item_5',
                    ],
                    priority='HIGH',
                ),
                second=True,
            )
            self.assertEqual(
                first=client.queue_pop(
                    queue_name='test_queue',
                    number_of_items=5,
                ),
                second=[
                    b'item_1',
                    b'item_2',
                    b'item_3',
                    b'item_4',
                    b'item_5',
                ],
            )

    def test_queue_priority(
        self,
    ):
        with self.ClientStub() as client:
            client.queue_delete(
                queue_name='test_queue',
            )

            self.assertEqual(
                first=client.queue_push(
                    queue_name='test_queue',
                    items=[
                        b'item_1',
                    ],
                    priority='NORMAL',
                ),
                second=True,
            )
            self.assertEqual(
                first=client.queue_push(
                    queue_name='test_queue',
                    items=[
                        b'item_2',
                    ],
                    priority='NORMAL',
                ),
                second=True,
            )
            self.assertEqual(
                first=client.queue_pop(
                    queue_name='test_queue',
                    number_of_items=1,
                ),
                second=[
                    b'item_1',
                ],
            )
            self.assertEqual(
                first=client.queue_pop(
                    queue_name='test_queue',
                    number_of_items=1,
                ),
                second=[
                    b'item_2',
                ],
            )

            self.assertEqual(
                first=client.queue_push(
                    queue_name='test_queue',
                    items=[
                        b'item_1',
                    ],
                    priority='NORMAL',
                ),
                second=True,
            )
            self.assertEqual(
                first=client.queue_push(
                    queue_name='test_queue',
                    items=[
                        b'item_2',
                    ],
                    priority='HIGH',
                ),
                second=True,
            )
            self.assertEqual(
                first=client.queue_pop(
                    queue_name='test_queue',
                    number_of_items=1,
                ),
                second=[
                    b'item_2',
                ],
            )
            self.assertEqual(
                first=client.queue_pop(
                    queue_name='test_queue',
                    number_of_items=1,
                ),
                second=[
                    b'item_1',
                ],
            )

            self.assertEqual(
                first=client.queue_push(
                    queue_name='test_queue',
                    items=[
                        b'item_1',
                        b'item_2',
                        b'item_3',
                        b'item_4',
                        b'item_5',
                    ],
                    priority='NORMAL',
                ),
                second=True,
            )
            self.assertEqual(
                first=client.queue_push(
                    queue_name='test_queue',
                    items=[
                        b'item_6',
                        b'item_7',
                        b'item_8',
                        b'item_9',
                        b'item_10',
                    ],
                    priority='HIGH',
                ),
                second=True,
            )
            self.assertEqual(
                first=client.queue_pop(
                    queue_name='test_queue',
                    number_of_items=5,
                ),
                second=[
                    b'item_6',
                    b'item_7',
                    b'item_8',
                    b'item_9',
                    b'item_10',
                ],
            )
            self.assertEqual(
                first=client.queue_pop(
                    queue_name='test_queue',
                    number_of_items=5,
                ),
                second=[
                    b'item_1',
                    b'item_2',
                    b'item_3',
                    b'item_4',
                    b'item_5',
                ],
            )

    def test_queue_delete_and_length(
        self,
    ):
        with self.ClientStub() as client:
            client.queue_delete(
                queue_name='test_queue',
            )

            self.assertEqual(
                first=client.queue_push(
                    queue_name='test_queue',
                    items=[
                        b'item_1',
                    ],
                    priority='NORMAL',
                ),
                second=True,
            )
            self.assertEqual(
                first=client.queue_length(
                    queue_name='test_queue',
                ),
                second=1,
            )
            self.assertEqual(
                first=client.queue_delete(
                    queue_name='test_queue',
                ),
                second=True,
            )
            self.assertEqual(
                first=client.queue_length(
                    queue_name='test_queue',
                ),
                second=0,
            )

            self.assertEqual(
                first=client.queue_push(
                    queue_name='test_queue',
                    items=[
                            b'item_1',
                    ] * 1000,
                    priority='NORMAL',
                ),
                second=True,
            )
            self.assertEqual(
                first=client.queue_length(
                    queue_name='test_queue',
                ),
                second=1000,
            )
            self.assertEqual(
                first=client.queue_delete(
                    queue_name='test_queue',
                ),
                second=True,
            )
            self.assertEqual(
                first=client.queue_length(
                    queue_name='test_queue',
                ),
                second=0,
            )

            self.assertEqual(
                first=client.queue_push(
                    queue_name='test_queue',
                    items=[
                            b'item_1',
                    ] * 1000,
                    priority='NORMAL',
                ),
                second=True,
            )
            self.assertEqual(
                first=client.queue_pop(
                    queue_name='test_queue',
                    number_of_items=500,
                ),
                second=[
                    b'item_1',
                ] * 500,
            )
            self.assertEqual(
                first=client.queue_length(
                    queue_name='test_queue',
                ),
                second=500,
            )
            self.assertEqual(
                first=client.queue_delete(
                    queue_name='test_queue',
                ),
                second=True,
            )
            self.assertEqual(
                first=client.queue_length(
                    queue_name='test_queue',
                ),
                second=0,
            )
