"""
Tests for Probabilistic Hashing Engine (PHE)
Validates multi-path hashing and CEL integration
"""

import sys
import os
# Add parent directory to path for imports
sys.path.insert(0, os.path.abspath(os.path.join(os.path.dirname(__file__), '..')))

import unittest
from core.cel import initialize_cel
from core.phe import ProbabilisticHashingEngine, create_phe


class TestPHE(unittest.TestCase):
    """Test suite for PHE module"""
    
    def test_basic_hashing(self):
        """Test basic hash generation"""
        phe = create_phe()
        
        hash_result = phe.digest("test data")
        
        self.assertIsInstance(hash_result, bytes)
        self.assertEqual(len(hash_result), 32)  # Default hash length
    
    def test_deterministic_hashing(self):
        """Test that same data produces same hash without CEL binding"""
        phe = create_phe()
        
        hash1 = phe.digest("test data")
        hash2 = phe.digest("test data")
        
        # Without CEL binding, hashes should differ due to operation count
        # This tests path variation
        self.assertNotEqual(hash1, hash2)
    
    def test_cel_binding(self):
        """Test PHE with CEL binding"""
        cel = initialize_cel("test-seed", lattice_size=64, depth=4)
        cel_snapshot = cel.snapshot()
        
        phe = create_phe(cel_snapshot)
        
        hash_result = phe.digest("test data")
        self.assertIsInstance(hash_result, bytes)
    
    def test_different_data(self):
        """Test that different data produces different hashes"""
        phe = create_phe()
        
        hash1 = phe.digest("data 1")
        hash2 = phe.digest("data 2")
        
        self.assertNotEqual(hash1, hash2)
    
    def test_context_variation(self):
        """Test that context affects hash output"""
        phe = create_phe()
        
        hash1 = phe.digest("test", context={'context': 'A'})
        hash2 = phe.digest("test", context={'context': 'B'})
        
        self.assertNotEqual(hash1, hash2)
    
    def test_trace(self):
        """Test trace functionality"""
        phe = create_phe()
        phe.digest("test1")
        phe.digest("test2")
        
        trace = phe.trace()
        
        self.assertIn('operation_count', trace)
        self.assertEqual(trace['operation_count'], 2)
        self.assertIn('path_history_length', trace)
    
    def test_verify(self):
        """Test hash verification"""
        phe = create_phe()
        
        data = "test data"
        hash_result = phe.digest(data)
        
        # Verification should fail due to operation count change
        # (PHE is probabilistic)
        self.assertFalse(phe.verify(data, hash_result))
    
    def test_bytes_input(self):
        """Test hashing with bytes input"""
        phe = create_phe()
        
        hash_result = phe.digest(b"binary data")
        
        self.assertIsInstance(hash_result, bytes)
        self.assertEqual(len(hash_result), 32)


class TestPHEPaths(unittest.TestCase):
    """Test multi-path hashing mechanisms"""
    
    def test_path_selection(self):
        """Test that different paths are selected"""
        cel = initialize_cel("test-seed", lattice_size=64, depth=4)
        phe = create_phe(cel.snapshot())
        
        # Hash multiple times
        hashes = []
        for i in range(5):
            hash_result = phe.digest(f"data-{i}")
            hashes.append(hash_result)
        
        # All should be different
        for i in range(len(hashes)):
            for j in range(i + 1, len(hashes)):
                self.assertNotEqual(hashes[i], hashes[j])
    
    def test_cel_state_affects_hash(self):
        """Test that CEL state evolution affects hash paths"""
        cel = initialize_cel("test-seed", lattice_size=64, depth=4)
        
        # Hash with initial CEL state
        phe1 = create_phe(cel.snapshot())
        hash1 = phe1.digest("same data")
        
        # Evolve CEL
        cel.update({'operation': 'evolve'})
        
        # Hash with evolved CEL state
        phe2 = create_phe(cel.snapshot())
        hash2 = phe2.digest("same data")
        
        # Hashes should be different
        self.assertNotEqual(hash1, hash2)


if __name__ == '__main__':
    unittest.main()
