"""
Integration tests for complete STC system
Tests end-to-end encryption, key derivation, and state management
"""

import sys
import os
# Add parent directory to path for imports
sys.path.insert(0, os.path.abspath(os.path.join(os.path.dirname(__file__), '..')))

import unittest
from interfaces.api import stc_api


class TestSTCIntegration(unittest.TestCase):
    """Integration tests for complete STC system"""
    
    def test_basic_encryption_decryption(self):
        """Test basic encrypt/decrypt cycle with separate contexts"""
        seed = "test-seed-12345"
        original = "Hello, Seigr Toolset Crypto!"
        
        # Encrypt with first context
        context1 = stc_api.initialize(seed)
        encrypted, metadata = context1.encrypt(original)
        
        # Decrypt with separate context (CEL state is restored from metadata)
        context2 = stc_api.initialize(seed)
        decrypted = context2.decrypt(encrypted, metadata)
        
        # Should match original
        self.assertEqual(original, decrypted)
    
    def test_binary_encryption(self):
        """Test encryption of binary data"""
        seed = "binary-seed"
        original = b"\x00\x01\x02\x03\x04\x05\xff\xfe\xfd"
        
        # Encrypt
        context1 = stc_api.initialize(seed)
        encrypted, metadata = context1.encrypt(original)
        
        # Decrypt with separate context
        context2 = stc_api.initialize(seed)
        decrypted = context2.decrypt(encrypted, metadata)
        
        self.assertEqual(original, decrypted)
    
    def test_large_data_encryption(self):
        """Test encryption of larger data"""
        seed = "large-seed"
        original = "A" * 10000
        
        # Encrypt
        context1 = stc_api.initialize(seed)
        encrypted, metadata = context1.encrypt(original)
        
        # Decrypt with separate context
        context2 = stc_api.initialize(seed)
        decrypted = context2.decrypt(encrypted, metadata)
        
        self.assertEqual(original, decrypted)
    
    def test_quick_encrypt_decrypt(self):
        """Test quick encryption/decryption functions"""
        seed = "quick-seed"
        original = "Quick test data"
        
        # Quick encrypt
        encrypted, metadata, enc_context = stc_api.quick_encrypt(original, seed)
        
        # Quick decrypt
        decrypted = stc_api.quick_decrypt(encrypted, metadata, seed)
        
        self.assertEqual(original, decrypted)
    
    def test_wrong_seed_fails(self):
        """Test that wrong seed fails decryption"""
        original = "Secret message"
        
        encrypted, metadata, _ = stc_api.quick_encrypt(original, "correct-seed")
        
        # Try to decrypt with wrong seed
        try:
            decrypted = stc_api.quick_decrypt(encrypted, metadata, "wrong-seed")
            # If decryption succeeds, it should produce garbage
            self.assertNotEqual(original, decrypted)
        except Exception:
            # Or it might raise an exception
            pass
    
    def test_deterministic_hashing(self):
        """Test deterministic hash generation"""
        context1 = stc_api.initialize("hash-seed")
        context2 = stc_api.initialize("hash-seed")
        
        # Same data, same seed, but different contexts
        # Hashes will differ due to CEL state evolution
        hash1 = context1.hash("test data")
        hash2 = context2.hash("test data")
        
        # Both should be valid hashes
        self.assertEqual(len(hash1), 32)
        self.assertEqual(len(hash2), 32)
    
    def test_key_derivation(self):
        """Test key derivation"""
        context = stc_api.initialize("derive-seed")
        
        key1 = context.derive_key(32)
        key2 = context.derive_key(64)
        
        self.assertEqual(len(key1), 32)
        self.assertEqual(len(key2), 64)
        self.assertNotEqual(key1, key2[:32])  # Different derivations
    
    def test_state_save_load(self):
        """Test state save and load"""
        context = stc_api.initialize("state-seed")
        
        # Perform operations
        context.hash("data1")
        context.hash("data2")
        
        # Save state
        state = context.save_state()
        
        self.assertIn('cel_state', state)
        self.assertIn('pcf_state', state)
    
    def test_context_status(self):
        """Test context status reporting"""
        context = stc_api.initialize("status-seed")
        
        # Perform operations
        context.hash("test")
        context.derive_key()
        
        status = context.get_status()
        
        self.assertIn("CEL Operation Count", status)
        self.assertIn("PCF Status", status)
    
    def test_multiple_encryptions_same_context(self):
        """Test multiple encryptions with proper decryption using separate contexts"""
        seed = "multi-seed"
        
        data1 = "First message"
        data2 = "Second message"
        data3 = "Third message"
        
        # Encrypt all with one context
        ctx_enc = stc_api.initialize(seed)
        enc1, meta1 = ctx_enc.encrypt(data1)
        enc2, meta2 = ctx_enc.encrypt(data2)
        enc3, meta3 = ctx_enc.encrypt(data3)
        
        # Decrypt each with fresh contexts (metadata contains the CEL state)
        ctx_dec1 = stc_api.initialize(seed)
        ctx_dec2 = stc_api.initialize(seed)
        ctx_dec3 = stc_api.initialize(seed)
        
        dec1 = ctx_dec1.decrypt(enc1, meta1)
        dec2 = ctx_dec2.decrypt(enc2, meta2)
        dec3 = ctx_dec3.decrypt(enc3, meta3)
        
        self.assertEqual(data1, dec1)
        self.assertEqual(data2, dec2)
        self.assertEqual(data3, dec3)
    
    def test_reproducibility(self):
        """Test that encryption is deterministic when using the same CEL snapshot"""
        seed = "reproducible-seed"
        data = "Reproducible data"
        
        # First encryption
        ctx1 = stc_api.initialize(seed)
        enc1, meta1 = ctx1.encrypt(data)
        
        # Second encryption with same seed BUT different CEL state (due to timing entropy)
        ctx2 = stc_api.initialize(seed)
        enc2, meta2 = ctx2.encrypt(data)
        
        # Encrypted data will DIFFER because CEL evolves with environmental entropy
        # This is intentional - provides probabilistic encryption
        self.assertNotEqual(enc1, enc2, "Encryption should vary due to CEL evolution with timing entropy")
        
        # However, both should decrypt correctly using their respective CEL snapshots
        dec_ctx1 = stc_api.initialize(seed)
        dec_ctx2 = stc_api.initialize(seed)
        
        dec1 = dec_ctx1.decrypt(enc1, meta1)
        dec2 = dec_ctx2.decrypt(enc2, meta2)
        
        # Both decrypt to the same original data
        self.assertEqual(data, dec1)
        self.assertEqual(data, dec2)


class TestPolymorphicBehavior(unittest.TestCase):
    """Test polymorphic flow and morphing"""
    
    def test_pcf_morphing(self):
        """Test that PCF morphs over time"""
        context = stc_api.initialize("morph-seed", morph_interval=10)
        
        initial_meta = context.pcf.get_meta_state()
        
        # Perform operations to trigger morph
        for i in range(15):
            context.hash(f"data-{i}")
        
        morphed_meta = context.pcf.get_meta_state()
        
        # Meta-state should have changed
        self.assertNotEqual(initial_meta, morphed_meta)
    
    def test_entropy_evolution(self):
        """Test CEL entropy evolution"""
        context = stc_api.initialize("evolution-seed")
        
        initial_hash = context.cel.get_state_hash()
        
        # Perform operations
        for i in range(10):
            context.hash(f"data-{i}")
        
        evolved_hash = context.cel.get_state_hash()
        
        # State should have evolved
        self.assertNotEqual(initial_hash, evolved_hash)


class TestEdgeCases(unittest.TestCase):
    """Test edge cases and error handling"""
    
    def test_empty_string_encryption(self):
        """Test encryption of empty string"""
        seed = "empty-seed"
        
        # Empty string should raise error or handle gracefully
        try:
            ctx1 = stc_api.initialize(seed)
            encrypted, metadata = ctx1.encrypt("")
            # If it succeeds, should be able to decrypt with fresh context
            ctx2 = stc_api.initialize(seed)
            decrypted = ctx2.decrypt(encrypted, metadata)
            self.assertEqual("", decrypted)
        except ValueError:
            # Or it might raise ValueError for empty data
            pass
    
    def test_unicode_encryption(self):
        """Test encryption of Unicode data"""
        seed = "unicode-seed"
        original = "Hello 世界 🌍 Привет"
        
        # Encrypt
        ctx1 = stc_api.initialize(seed)
        encrypted, metadata = ctx1.encrypt(original)
        
        # Decrypt with fresh context
        ctx2 = stc_api.initialize(seed)
        decrypted = ctx2.decrypt(encrypted, metadata)
        
        self.assertEqual(original, decrypted)
    
    def test_very_small_lattice(self):
        """Test with minimal lattice size"""
        seed = "small-seed"
        data = "Test with small lattice"
        
        # Encrypt with small lattice
        ctx1 = stc_api.initialize(seed, lattice_size=16, depth=2)
        encrypted, metadata = ctx1.encrypt(data)
        
        # Decrypt with matching small lattice
        ctx2 = stc_api.initialize(seed, lattice_size=16, depth=2)
        decrypted = ctx2.decrypt(encrypted, metadata)
        
        self.assertEqual(data, decrypted)


if __name__ == '__main__':
    unittest.main()
