#!/usr/bin/env python3
"""
Test decoy vector functionality in v0.2.1
"""
import sys
from pathlib import Path

# Add parent directory to path for imports
sys.path.insert(0, str(Path(__file__).parent.parent))

from interfaces.api.stc_api import STCContext

print("Testing decoy vector encryption/decryption...")

ctx = STCContext('test-seed')

# Test with decoys enabled (default in v0.2.1)
print("\n1. Testing WITH decoys (use_decoys=True):")
encrypted, metadata = ctx.encrypt("Secret message", password="strong_pw", use_decoys=True, num_decoys=3)
print(f"   - Encrypted: {len(encrypted)} bytes")
print(f"   - Metadata: {len(metadata)} bytes ({len(metadata)//1024}KB)")

decrypted = ctx.decrypt(encrypted, metadata, password="strong_pw")
print(f"   - Decrypted: {decrypted}")
print(f"   - ✓ Success!" if decrypted == "Secret message" else "   - ✗ FAILED!")

# Test with wrong password (should fail)
print("\n2. Testing wrong password (should reject):")
try:
    bad_decrypt = ctx.decrypt(encrypted, metadata, password="wrong_password")
    print(f"   - ✗ SECURITY ISSUE: Decrypted with wrong password!")
except ValueError as e:
    print(f"   - ✓ Correctly rejected: {e}")

# Test without decoys
print("\n3. Testing WITHOUT decoys (use_decoys=False):")
encrypted2, metadata2 = ctx.encrypt("Another secret", password="pw2", use_decoys=False)
print(f"   - Encrypted: {len(encrypted2)} bytes")
print(f"   - Metadata: {len(metadata2)} bytes ({len(metadata2)//1024}KB)")

decrypted2 = ctx.decrypt(encrypted2, metadata2, password="pw2")
print(f"   - Decrypted: {decrypted2}")
print(f"   - ✓ Success!" if decrypted2 == "Another secret" else "   - ✗ FAILED!")

# Compare metadata sizes
print(f"\n4. Metadata size comparison:")
print(f"   - With decoys (3): {len(metadata)//1024}KB")
print(f"   - Without decoys: {len(metadata2)//1024}KB")
print(f"   - Overhead: {(len(metadata) - len(metadata2))//1024}KB (+{((len(metadata)/len(metadata2))-1)*100:.1f}%)")

print("\n✓ All decoy vector tests passed!")
