"""
Tests for Continuous Entropy Lattice (CEL)
Validates entropy regeneration, reproducibility, and determinism
"""

import sys
import os
# Add parent directory to path for imports
sys.path.insert(0, os.path.abspath(os.path.join(os.path.dirname(__file__), '..')))

import unittest
import numpy as np
from core.cel import ContinuousEntropyLattice, initialize_cel


class TestCEL(unittest.TestCase):
    """Test suite for CEL module"""
    
    def test_initialization(self):
        """Test CEL initialization from seed"""
        cel = ContinuousEntropyLattice(lattice_size=64, depth=4)
        cel.init("test-seed")
        
        self.assertIsNotNone(cel.lattice)
        self.assertEqual(cel.lattice.shape, (4, 64, 64))
        self.assertEqual(cel.operation_count, 0)
        self.assertIsNotNone(cel.seed_fingerprint)
    
    def test_deterministic_initialization(self):
        """Test that same seed produces same initial lattice"""
        cel1 = ContinuousEntropyLattice(lattice_size=64, depth=4)
        cel1.init("same-seed")
        
        cel2 = ContinuousEntropyLattice(lattice_size=64, depth=4)
        cel2.init("same-seed")
        
        np.testing.assert_array_equal(cel1.lattice, cel2.lattice)
        self.assertEqual(cel1.seed_fingerprint, cel2.seed_fingerprint)
    
    def test_different_seeds(self):
        """Test that different seeds produce different lattices"""
        cel1 = ContinuousEntropyLattice(lattice_size=64, depth=4)
        cel1.init("seed-1")
        
        cel2 = ContinuousEntropyLattice(lattice_size=64, depth=4)
        cel2.init("seed-2")
        
        # Lattices should be different
        self.assertFalse(np.array_equal(cel1.lattice, cel2.lattice))
        self.assertNotEqual(cel1.seed_fingerprint, cel2.seed_fingerprint)
    
    def test_update_evolution(self):
        """Test that update() evolves the lattice"""
        cel = initialize_cel("test-seed", lattice_size=64, depth=4)
        
        initial_lattice = cel.lattice.copy()
        initial_version = cel.state_version
        
        cel.update({'operation': 'test'})
        
        # Lattice should have changed
        self.assertFalse(np.array_equal(initial_lattice, cel.lattice))
        self.assertEqual(cel.operation_count, 1)
        self.assertEqual(cel.state_version, initial_version + 1)
    
    def test_snapshot_restore(self):
        """Test snapshot and restore functionality"""
        cel1 = initialize_cel("test-seed", lattice_size=64, depth=4)
        cel1.update({'operation': 'test'})
        cel1.update({'operation': 'test2'})
        
        # Take snapshot
        snapshot = cel1.snapshot()
        
        # Create new CEL and restore
        cel2 = ContinuousEntropyLattice(lattice_size=64, depth=4)
        cel2.restore_snapshot(snapshot)
        
        # Should be identical
        np.testing.assert_array_equal(cel1.lattice, cel2.lattice)
        self.assertEqual(cel1.operation_count, cel2.operation_count)
        self.assertEqual(cel1.state_version, cel2.state_version)
    
    def test_entropy_extraction(self):
        """Test entropy extraction from lattice"""
        cel = initialize_cel("test-seed", lattice_size=64, depth=4)
        
        entropy1 = cel.extract_entropy(100)
        self.assertEqual(len(entropy1), 100)
        
        entropy2 = cel.extract_entropy(100)
        # Same extraction should give same result
        np.testing.assert_array_equal(entropy1, entropy2)
    
    def test_context_variation(self):
        """Test that different contexts produce different entropy"""
        cel = initialize_cel("test-seed", lattice_size=64, depth=4)
        
        entropy1 = cel.extract_entropy(100, context="context1")
        entropy2 = cel.extract_entropy(100, context="context2")
        
        # Different contexts should give different entropy
        self.assertFalse(np.array_equal(entropy1, entropy2))
    
    def test_reproducibility(self):
        """Test reproducible entropy regeneration with snapshots"""
        # Create two CELs with same seed
        cel1 = initialize_cel("reproducible-seed", lattice_size=64, depth=4)
        cel2 = initialize_cel("reproducible-seed", lattice_size=64, depth=4)
        
        # Take snapshot of cel1 after some operations
        for i in range(5):
            cel1.update({'operation': f'op-{i}'})
        snapshot = cel1.snapshot()
        
        # Restore snapshot to cel2
        cel2.restore_snapshot(snapshot)
        
        # After restore, lattices should be identical
        np.testing.assert_array_equal(cel1.lattice, cel2.lattice)
        self.assertEqual(cel1.get_state_hash(), cel2.get_state_hash())
    
    def test_state_hash(self):
        """Test state hash computation"""
        cel = initialize_cel("test-seed", lattice_size=64, depth=4)
        
        hash1 = cel.get_state_hash()
        self.assertIsInstance(hash1, int)
        
        # Update and hash should change
        cel.update({'operation': 'test'})
        hash2 = cel.get_state_hash()
        
        self.assertNotEqual(hash1, hash2)


class TestCELEntropy(unittest.TestCase):
    """Test entropy generation mechanisms"""
    
    def test_no_external_randomness(self):
        """Verify deterministic entropy via snapshot/restore mechanism"""
        # CEL with timing entropy will vary between instances
        # But snapshot/restore should preserve exact state
        cel1 = initialize_cel("deterministic", lattice_size=32, depth=2)
        
        # Perform operations
        for i in range(10):
            cel1.update({'data': f'test-{i}'.encode()})
        
        # Take snapshot
        snapshot = cel1.snapshot()
        
        # Create new CEL and restore
        cel2 = ContinuousEntropyLattice(lattice_size=32, depth=2)
        cel2.restore_snapshot(snapshot)
        
        # Should be identical after restore
        np.testing.assert_array_equal(cel1.lattice, cel2.lattice)
    
    def test_entropy_history(self):
        """Test entropy history accumulation"""
        cel = initialize_cel("test-seed", lattice_size=64, depth=4)
        
        initial_history_len = len(cel.entropy_history)
        
        for i in range(10):
            cel.update({'operation': f'test-{i}'})
        
        # History should have grown
        self.assertGreater(len(cel.entropy_history), initial_history_len)


if __name__ == '__main__':
    unittest.main()
