#!/usr/bin/env python3
"""
Quick performance benchmark for v0.2.1
"""
import sys
from pathlib import Path

# Add parent directory to path for imports
sys.path.insert(0, str(Path(__file__).parent.parent))

import time
from interfaces.api.stc_api import STCContext

ctx = STCContext('test-seed')

# Warmup
e, m = ctx.encrypt('Warmup', password='pw')
d = ctx.decrypt(e, m, password='pw')

# Benchmark
times = []
for i in range(5):
    start = time.time()
    encrypted, metadata = ctx.encrypt(f'Test message {i}', password='strong_password')
    enc_time = time.time() - start
    
    start = time.time()
    decrypted = ctx.decrypt(encrypted, metadata, password='strong_password')
    dec_time = time.time() - start
    
    total = enc_time + dec_time
    times.append((enc_time, dec_time, total))
    print(f'Run {i+1}: Enc={enc_time:.3f}s  Dec={dec_time:.3f}s  Total={total:.3f}s  Metadata={len(metadata)//1024}KB')

# Statistics
avg_enc = sum(t[0] for t in times) / len(times)
avg_dec = sum(t[1] for t in times) / len(times)
avg_total = sum(t[2] for t in times) / len(times)

print(f'\n{"="*60}')
print(f'AVERAGE: Enc={avg_enc:.3f}s  Dec={avg_dec:.3f}s  Total={avg_total:.3f}s')
print(f'{"="*60}')
print(f'\nv0.2.0 baseline: 1.33s enc + 0.94s dec = 2.27s total, 786KB metadata')
print(f'v0.2.1 results:  {avg_enc:.2f}s enc + {avg_dec:.2f}s dec = {avg_total:.2f}s total, ~{len(metadata)//1024}KB metadata')

# Calculate improvements
speedup = 2.27 / avg_total
metadata_reduction = (786 - len(metadata)//1024) / 786 * 100
print(f'\nSpeedup: {speedup:.2f}x faster than v0.2.0')
print(f'Metadata reduction: {metadata_reduction:.1f}% smaller')
