#!/usr/bin/env python3
"""
STC CLI - Command-line interface for Seigr Toolset Crypto

Provides commands for encryption, hashing, and cryptographic operations.
"""

import argparse
import sys
import os
from typing import Optional

from interfaces.api import stc_api


def cmd_encrypt(args):
    """Encrypt data"""
    # Read input
    if args.input:
        with open(args.input, 'rb') as f:
            data = f.read()
    else:
        data = input("Enter data to encrypt: ")
    
    # Get seed
    seed = args.seed or input("Enter seed: ")
    
    # Initialize context
    print("Initializing STC context...")
    context = stc_api.initialize(seed)
    
    # Encrypt
    print("Encrypting...")
    encrypted, metadata = context.encrypt(data)
    
    # Output
    if args.output:
        with open(args.output, 'wb') as f:
            f.write(encrypted)
        
        # Save metadata
        metadata_file = args.output + ".meta"
        import json
        with open(metadata_file, 'w') as f:
            json.dump(metadata, f, indent=2)
        
        print(f"Encrypted data saved to: {args.output}")
        print(f"Metadata saved to: {metadata_file}")
    else:
        print(f"\nEncrypted (hex): {encrypted.hex()}")
        print(f"\nMetadata: {metadata}")


def cmd_decrypt(args):
    """Decrypt data"""
    # Read encrypted data
    if not args.input:
        print("Error: Input file required for decryption")
        sys.exit(1)
    
    with open(args.input, 'rb') as f:
        encrypted = f.read()
    
    # Read metadata
    metadata_file = args.metadata or (args.input + ".meta")
    if not os.path.exists(metadata_file):
        print(f"Error: Metadata file not found: {metadata_file}")
        sys.exit(1)
    
    import json
    with open(metadata_file, 'r') as f:
        metadata = json.load(f)
    
    # Get seed
    seed = args.seed or input("Enter seed: ")
    
    # Initialize context and decrypt
    print("Initializing STC context...")
    context = stc_api.initialize(seed)
    
    print("Decrypting...")
    try:
        decrypted = stc_api.quick_decrypt(encrypted, metadata, seed)
        
        # Output
        if args.output:
            if isinstance(decrypted, str):
                with open(args.output, 'w') as f:
                    f.write(decrypted)
            else:
                with open(args.output, 'wb') as f:
                    f.write(decrypted)
            print(f"Decrypted data saved to: {args.output}")
        else:
            if isinstance(decrypted, str):
                print(f"\nDecrypted: {decrypted}")
            else:
                print(f"\nDecrypted (hex): {decrypted.hex()}")
    except Exception as e:
        print(f"Error during decryption: {e}")
        sys.exit(1)


def cmd_hash(args):
    """Generate hash"""
    # Read input
    if args.input:
        with open(args.input, 'rb') as f:
            data = f.read()
    else:
        data = input("Enter data to hash: ")
    
    # Get seed
    seed = args.seed or input("Enter seed: ")
    
    # Initialize context
    context = stc_api.initialize(seed)
    
    # Hash
    hash_result = context.hash(data)
    
    print(f"\nHash (hex): {hash_result.hex()}")
    print(f"Hash (base64): {__import__('base64').b64encode(hash_result).decode()}")


def cmd_derive_key(args):
    """Derive key"""
    # Get seed
    seed = args.seed or input("Enter seed: ")
    
    # Get key length
    length = args.length or 32
    
    # Initialize context
    context = stc_api.initialize(seed)
    
    # Derive key
    key = context.derive_key(length=length)
    
    print(f"\nDerived Key ({length} bytes):")
    print(f"Hex: {key.hex()}")
    print(f"Base64: {__import__('base64').b64encode(key).decode()}")


def cmd_status(args):
    """Show context status"""
    seed = args.seed or input("Enter seed: ")
    
    context = stc_api.initialize(seed)
    
    # Perform some operations to show evolution
    context.hash("test data 1")
    context.hash("test data 2")
    context.derive_key()
    
    print("\n" + context.get_status())


def main():
    """Main CLI entry point"""
    parser = argparse.ArgumentParser(
        description="Seigr Toolset Crypto - Post-classical cryptographic engine",
        formatter_class=argparse.RawDescriptionHelpFormatter,
        epilog="""
Examples:
  # Encrypt data
  stc encrypt -i data.txt -o encrypted.bin -s "my-seed"
  
  # Decrypt data
  stc decrypt -i encrypted.bin -o decrypted.txt -s "my-seed"
  
  # Generate hash
  stc hash -i data.txt -s "my-seed"
  
  # Derive key
  stc derive-key -s "my-seed" -l 32
  
  # Show status
  stc status -s "my-seed"
        """
    )
    
    subparsers = parser.add_subparsers(dest='command', help='Command to execute')
    
    # Encrypt command
    encrypt_parser = subparsers.add_parser('encrypt', help='Encrypt data')
    encrypt_parser.add_argument('-i', '--input', help='Input file (stdin if not provided)')
    encrypt_parser.add_argument('-o', '--output', help='Output file (stdout if not provided)')
    encrypt_parser.add_argument('-s', '--seed', help='Seed for encryption')
    
    # Decrypt command
    decrypt_parser = subparsers.add_parser('decrypt', help='Decrypt data')
    decrypt_parser.add_argument('-i', '--input', required=True, help='Input encrypted file')
    decrypt_parser.add_argument('-o', '--output', help='Output file (stdout if not provided)')
    decrypt_parser.add_argument('-m', '--metadata', help='Metadata file (auto-detected if not provided)')
    decrypt_parser.add_argument('-s', '--seed', help='Seed for decryption')
    
    # Hash command
    hash_parser = subparsers.add_parser('hash', help='Generate probabilistic hash')
    hash_parser.add_argument('-i', '--input', help='Input file (stdin if not provided)')
    hash_parser.add_argument('-s', '--seed', help='Seed for hashing')
    
    # Derive key command
    derive_parser = subparsers.add_parser('derive-key', help='Derive ephemeral key')
    derive_parser.add_argument('-s', '--seed', help='Seed for key derivation')
    derive_parser.add_argument('-l', '--length', type=int, default=32, help='Key length in bytes')
    
    # Status command
    status_parser = subparsers.add_parser('status', help='Show context status')
    status_parser.add_argument('-s', '--seed', help='Seed for context')
    
    args = parser.parse_args()
    
    if not args.command:
        parser.print_help()
        sys.exit(1)
    
    # Execute command
    if args.command == 'encrypt':
        cmd_encrypt(args)
    elif args.command == 'decrypt':
        cmd_decrypt(args)
    elif args.command == 'hash':
        cmd_hash(args)
    elif args.command == 'derive-key':
        cmd_derive_key(args)
    elif args.command == 'status':
        cmd_status(args)


if __name__ == '__main__':
    main()
